package gov.va.cpss.job.loadbill;

import static gov.va.cpss.job.fps.FpsProcessingConstants.RECEIVED_ID_KEY;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;
import static org.junit.Assume.assumeTrue;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.StepExecution;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.model.bal.SiteBalance;
import gov.va.cpss.model.loadbill.LoadBillDetail;
import gov.va.cpss.model.loadbill.LoadBillHeader;
import gov.va.cpss.service.LoadBillService;
import gov.va.cpss.service.VistaAccountService;

/**
 * Test for LoadBillDataProcessor class.
 * 
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context-memory.xml", "/cpss-context.xml" })
public class LoadBillProcessorTest {
	
	private String TEST_TOT_NUM_PATIENT_RECORDS = "1";
	
	private long TEST_CBSS_ACCOUNT_ID = 77L;
	
	private String TEST_ICN_NUM = "4420061118V876665";
	
	private String TEST_STATION_NUM = "442";
	
	private String TEST_PATIENT_ACCOUNT_NUM = "442507010133TESTP";
	
	private long TEST_DFN_NUM = 1112223334445557L;
	
	private long TEST_RECEIVED_ID = 123L;

	private String TEST_AITC_VALUE = "4000+";
	
	private String TEST_TOTAL_BALANCE = "40.0";
	
	private String TEST_OLDEST_BILL_DATE = "05062016";
	
	private Date TEST_OLDEST_BILL_DATE_DATE = new GregorianCalendar(2016, Calendar.MAY, 6).getTime();
	
	private String TEST_PROCESS_DATE = "06072016";
	
	private Date TEST_PROCESS_DATE_DATE = new GregorianCalendar(2016, Calendar.JUNE, 7).getTime();
	
	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private VistaAccountService vistaAccountService;

	@Autowired
	private LoadBillService loadBillService;

	private LoadBillDataProcessor loadBillProcessor;

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		loadBillProcessor = new LoadBillDataProcessor();
		loadBillProcessor.setLoadBillService(loadBillService);
		loadBillProcessor.setVistaAccountService(vistaAccountService);
		setJobExecution();
	}

	@Test
	public void test1() {
		
		// Prerequisite: This test requires the in memory db has a CBSAccount record ID: 77, ICN: 4420061118V876665

		assertNotNull(loadBillService);
		assertNotNull(vistaAccountService);
		assertNotNull(loadBillProcessor);

		// This will return a map entry of header to list of details.
		LoadBillHeader header = buildHeader();
		List<LoadBillDetail> detailL = new ArrayList<>();
		LoadBillDetail detail = buildDetail();
		detailL.add(detail);
		Map.Entry<LoadBillHeader, List<LoadBillDetail>> entry = new AbstractMap.SimpleEntry<>(header, detailL);

		try {
			List<SiteBalance> siteBalanceL = loadBillProcessor.process(entry);

			assertEquals(1, siteBalanceL.size());
			
			for (SiteBalance siteBal : siteBalanceL) {
				System.out.println(siteBal.toString());
				

				// CBSS account number
				assertEquals(TEST_CBSS_ACCOUNT_ID, siteBal.getCbsAccount().getId());
				
				// Facility number
				assertEquals(TEST_STATION_NUM, siteBal.getStationNum());

				// patient account number at facility
				assertEquals(TEST_PATIENT_ACCOUNT_NUM, siteBal.getOldAcctNum());
				
				assertEquals(TEST_DFN_NUM, siteBal.getPatientDfn());
				
				//Refactor To AITC
				// patient account total balance
				assertEquals(TEST_TOTAL_BALANCE, String.valueOf(siteBal.getBalance()));

				// patient acount's oldest bill date
				assertEquals(TEST_OLDEST_BILL_DATE_DATE, siteBal.getOldestBillDate());

				// data process date
				assertEquals(TEST_PROCESS_DATE_DATE, siteBal.getProcessDate());

				// Verify Reference back to the Site Balance Received.
				assertEquals(TEST_RECEIVED_ID, siteBal.getSiteBalRec().getId());
				
			}
			
		} catch (Exception e) {
			System.out.println(e.getMessage());
			fail(e.getMessage());
		}
	}
	
	@Test
	public void test2() {
		
		// Prerequisite: This test requires the in memory db has a CBSAccount record ID: 77, ICN: 4420061118V876665

		assertNotNull(loadBillService);
		assertNotNull(vistaAccountService);
		assertNotNull(loadBillProcessor);

		// This will return a map entry of header to list of details.
		LoadBillHeader header = buildHeader();
		List<LoadBillDetail> detailL = new ArrayList<>();
		LoadBillDetail detail = buildDetail();
		
		detail.setPatientDfn("BADDFN");
		
		detailL.add(detail);
		Map.Entry<LoadBillHeader, List<LoadBillDetail>> entry = new AbstractMap.SimpleEntry<>(header, detailL);

		try {
			List<SiteBalance> siteBalanceL = loadBillProcessor.process(entry);

			assertEquals(1, siteBalanceL.size());
			
			for (SiteBalance siteBal : siteBalanceL) {
				System.out.println(siteBal.toString());
				

				// CBSS account number
				assertEquals(TEST_CBSS_ACCOUNT_ID, siteBal.getCbsAccount().getId());
				
				// Facility number
				assertEquals(TEST_STATION_NUM, siteBal.getStationNum());

				// patient account number at facility
				assertEquals(TEST_PATIENT_ACCOUNT_NUM, siteBal.getOldAcctNum());
				
				assertEquals(TEST_DFN_NUM, siteBal.getPatientDfn());
				
				//Refactor To AITC
				// patient account total balance
				assertEquals(TEST_TOTAL_BALANCE, String.valueOf(siteBal.getBalance()));

				// patient acount's oldest bill date
				assertEquals(TEST_OLDEST_BILL_DATE_DATE, siteBal.getOldestBillDate());

				// data process date
				assertEquals(TEST_PROCESS_DATE_DATE, siteBal.getProcessDate());

				// Verify Reference back to the Site Balance Received.
				assertEquals(TEST_RECEIVED_ID, siteBal.getSiteBalRec().getId());
				
			}
			
		} catch (Exception e) {
			
			assertEquals("For input string: \"BADDFN\"", e.getMessage());
		}
	}

	private LoadBillHeader buildHeader() {
		LoadBillHeader header = new LoadBillHeader();
		header.setStationNum(TEST_STATION_NUM);
		header.setTotNumPatientRecords(TEST_TOT_NUM_PATIENT_RECORDS);
		header.setProcessDate(TEST_PROCESS_DATE);
		return header;
	}

	private LoadBillDetail buildDetail() {
		LoadBillDetail detail = new LoadBillDetail();
		detail.setIcnNumber(TEST_ICN_NUM);
		detail.setPatientAccountNumber(TEST_PATIENT_ACCOUNT_NUM);
		detail.setPatientDfn(String.valueOf(TEST_DFN_NUM));
		detail.setPatientSubtotal(TEST_AITC_VALUE);
		detail.setOldestBillDate(TEST_OLDEST_BILL_DATE);
		return detail;
	}

	/*
	 * Set the test execution with a fake received id parameter.
	 */
	private void setJobExecution() {
		JobExecution j = new JobExecution(1L, new JobParametersBuilder().addLong(RECEIVED_ID_KEY, new Long(TEST_RECEIVED_ID)).toJobParameters());
		StepExecution s = new StepExecution("LoadBillProcessorTest", j);
		loadBillProcessor.beforeStep(s);
	}
	
}
