package gov.va.cpss.job.loadbill;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.service.SftpService;

/**
 * Integration tests to test outcomes of the Load Bill FTP 
 * 
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/cpss-batch.xml",
		"/cpss-batch-load-bill-data.xml", "/load-bill-data-job-test-context.xml", "/cpss-email.xml" })
public class LoadBillDataJobFtpIT {

	public static final String TEST_FILE_DIR_PATH = "loadbill/";

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private TestLoadBillDataJob testLoadBillDataJob;

	@Autowired
	private SftpService sftpService;

	@Autowired
	ProcessStatusDAO processStatusDAO;

	@Autowired
	private String loadBillServerTargetDirectory;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		cleanup();

	}

	@After
	public final void afterTest() {
		if (runIntegrationTest) {
			cleanup();
		}
	}

	/**
	 * Test a simple successful input file.
	 */
	@Test
	public void testSimpleSuccessfulInputFile() {

		final List<String> expectedSuccessFileNames = new ArrayList<String>(1);
		expectedSuccessFileNames.add("payment_update_2.txt");

		final List<String> expectedFailFileNames = null;

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				final boolean runSuccess = testLoadBillDataJob.testJob();

				// Verify the job was successful.
				assertTrue(runSuccess);

				// Read the total files received
				final int totalFiles = testLoadBillDataJob.getFileCount();

				// Verify the total files received
				assertEquals(1, totalFiles);

				// Read the total files processed
				final int totalProcessed = testLoadBillDataJob.getSuccessfulFileCount();

				// Verify the total files processed
				assertEquals(1, totalProcessed);

				// Calculate the Unprocessed Files
				final int totalUnprocessed = totalFiles - totalProcessed;
				assertEquals(0, totalUnprocessed);

				// Check exit status
				assertNull(testLoadBillDataJob.getErrorMessage());
			}

		}).run();

	}

	/**
	 * Test a invalid header total number input file.
	 */
	@Test
	public void testInvalidHeaderTotalNumberInputFile() {

		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add("payment_update_2_invalid_header_total_number.txt");

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				final boolean runSuccess = testLoadBillDataJob.testJob();

				// Verify the job was not successful.
				assertFalse(runSuccess);

				// Read the total files received
				final int totalFiles = testLoadBillDataJob.getFileCount();

				// Verify the total files received
				assertEquals(1, totalFiles);

				// Read the total files processed
				final int totalProcessed = testLoadBillDataJob.getSuccessfulFileCount();

				// Verify the total files processed
				assertEquals(0, totalProcessed);

				// Calculate the Unprocessed Files
				final int totalUnprocessed = totalFiles - totalProcessed;
				assertEquals(1, totalUnprocessed);

				// Check exit status
				final String errorMessage = testLoadBillDataJob.getErrorMessage();
				assertNotNull(errorMessage);

				final String expectedErrorMessage = "payment_update_2_invalid_header_total_number.txt: Invalid header found (total number is not equal to current message number or 0)";
				assertTrue(errorMessage.contains(expectedErrorMessage));
			}

		}).run();

	}

	@Test
	public void testInvalidHeaderMessageNumberInputFile() {

		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add("payment_update_2_invalid_header_message_number.txt");

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				final boolean runSuccess = testLoadBillDataJob.testJob();

				// Verify the job was not successful.
				assertFalse(runSuccess);

				// Read the total files received
				final int totalFiles = testLoadBillDataJob.getFileCount();

				// Verify the total files received
				assertEquals(1, totalFiles);

				// Read the total files processed
				final int totalProcessed = testLoadBillDataJob.getSuccessfulFileCount();

				// Verify the total files processed
				assertEquals(0, totalProcessed);

				// Calculate the Unprocessed Files
				final int totalUnprocessed = totalFiles - totalProcessed;
				assertEquals(1, totalUnprocessed);

				// Check exit status
				final String errorMessage = testLoadBillDataJob.getErrorMessage();
				assertNotNull(errorMessage);

				final String expectedErrorMessage = "payment_update_2_invalid_header_message_number.txt: Invalid header found (message number is incorrect)";
				assertTrue(errorMessage.contains(expectedErrorMessage));
			}

		}).run();

	}

	/**
	 * Test a invalid input file.
	 */
	@Test
	public void testIncompleteInputFile() {

		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add("payment_update_2_incomplete_header_count.txt");

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				final boolean runSuccess = testLoadBillDataJob.testJob(false);

				// Verify the job was not successful.
				assertFalse(runSuccess);

				// Read the total files received
				final int totalFiles = testLoadBillDataJob.getFileCount();

				// Verify the total files received
				assertEquals(1, totalFiles);

				// Read the total files processed
				final int totalProcessed = testLoadBillDataJob.getSuccessfulFileCount();

				// Verify the total files processed
				assertEquals(0, totalProcessed);

				// Calculate the Unprocessed Files
				final int totalUnprocessed = totalFiles - totalProcessed;
				assertEquals(1, totalUnprocessed);

				// Check exit status
				final String errorMessage = testLoadBillDataJob.getErrorMessage();
				assertNotNull(errorMessage);

				final String expectedErrorMessage = "payment_update_2_incomplete_header_count.txt: Incomplete header found";
				assertTrue(errorMessage.contains(expectedErrorMessage));
			}

		}).run();

	}

	private abstract class FileTestWrapper {

		private List<String> expectSuccessFileNames;
		private List<String> expectFailFileNames;

		protected FileTestWrapper(final List<String> expectSuccessFileNames, final List<String> expectFailFileNames) {
			this.expectSuccessFileNames = expectSuccessFileNames;
			this.expectFailFileNames = expectFailFileNames;
		}

		private void checkSetup() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getDataDirectory(),
							true));
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testLoadBillDataJob.getArchiveDirectory(), true));
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getErrorDirectory(),
							true));
				}
			}
		}

		private void stageFiles() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					// This is local directory in test tree.
					assertTrue(ftpFileToServer(TEST_FILE_DIR_PATH + fileName));
				}
			}
		}

		private void checkDirectories() {
			if (expectSuccessFileNames != null) {
				for (String fileName : expectSuccessFileNames) {
					// Verify file is not in the data directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getDataDirectory(),
							true));
					// Verify file is in the archive directory.
					assertTrue(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getArchiveDirectory(),
							true));
					// Verify file is not in the error directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getErrorDirectory(),
							true));
				}
			}
			if (expectFailFileNames != null) {
				for (String fileName : expectFailFileNames) {
					// Verify file is not in the data directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getDataDirectory(),
							true));
					// Verify file is not in the archive directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testLoadBillDataJob.getArchiveDirectory(), true));
					// Verify file is in the error directory.
					assertTrue(sftpService.ftpFileExistsInDirectory(fileName, testLoadBillDataJob.getErrorDirectory(),
							true));
				}
			}
		}

		private void cleanup() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					sftpService.ftpRemoveFileFromDirectory(fileName, testLoadBillDataJob.getDataDirectory());
					sftpService.ftpRemoveFileFromDirectory(fileName, testLoadBillDataJob.getArchiveDirectory());
					sftpService.ftpRemoveFileFromDirectory(fileName, testLoadBillDataJob.getErrorDirectory());
				}
			}
		}

		protected void run() {
			// Ensure test server is in expected initial state.
			checkSetup();

			try {
				// FTP test files to server.
				stageFiles();

				// Run test
				test();

				checkDirectories();

			} finally {

				// Cleanup.
				cleanup();
			}
		}

		protected abstract void test();
	}

	private boolean ftpFileToServer(final String filepath) {
		// Transfer data file.
		final boolean dataFileResult = sftpService.ftpFileToServer(
				new File(this.getClass().getClassLoader().getResource(filepath).getFile()).getAbsolutePath(),
				loadBillServerTargetDirectory);

		// Transfer don file.
		final String emptyFilename = "CCPC-to-CPSS_empty.txt";
		final String emptyFilepath = new File(
				this.getClass().getClassLoader().getResource("fps/" + emptyFilename).getFile()).getAbsolutePath();

		final String dataFilename = new File(this.getClass().getClassLoader().getResource(filepath).getFile())
				.getName();
		final String indicatorFilename = dataFilename.substring(0, dataFilename.lastIndexOf(".txt")) + ".don";

		final boolean indicatorFileResult = sftpService.ftpFileToServerWithName(emptyFilepath, indicatorFilename,
				loadBillServerTargetDirectory);

		return (dataFileResult && indicatorFileResult);
	}

	public void cleanup() {
		final String sql = "DELETE FROM SiteBal WHERE DFN like ?";
		jdbcTemplate.update(sql, "111222333444555%");
		final String sql2 = "DELETE FROM vistaAccount WHERE DFN like ?";
		jdbcTemplate.update(sql2, "111222333444555%");
	}

}
