package gov.va.cpss.job;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

import org.apache.log4j.Logger;

import gov.va.cpss.service.SftpService;
import gov.va.cpss.service.SftpStreamSession;

/**
 * Represents an output file from a batch job on an sFTP server.
 * 
 * @author Brad Pickle
 *
 */
public class JobOutputRemoteFile {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	private SftpService sftpService;

	private String remoteFileName;

	private String remoteTargetDirectory;

	/**
	 * Create a JobOutputRemoteFile with the given name in the given directory
	 * on the sFTP server represented by the given SftpService.
	 * 
	 * @param sftpService
	 *            SftpService providing access to the remote file.
	 * @param remoteFileName
	 *            Name of the remote file.
	 * @param remoteTargetDirectory
	 *            Remote directory where the file resides.
	 */
	public JobOutputRemoteFile(SftpService sftpService, String remoteFileName, String remoteTargetDirectory) {
		this.sftpService = sftpService;
		this.remoteFileName = remoteFileName;
		this.remoteTargetDirectory = remoteTargetDirectory;
	}

	/**
	 * Compare the contents of the remote file with the given String content.
	 * 
	 * @param expectedContent
	 *            String content to compare to.
	 * @return True if the file content equals the expectedContent.
	 * @throws Exception
	 */
	public boolean fileContentsEqual(String expectedContent) throws Exception {
		SftpStreamSession session = null;
		boolean error = false;
		try (InputStream expectedContentIS = new ByteArrayInputStream(expectedContent.getBytes(StandardCharsets.UTF_8))) {
			session = sftpService.openFileStream(remoteFileName, remoteTargetDirectory);
			final boolean contentsEqual = inputStreamContentsEqual(expectedContentIS, session.getInputStream());
			if (! contentsEqual) {
				logger.error("Remote file contents do not match expected content.");
			}
			
			return contentsEqual;
		} catch (Exception e) {
			error = true;
			throw e;
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
					logger.error("Unexpected exception: " + e.getMessage());

					if (!error)
						throw e;
				}
			}
		}

	}

	/**
	 * Compare contents the two InputStreams.
	 * 
	 * @param is1
	 * @param is2
	 * @return True if the contents of the two InputStreams are equal.
	 * @throws IOException
	 */
	private boolean inputStreamContentsEqual(InputStream is1, InputStream is2) throws IOException {
		byte[] isBuffer1 = new byte[1024];
		byte[] isBuffer2 = new byte[1024];

		int numRead1 = 0;
		int numRead2 = 0;

		while (true) {
			numRead1 = is1.read(isBuffer1);
			if (numRead1 > -1) {
				logger.debug("Expected:" + new String(isBuffer1));
			}
			numRead2 = is2.read(isBuffer2);
			if (numRead2 > -1) {
				logger.debug("Actual  :" + new String(isBuffer2));
			}
			if (numRead1 > -1) {
				if (numRead1 != numRead2) {
					return false;
				}
				if (!Arrays.equals(isBuffer1, isBuffer2)) {
					return false;
				}
			} else {
				return numRead2 < 0;
			}
		}
	}
}
