package gov.va.cpss.dao.impl;

import static org.junit.Assert.*;
import static org.junit.Assume.assumeTrue;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.VistaAccountDAO;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.icn.VistaAccount;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml"})
/**
 * Unit test for VistaAccountDAOImpl.
 * 
 * @author Andrew Vance
 *
 */
public final class VistaAccountDAOImplTest {
	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;
	
	@Autowired
	private ProcessStatusDAO processStatusDao;
	
	@Autowired
	private VistaAccountDAO vistaAccountDao;
	
	private List<VistaAccount> vistaAccountL;
	
	private JdbcTemplate jdbcTemplate;
	
	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}
	
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		
		insertTestData();
	}
	
	@After
	public final void afterTest() {
		if(runIntegrationTest) {
			cleanupTestData();
		}
	}
	
	@Test
	public final void testGet() {
		final VistaAccount vistaAccountExpected = vistaAccountL.get(0);
		
		final VistaAccount vistaAccountRetrieved = vistaAccountDao.getVistaAccount(vistaAccountExpected.getDfn(), vistaAccountExpected.getStationNum());
		
		assertVistaAccountEquals(vistaAccountExpected, vistaAccountRetrieved);
	}
	
	@Test
	public final void testExists() {
		final VistaAccount vistaAccountExpected = vistaAccountL.get(0);
		
		assertTrue(vistaAccountDao.exists(vistaAccountExpected.getDfn(), vistaAccountExpected.getStationNum()));
		
		assertFalse(vistaAccountDao.exists(-1, "doesNotExist"));
	}
	
	@Test
	public final void testBatchSelect() {
		final String[] icns = new String[vistaAccountL.size()];
		for(int idx=0; idx<icns.length; idx++){
			icns[idx] = vistaAccountL.get(idx).getIcn();
		}
		
		final List<VistaAccount> vistaAccountRetrievedL = vistaAccountDao.batchSelect(icns);		
		assertNotNull(vistaAccountRetrievedL);
		
		final List<Long> vistaAccountRetrievedDfnL = vistaAccountRetrievedL.stream().map(VistaAccount::getDfn).collect(Collectors.toList());
		assertTrue(vistaAccountRetrievedDfnL.containsAll(vistaAccountL.stream().map(VistaAccount::getDfn).collect(Collectors.toList())));
	}
	
	@Test
	public final void testUpdate() {
		// Test update single account
		final VistaAccount vistaAccountOriginal = vistaAccountL.get(0);
		
		final VistaAccount vistaAccountOriginalRetrieved = vistaAccountDao.getVistaAccount(vistaAccountOriginal.getDfn(), vistaAccountOriginal.getStationNum());
		assertVistaAccountEquals(vistaAccountOriginal, vistaAccountOriginalRetrieved);
		
		 vistaAccountL.get(0).setCbssAcntId(4444);
		 vistaAccountL.get(0).setIcn("changed");
		
		final List<VistaAccount> accountsToUpdate = new ArrayList<VistaAccount>();
		accountsToUpdate.add( vistaAccountL.get(0));
		
		final int[] updatedSingle = vistaAccountDao.update(accountsToUpdate);
		assertEquals(1, updatedSingle.length);
		
		final VistaAccount vistaAccountUpdatedRetrieved = vistaAccountDao.getVistaAccount( vistaAccountL.get(0).getDfn(),  vistaAccountL.get(0).getStationNum());
		assertVistaAccountEquals( vistaAccountL.get(0), vistaAccountUpdatedRetrieved);
		
		// Test batch update
		final int[] updatedBatch = vistaAccountDao.update(vistaAccountL);
		assertEquals(vistaAccountL.size(), updatedBatch.length);
	}
	
	private final void insertTestData() {
		vistaAccountL = new ArrayList<VistaAccount>();
		
		final VistaAccount vistaAccount1 = new VistaAccount();
		vistaAccount1.setDfn(7148281);
		vistaAccount1.setStationNum("123");
		vistaAccount1.setIcn("a1b2c3d4");
		vistaAccount1.setCbssAcntId(1111);
		vistaAccount1.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		vistaAccountL.add(vistaAccount1);

		final VistaAccount vistaAccount2 = new VistaAccount();
		vistaAccount2.setDfn(724242);
		vistaAccount2.setStationNum("123");
		vistaAccount2.setIcn("a0b0c0d0e0");
		vistaAccount2.setCbssAcntId(2221);
		vistaAccount2.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		
		vistaAccountL.add(vistaAccount2);
		
		cleanupTestData();		
		
		assertTrue(saveVistaAccount(vistaAccount1));
		assertTrue(saveVistaAccount(vistaAccount2));
	}
	
	private final void cleanupTestData() {
		final String sql = "DELETE FROM VistaAccount WHERE DFN=? AND StationNum=?";
		
		vistaAccountL.forEach(acnt -> {
			jdbcTemplate.update(new PreparedStatementCreator() {
				@Override
				public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
					final PreparedStatement ps = con.prepareStatement(sql);
					ps.setLong(1, acnt.getDfn());
					ps.setString(2, acnt.getStationNum());
					return ps;
				}							
			});
		});
	}
	
	private final boolean saveVistaAccount(final VistaAccount vistaAccount) {
		return vistaAccountDao.save(vistaAccount.getStatusId(),
				vistaAccount.getDfn(), 
				vistaAccount.getStationNum(), 
				vistaAccount.getIcn(), 
				vistaAccount.getCbssAcntId());
	}
	
	private static final void assertVistaAccountEquals(final VistaAccount expected, final VistaAccount retrieved) {
		assertEquals(expected.getCbssAcntId(), retrieved.getCbssAcntId());
		assertEquals(expected.getDfn(), retrieved.getDfn());
		assertEquals(expected.getIcn(), retrieved.getIcn());
		assertEquals(expected.getStationNum(), retrieved.getStationNum());
		assertEquals(expected.getStatusId(), retrieved.getStatusId());
	}
}
