package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.time.Instant;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.fps.PSReceived;
import gov.va.cpss.model.fps.PSSite;

/**
 *  Unit test for PSSiteDAOImpl class.
 *
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public final class PSSiteDAOImplTest extends AbstractCbsDaoImplTest {

	@Test
	public final void testGetByIdAndLastIndex() {
		final PSSite psSiteExpected = testPsSiteL.get(0);
		
		final PSSite psSiteRetrieved = psSiteDao.getById(psSiteExpected.getId());
		
		assertNotNull(psSiteRetrieved);
		assertPsSiteEquals(psSiteExpected, psSiteRetrieved);
		// getLastIndex only behaves this way because this test guarantees the last inserted PSSite was psExpected
		assertEquals(psSiteExpected.getId(), (long)psSiteDao.getLastIndex());
	}
	
	@Test
	public final void testGetAllForPSReceivedStatusId() {
		final PSSite psSiteExpected = testPsSiteL.get(0);
		
		final List<PSSite> psSiteRetrievedL = psSiteDao.getAllForPSReceivedStatusId(testPsReceivedL.get(0).getStatusId());
		assertNotNull(psSiteRetrievedL);
		assertTrue(psSiteRetrievedL.stream().map(PSSite::getId).collect(Collectors.toList()).contains(psSiteExpected.getId()));
	}
	
	@Test
	public final void testGetAllByPSReceivedID() {
		final PSSite psSiteExpected = testPsSiteL.get(0);
		
		final List<PSSite> psSiteRetrievedL = psSiteDao.getAllByPSReceivedID(testPsReceivedL.get(0).getId());
		assertNotNull(psSiteRetrievedL);
		assertTrue(psSiteRetrievedL.stream().map(PSSite::getId).collect(Collectors.toList()).contains(psSiteExpected.getId()));
	}
	
	@Test
	public final void testGetAllIDByPSReceivedID() {
		final PSSite psSiteExpected = testPsSiteL.get(0);
		
		final List<Long> psSiteRetrievedIdL = psSiteDao.getAllIDByPSReceivedID(testPsReceivedL.get(0).getId());
		assertNotNull(psSiteRetrievedIdL);
		assertTrue(psSiteRetrievedIdL.contains(psSiteExpected.getId()));
	}	
	
	@Test
	public final void deleteByReceievedId() {
		final BatchRun batchRun = testBatchRunL.get(0);
		
		final PSReceived psReceived1 = new PSReceived(batchRun.getId(), batchRun.getStatusId(), batchRun.getStartDate(), "psReceived.filename");
		psReceived1.setNumOfSite(2);
		psReceived1.setNumOfStatement(1);
		psReceived1.setId(psReceivedDao.save(psReceived1));
		
		final PSSite psSiteExpected = new PSSite();
		psSiteExpected.setPsReceivedId(psReceived1.getId());
		psSiteExpected.setSeqNum(1);
		psSiteExpected.setTotSeqNum(1);
		psSiteExpected.setFacilityNum("123");
		psSiteExpected.setFacilityPhoneNum("555-5555");
		psSiteExpected.setTotStatement(1);
		psSiteExpected.setStatementVal(new Money(12345, 10));
		psSiteExpected.setProcessDate(Date.from(Instant.now()));
		psSiteExpected.setStatementDate(Date.from(Instant.now()));
		psSiteExpected.setId(psSiteDao.save(psSiteExpected));
		testPsSiteL.add(psSiteExpected);

		assertNotNull(psSiteDao.getById(psSiteExpected.getId()));		
		psSiteDao.deleteByReceivedId(psReceived1.getId());		
		assertNull(psSiteDao.getById(psSiteExpected.getId()));
	}
	
	@Override
	protected final void insertTestData() {
		// Create batch run
		final BatchRun batchRun = cbsDaoImplUtils.createTestBatchRun();
		batchRun.setId(batchRunDao.save(batchRun));		
		testBatchRunL.add(batchRun);
		
		// Create PSReceived
		final PSReceived psReceived1 = cbsDaoImplUtils.createTestPSReceived(batchRun);
		psReceived1.setId(psReceivedDao.save(psReceived1));
		testPsReceivedL.add(psReceived1);
		
		// Create PSSite;
		final PSSite psSite1 = cbsDaoImplUtils.createTestPSSite(psReceived1);
		psSite1.setId(psSiteDao.save(psSite1));
		testPsSiteL.add(psSite1);
	}
	
	private static final void assertPsSiteEquals(final PSSite expected, final PSSite retrieved) {
		assertEquals(expected.getId(), retrieved.getId());
		assertEquals(expected.getPsReceivedId(), retrieved.getPsReceivedId());
		assertEquals(expected.getSeqNum(), retrieved.getSeqNum());
		assertEquals(expected.getTotSeqNum(), retrieved.getTotSeqNum());
		assertEquals(expected.getFacilityNum(), retrieved.getFacilityNum());
		assertEquals(expected.getFacilityPhoneNum(), retrieved.getFacilityPhoneNum());
		assertEquals(expected.getTotStatement(), retrieved.getTotStatement());
		assertEquals(expected.getStatementVal(), retrieved.getStatementVal());
		assertEquals(expected.getStatementDate().getTime(), retrieved.getStatementDate().getTime(), 1000);
	}
}
