package gov.va.cpss.dao.impl;

import static org.junit.Assert.*;
import static org.junit.Assume.assumeTrue;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.BatchJobDAO;
import gov.va.cpss.dao.BatchRunDAO;
import gov.va.cpss.dao.PSReceivedDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.fps.PSReceived;

/**
 * Unit test for PSReceivedDAOImpl class.
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public final class PSReceivedDAOImplTest {
	private boolean runIntegrationTest = true;
	
	@Autowired
	private BatchRunDAO batchRunDao;
	
	@Autowired
	private BatchJobDAO batchJobDao;
	
	@Autowired
	private PSReceivedDAO psReceivedDao;
	
	@Autowired
	private ProcessStatusDAO processStatusDao;
	
	private List<BatchRun> testBatchRunL;
	
	private List<PSReceived> testPsReceivedL;
	
	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}
	
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		
		insertTestData();
	}
	
	@After
	public final void afterTest() {
		if(runIntegrationTest) {
			cleanupTestData();
		}
	}
	
	@Test
	public final void testUpdateStatus() {
		final PSReceived psReceivedExpected = testPsReceivedL.get(0);
		psReceivedExpected.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.REPLACED));
		
		psReceivedDao.updateStatus(psReceivedExpected);
		
		final PSReceived psReceivedRetrieved = getPsReceived(psReceivedExpected);
		assertNotNull(psReceivedRetrieved);
		assertEquals(psReceivedExpected.getId(), psReceivedRetrieved.getId());
		assertEquals(psReceivedExpected.getStatusId(), psReceivedRetrieved.getStatusId());
	}
	
	@Test
	public final void testUpdateResults() {
		final PSReceived psReceivedExpected = testPsReceivedL.get(0);
		psReceivedExpected.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.SUCCESS));
		psReceivedExpected.setNumOfSite(17);
		psReceivedExpected.setNumOfStatement(38);
		
		psReceivedDao.updateResults(psReceivedExpected);
		
		final PSReceived psReceivedRetrieved = getPsReceived(psReceivedExpected);
		assertNotNull(psReceivedRetrieved);
		assertEquals(psReceivedExpected.getId(), psReceivedRetrieved.getId());
		assertEquals(psReceivedExpected.getStatusId(), psReceivedRetrieved.getStatusId());
		assertEquals(psReceivedExpected.getNumOfSite(), psReceivedRetrieved.getNumOfSite());
		assertEquals(psReceivedExpected.getNumOfStatement(), psReceivedRetrieved.getNumOfStatement());
	}
	
	@Test
	public final void testGetPSReceivedIDListByStatusID() {
		final PSReceived psReceivedExpected = testPsReceivedL.get(0);
		
		final List<Long> psReceivedIdL = psReceivedDao.getPSReceivedIDListByStatusID(psReceivedExpected.getStatusId());
		assertNotNull(psReceivedIdL);
		assertTrue(psReceivedIdL.contains(psReceivedExpected.getId()));
	}
	
	@Test
	public final void testUpdateStatusById() {
		final PSReceived psReceivedExpected = testPsReceivedL.get(0);
		psReceivedExpected.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.REPLACED));
		
		psReceivedDao.updateStatusById(psReceivedExpected.getId(), psReceivedExpected.getStatusId());
		
		final PSReceived psReceivedRetrieved = getPsReceived(psReceivedExpected);
		assertNotNull(psReceivedRetrieved);
		assertEquals(psReceivedExpected.getId(), psReceivedRetrieved.getId());
		assertEquals(psReceivedExpected.getStatusId(), psReceivedRetrieved.getStatusId());
	}
	
	private final void insertTestData() {
		// Create batch run
		final BatchRun batchRun = new BatchRun();
		batchRun.setJobId(batchJobDao.batchList().get(0).getId());
		batchRun.setStartDate(Timestamp.from(Instant.now()));
		batchRun.setEndDate(Timestamp.from(Instant.now()));
		batchRun.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		batchRun.setMessage("testmessage");
		batchRun.setId(batchRunDao.save(batchRun));		
		testBatchRunL = new ArrayList<BatchRun>();
		testBatchRunL.add(batchRun);
		
		// Create PSReceived
		final PSReceived psReceived = new PSReceived(batchRun.getId(), batchRun.getStatusId(), batchRun.getStartDate(), "psReceived.filename");
		psReceived.setNumOfSite(2);
		psReceived.setNumOfStatement(1);
		psReceived.setId(psReceivedDao.save(psReceived));
		testPsReceivedL = new ArrayList<PSReceived>();
		testPsReceivedL.add(psReceived);
	}
	
	private final void cleanupTestData() {
		testPsReceivedL.forEach(psReceived -> {
			final String sql = "DELETE FROM PSReceived WHERE ID = ?";
			
			jdbcTemplate.update(sql, psReceived.getId());
		});
		
		testBatchRunL.forEach(batchRun -> {
			final String sql = "DELETE FROM BatchRun WHERE ID = ?";
			
			jdbcTemplate.update(sql, batchRun.getId());
		});
	}
	
	private final PSReceived getPsReceived(final PSReceived inPsReceived) {
		final String sql = "SELECT * FROM PSReceived WHERE ID = " + inPsReceived.getId();
		
		final List<PSReceived> psReceivedL = jdbcTemplate.query(sql, new RowMapper<PSReceived>() {

			@Override
			public PSReceived mapRow(ResultSet rs, int rowNum) throws SQLException {
				final PSReceived psReceived = new PSReceived(rs.getInt("batchRun_id"), 
						rs.getInt("status_id"), 
						rs.getTimestamp("dateReceived"),
						rs.getString("filename"));
				psReceived.setId(rs.getLong("id"));
				psReceived.setNumOfSite(rs.getLong("numOfSite"));
				psReceived.setNumOfStatement(rs.getLong("numOfStatement"));
				return psReceived;
			}
			
		});
		
		return psReceivedL.size() > 0 ? psReceivedL.get(0) : null;		
	}
}
