package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.PrintAckRecDAO;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PrintAckFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * Unit tests the CBSStmtDAOImpl class.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class CBSStmtDAOImplTest {

	private final Logger testLogger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSStmtDAO cbsStmtDao;

	@Autowired
	private PrintAckRecDAO printAckRecDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the save() and getAllByCBSStmtID() methods of CBSStmtDAOImpl
	 */
	@Test
	public final void testSaveGet() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final int cbsStmtKey = 1;
				final long stationNumLong = 12345L;

				final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(cbsStmtKey,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNumLong);
				final CBSStmt originalCBSStmt = cbsDAOImplUtils.copyCBSStmt(cbsStmt);

				final long id = cbsStmtDao.save(cbsStmt);
				getCbsStmtIdS().add(cbsStmt.getId());
				getAccountIdS().add(cbsStmt.getAccountId());

				assertEquals(id, cbsStmt.getId());
				assertTrue(cbsStmt.getId() > 0);
				// assertNotNull(cbsStmt.getCreatedBy());
				assertNotNull(cbsStmt.getCreatedDate());
				// assertNotNull(cbsStmt.getModifiedBy());
				assertNotNull(cbsStmt.getModifiedDate());

				// Fields not set before save
				originalCBSStmt.setId(cbsStmt.getId());
				originalCBSStmt.setCreatedBy(cbsStmt.getCreatedBy());
				originalCBSStmt.setCreatedDate(cbsStmt.getCreatedDate());
				originalCBSStmt.setModifiedBy(cbsStmt.getModifiedBy());
				originalCBSStmt.setModifiedDate(cbsStmt.getModifiedDate());

				// Assert relevant fields equal with original
				cbsDAOImplUtils.compareCBSStmt(cbsStmt, originalCBSStmt);

				CBSStmt newCBSStmt = cbsStmtDao.get(id);

				assertNotNull(newCBSStmt);

				cbsDAOImplUtils.compareCBSStmt(cbsStmt, newCBSStmt);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the getExistingStatementByAccountAndMonth() method of
	 * CBSStmtDAOImpl
	 */
	@Test
	public final void testGetExistingStatementByAccountAndMonth() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final long stationNumLong = 12345L;

				Date statementDate1 = null;
				Date statementDate2 = null;
				Date statementDate3 = null;
				try {
					statementDate1 = (new SimpleDateFormat("MMddyyyy")).parse("05202016");
					statementDate2 = (new SimpleDateFormat("MMddyyyy")).parse("05252016");
					statementDate3 = (new SimpleDateFormat("MMddyyyy")).parse("04252016");
				} catch (ParseException e) {
				}

				final String icn1 = CBSDAOImplUtils.getTestIcn(1);
				final String icn2 = CBSDAOImplUtils.getTestIcn(2);

				// Create statements for icn1, one for each of the three dates
				final CBSStmt cbsStmtIcn1Date1 = cbsDAOImplUtils.createSimpleTestCBSStmt(1,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNumLong, icn1, statementDate1);
				final CBSStmt originalCBSStmtIcn1Date1 = cbsDAOImplUtils.copyCBSStmt(cbsStmtIcn1Date1);

				getCbsStmtIdS().add(cbsStmtDao.save(cbsStmtIcn1Date1));
				getAccountIdS().add(cbsStmtIcn1Date1.getAccountId());

				final CBSStmt cbsStmtIcn1Date2 = cbsDAOImplUtils.createSimpleTestCBSStmt(1,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNumLong, icn1, statementDate2);
				final CBSStmt originalCBSStmtIcn1Date2 = cbsDAOImplUtils.copyCBSStmt(cbsStmtIcn1Date2);

				getCbsStmtIdS().add(cbsStmtDao.save(cbsStmtIcn1Date2));

				final CBSStmt cbsStmtIcn1Date3 = cbsDAOImplUtils.createSimpleTestCBSStmt(1,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNumLong, icn1, statementDate3);
				final CBSStmt originalCBSStmtIcn1Date3 = cbsDAOImplUtils.copyCBSStmt(cbsStmtIcn1Date3);

				getCbsStmtIdS().add(cbsStmtDao.save(cbsStmtIcn1Date3));

				// Create statement for icn2 for only one date
				final CBSStmt cbsStmtIcn2Date1 = cbsDAOImplUtils.createSimpleTestCBSStmt(2,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNumLong, icn2, statementDate1);
				final CBSStmt originalCBSStmtIcn2Date1 = cbsDAOImplUtils.copyCBSStmt(cbsStmtIcn2Date1);

				getCbsStmtIdS().add(cbsStmtDao.save(cbsStmtIcn2Date1));
				getAccountIdS().add(cbsStmtIcn2Date1.getAccountId());

				// Fields not set before save
				originalCBSStmtIcn1Date1.setId(cbsStmtIcn1Date1.getId());
				originalCBSStmtIcn1Date1.setCreatedBy(cbsStmtIcn1Date1.getCreatedBy());
				originalCBSStmtIcn1Date1.setCreatedDate(cbsStmtIcn1Date1.getCreatedDate());
				originalCBSStmtIcn1Date1.setModifiedBy(cbsStmtIcn1Date1.getModifiedBy());
				originalCBSStmtIcn1Date1.setModifiedDate(cbsStmtIcn1Date1.getModifiedDate());

				originalCBSStmtIcn1Date2.setId(cbsStmtIcn1Date2.getId());
				originalCBSStmtIcn1Date2.setCreatedBy(cbsStmtIcn1Date2.getCreatedBy());
				originalCBSStmtIcn1Date2.setCreatedDate(cbsStmtIcn1Date2.getCreatedDate());
				originalCBSStmtIcn1Date2.setModifiedBy(cbsStmtIcn1Date2.getModifiedBy());
				originalCBSStmtIcn1Date2.setModifiedDate(cbsStmtIcn1Date2.getModifiedDate());

				originalCBSStmtIcn1Date3.setId(cbsStmtIcn1Date3.getId());
				originalCBSStmtIcn1Date3.setCreatedBy(cbsStmtIcn1Date3.getCreatedBy());
				originalCBSStmtIcn1Date3.setCreatedDate(cbsStmtIcn1Date3.getCreatedDate());
				originalCBSStmtIcn1Date3.setModifiedBy(cbsStmtIcn1Date3.getModifiedBy());
				originalCBSStmtIcn1Date3.setModifiedDate(cbsStmtIcn1Date3.getModifiedDate());

				originalCBSStmtIcn2Date1.setId(cbsStmtIcn2Date1.getId());
				originalCBSStmtIcn2Date1.setCreatedBy(cbsStmtIcn2Date1.getCreatedBy());
				originalCBSStmtIcn2Date1.setCreatedDate(cbsStmtIcn2Date1.getCreatedDate());
				originalCBSStmtIcn2Date1.setModifiedBy(cbsStmtIcn2Date1.getModifiedBy());
				originalCBSStmtIcn2Date1.setModifiedDate(cbsStmtIcn2Date1.getModifiedDate());

				// Test getting icn1 for month with two statements, expect to
				// return
				// the last created
				final CBSStmt cbsStmtTest1 = cbsStmtDao
						.getExistingStatementByAccountAndMonth(cbsStmtIcn1Date1.getAccountId(), statementDate1);
				assertNotNull(cbsStmtTest1);
				// Expect second date since same month, but created second
				cbsDAOImplUtils.compareCBSStmt(cbsStmtTest1, cbsStmtIcn1Date2);

				// Test getting icn1 for month with one statements
				final CBSStmt cbsStmtTest2 = cbsStmtDao
						.getExistingStatementByAccountAndMonth(cbsStmtIcn1Date1.getAccountId(), statementDate3);
				assertNotNull(cbsStmtTest2);
				cbsDAOImplUtils.compareCBSStmt(cbsStmtTest2, cbsStmtIcn1Date3);

				// Test getting icn2
				final CBSStmt cbsStmtTest3 = cbsStmtDao
						.getExistingStatementByAccountAndMonth(cbsStmtIcn2Date1.getAccountId(), statementDate1);
				assertNotNull(cbsStmtTest3);
				cbsDAOImplUtils.compareCBSStmt(cbsStmtTest3, cbsStmtIcn2Date1);

				// Test getting existing icn but no statements in month
				final CBSStmt cbsStmtTest4 = cbsStmtDao
						.getExistingStatementByAccountAndMonth(cbsStmtIcn2Date1.getAccountId(), statementDate3);
				assertNull(cbsStmtTest4);

				// Test getting icn that doesn't exist
				// Remove one that we just used, so we know it won't exist
				jdbcTemplate.update("DELETE FROM CBSStmt WHERE id=" + cbsStmtIcn2Date1.getId());
				getCbsStmtIdS().remove(cbsStmtIcn2Date1.getId());
				jdbcTemplate.update("DELETE FROM CBSAccount WHERE id=" + cbsStmtIcn2Date1.getAccountId());
				getAccountIdS().remove(cbsStmtIcn2Date1.getAccountId());

				final CBSStmt cbsStmtTest5 = cbsStmtDao
						.getExistingStatementByAccountAndMonth(cbsStmtIcn2Date1.getAccountId(), statementDate1);
				assertNull(cbsStmtTest5);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the getMessageStatementsForSite() method of CBSStmtDAOImpl
	 */
	@Test
	public final void testGetMessageStatementsForSite() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final long stationNum1 = 1L;
				final long stationNum2 = 2L;
				final long stationNum3 = 3L;

				final int sentBatchRunId = 1;
				final String sentFileName = "cbsTestFileNameSent.txt";

				final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();

				// Create an already sent statement for station 1
				final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(1, CBSDAOImplUtils.STATUS_SENT_ID,
						stationNum1);
				cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
				getCbsStmtIdS().add(sentCBSStmt1.getId());
				getAccountIdS().add(sentCBSStmt1.getAccountId());

				// Create the CBSMessage record for sent statement
				final long sentMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL,
						CBSDAOImplUtils.STATUS_SUCCESS_ID, sentBatchRunId, sentFileName);
				getCbsMessageIdS().add(sentMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(sentCBSStmt1.getId(), sentMessageId);
				sentCBSStmt1.setMessageId(sentMessageId);

				final List<CBSStmt> simpleResultsL = cbsStmtDao.getMessageStatementsForSite(sentMessageId,
						"" + stationNum1);
				assertNotNull(simpleResultsL);
				assertEquals(simpleResultsL.size(), 1);
				assertNotNull(simpleResultsL.get(0));
				cbsDAOImplUtils.compareCBSStmt(sentCBSStmt1, simpleResultsL.get(0));

				// Create a NEW statement for station 1
				final CBSStmt newCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(2, CBSDAOImplUtils.STATUS_NEW_ID,
						stationNum1);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt1);
				getCbsStmtIdS().add(newCBSStmt1.getId());
				getAccountIdS().add(newCBSStmt1.getAccountId());

				// Create a NEW statement for station 1 and station2. station 1
				// is
				// primary
				final CBSStmt newCBSStmt1and2 = cbsDAOImplUtils.createDoubleTestCBSStmt(3,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNum1, stationNum2);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt1and2);
				getCbsStmtIdS().add(newCBSStmt1and2.getId());
				getAccountIdS().add(newCBSStmt1and2.getAccountId());

				// Create a NEW statement for station 2 and station3. station 2
				// is
				// primary
				final CBSStmt newCBSStmt2and3 = cbsDAOImplUtils.createDoubleTestCBSStmt(4,
						CBSDAOImplUtils.STATUS_NEW_ID, stationNum2, stationNum3);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt2and3);
				getCbsStmtIdS().add(newCBSStmt2and3.getId());
				getAccountIdS().add(newCBSStmt2and3.getAccountId());

				// Create the CBSMessage record for NEW statement
				final int newBatchRunId = 2;
				final String newFileName = "cbsTestFileNameNew.txt";
				final long newMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL,
						CBSDAOImplUtils.STATUS_INITIAL_ID, newBatchRunId, newFileName);
				getCbsMessageIdS().add(newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1.getId(), newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1and2.getId(), newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt2and3.getId(), newMessageId);
				newCBSStmt1.setMessageId(newMessageId);
				newCBSStmt1and2.setMessageId(newMessageId);
				newCBSStmt2and3.setMessageId(newMessageId);

				final List<CBSStmt> doubleStat1ResultsL = cbsStmtDao.getMessageStatementsForSite(newMessageId,
						"" + stationNum1);
				assertNotNull(doubleStat1ResultsL);
				assertEquals(doubleStat1ResultsL.size(), 2);
				final List<CBSStmt> newStmts = new ArrayList<CBSStmt>();
				newStmts.add(newCBSStmt1);
				newStmts.add(newCBSStmt1and2);
				for (CBSStmt cbsStmt : doubleStat1ResultsL) {
					assertNotNull(cbsStmt);
					CBSStmt expectedCbsStmt = CBSDAOImplUtils.getStmtWithId(newStmts, cbsStmt.getId());
					assertNotNull(expectedCbsStmt);
					cbsDAOImplUtils.compareCBSStmt(expectedCbsStmt, cbsStmt);
				}

				final List<CBSStmt> doubleStat2ResultsL = cbsStmtDao.getMessageStatementsForSite(newMessageId,
						"" + stationNum2);
				assertNotNull(doubleStat2ResultsL);
				assertEquals(doubleStat2ResultsL.size(), 1);
				cbsDAOImplUtils.compareCBSStmt(newCBSStmt2and3, doubleStat2ResultsL.get(0));

				final List<CBSStmt> doubleStat3ResultsL = cbsStmtDao.getMessageStatementsForSite(newMessageId,
						"" + stationNum3);
				assertNotNull(doubleStat3ResultsL);
				assertEquals(doubleStat3ResultsL.size(), 0);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the getStatementsForPrintAck(), updatePrintAckFields() and
	 * updateStatusFromToForPrintAckId() and undoPrintAcknowledgement() methods
	 * of CBSStmtDAOImpl
	 */
	@Test
	public final void testPrintAckMethods() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {

				final int ackPatient1Key = 1;
				final int ackPatient2Key = 2;
				final int nonAckPatientKey = 3;

				final String ackPatient1ICN = CBSDAOImplUtils.getTestIcn(ackPatient1Key);
				final String ackPatient2ICN = CBSDAOImplUtils.getTestIcn(ackPatient2Key);
				final String nonAckPatientICN = CBSDAOImplUtils.getTestIcn(nonAckPatientKey);

				final long ackStationNum = 1L;
				final long nonAckStationNum = 2L;

				Date ackStatementDate = null;
				Date nonAckStatementDate = null;
				Date ackReceivedDate = null;
				Date ackPrintDate = null;
				try {
					ackStatementDate = (new SimpleDateFormat("MMddyyyy")).parse("06012016");
					nonAckStatementDate = (new SimpleDateFormat("MMddyyyy")).parse("05252016");
					ackReceivedDate = (new SimpleDateFormat("MMddyyyy")).parse("05312016");
					ackPrintDate = (new SimpleDateFormat("MMddyyyy")).parse("05302016");
				} catch (ParseException e) {
				}

				// Create a statement with a different statement date
				final CBSStmt nonAckStatementDateStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_SENT_ID, ackStationNum, ackPatient1ICN, nonAckStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckStatementDateStmt);
				getCbsStmtIdS().add(nonAckStatementDateStmt.getId());
				testLogger.debug("nonAckStatementDateStmt.getId():" + nonAckStatementDateStmt.getId());
				getAccountIdS().add(nonAckStatementDateStmt.getAccountId());

				// Create a statement with a different station
				final CBSStmt nonAckStationStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_SENT_ID, nonAckStationNum, ackPatient1ICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckStationStmt);
				getCbsStmtIdS().add(nonAckStationStmt.getId());
				testLogger.debug("nonAckStationStmt.getId():" + nonAckStationStmt.getId());
				getAccountIdS().add(nonAckStationStmt.getAccountId());

				// Create a statement in NEW status
				final CBSStmt nonAckStatusStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_NEW_ID, ackStationNum, ackPatient1ICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckStatusStmt);
				getCbsStmtIdS().add(nonAckStatusStmt.getId());
				testLogger.debug("nonAckStatusStmt.getId():" + nonAckStatusStmt.getId());
				getAccountIdS().add(nonAckStatusStmt.getAccountId());

				// Create a statement with a different patient
				final CBSStmt nonAckPatientStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(nonAckPatientKey,
						CBSDAOImplUtils.STATUS_SENT_ID, ackStationNum, nonAckPatientICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckPatientStmt);
				getCbsStmtIdS().add(nonAckPatientStmt.getId());
				testLogger.debug("nonAckPatientStmt.getId():" + nonAckPatientStmt.getId());
				getAccountIdS().add(nonAckPatientStmt.getAccountId());

				// Create a statement with a different primary site
				// Removed, no longer considering primary site
				// final CBSStmt nonAckPrimarySiteStmt =
				// cbsDAOImplUtils.createDoubleTestCBSStmt(ackPatient1Key,
				// CBSDAOImplUtils.STATUS_SENT_ID, nonAckStationNum,
				// ackStationNum);
				// nonAckPrimarySiteStmt.setStatementDate(ackStatementDate);
				// cbsDAOImplUtils.saveCBSStmt(nonAckPrimarySiteStmt);
				// getCbsStmtIdS().add(nonAckPrimarySiteStmt.getId());
				// testLogger.debug("nonAckPrimarySiteStmt.getId():" +
				// nonAckPrimarySiteStmt.getId());
				// getAccountIdS().add(nonAckPrimarySiteStmt.getAccountId());

				// Stmt for matching patient1 SENT status. Don't create in
				// database yet.
				final CBSStmt ackPatient1Stmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_SENT_ID, ackStationNum, ackPatient1ICN, ackStatementDate);

				// Stmt for matching patient2 ACK status. Don't create in
				// database yet.
				final CBSStmt ackPatient2Stmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient2Key,
						CBSDAOImplUtils.STATUS_ACK_ID, ackStationNum, ackPatient2ICN, ackStatementDate);

				// Set up the PrintAckRec
				final int batchRunId = 1;
				final String printAckFileName = "printAck.txt";

				final List<ADFileRecord> printAckDetailL = new ArrayList<ADFileRecord>(2);

				final ADFileRecord patient1AD = new ADFileRecord();
				patient1AD.setType(PrintAckFileRecord.DataType.AD);
				patient1AD.setPatientAccount(ackPatient1Stmt.getSiteStmtL().get(0).getSitePatient().getOldAcntNum());
				printAckDetailL.add(patient1AD);

				final ADFileRecord patient2AD = new ADFileRecord();
				patient2AD.setType(PrintAckFileRecord.DataType.AD);
				patient2AD.setPatientAccount(ackPatient2Stmt.getSiteStmtL().get(0).getSitePatient().getOldAcntNum());
				printAckDetailL.add(patient2AD);

				final PrintAckRec printAckRec = new PrintAckRec();
				printAckRec.setDateReceived(new Timestamp(ackReceivedDate.getTime()));
				printAckRec.setStationNum("" + ackStationNum);
				printAckRec.setStatusId(CBSDAOImplUtils.STATUS_INITIAL_ID);
				printAckRec.setFileName(printAckFileName);
				printAckRec.setBatchRunId(batchRunId);
				printAckRec.setPrintAckDetailL(printAckDetailL);
				getPrintAckRecIdS().add(printAckRecDao.insert(printAckRec));
				assertTrue(printAckRec.getId() > 0);

				final List<CBSStmt> noResultsCBSStmtL = cbsStmtDao.getStatementsForPrintAck(printAckRec,
						ackStatementDate, CBSDAOImplUtils.STATUS_SENT_ID, CBSDAOImplUtils.STATUS_ACK_ID)
						// Below is a temporary fix
						.values().stream().collect(Collectors.toList());
				testLogger.debug(CBSDAOImplUtils.cbsStmtListToString(noResultsCBSStmtL, false));
				assertNotNull(noResultsCBSStmtL);
				for (CBSStmt resultCBSStmt : noResultsCBSStmtL) {
					testLogger
							.debug("resultCBSStmt.getId():" + (resultCBSStmt == null ? "null" : resultCBSStmt.getId()));
				}
				assertEquals(2, noResultsCBSStmtL.size());
				assertNull(noResultsCBSStmtL.get(0));
				assertNull(noResultsCBSStmtL.get(0));

				// Create matching patient1 stmt
				cbsDAOImplUtils.saveCBSStmt(ackPatient1Stmt);
				getCbsStmtIdS().add(ackPatient1Stmt.getId());
				getAccountIdS().add(ackPatient1Stmt.getAccountId());

				// Create patient2 stmt in ACK
				cbsDAOImplUtils.saveCBSStmt(ackPatient2Stmt);
				getCbsStmtIdS().add(ackPatient2Stmt.getId());
				getAccountIdS().add(ackPatient2Stmt.getAccountId());

				final List<CBSStmt> twoResultsCBSStmtL = cbsStmtDao.getStatementsForPrintAck(printAckRec,
						ackStatementDate, CBSDAOImplUtils.STATUS_SENT_ID, CBSDAOImplUtils.STATUS_ACK_ID)
						// Below is a temporary fix
						.values().stream().collect(Collectors.toList());
				testLogger.debug(CBSDAOImplUtils.cbsStmtListToString(twoResultsCBSStmtL, false));
				assertNotNull(twoResultsCBSStmtL);
				assertEquals(2, twoResultsCBSStmtL.size());

				final List<CBSStmt> expectedTwoResultsCBSStmtL = new ArrayList<CBSStmt>(2);
				expectedTwoResultsCBSStmtL.add(ackPatient1Stmt);
				expectedTwoResultsCBSStmtL.add(ackPatient2Stmt);

				cbsDAOImplUtils.compareCBSStmtLists(twoResultsCBSStmtL, expectedTwoResultsCBSStmtL, false);

				cbsStmtDao.updatePrintAckFields(twoResultsCBSStmtL, printAckRec.getId(), ackPrintDate,
						CBSDAOImplUtils.STATUS_ACK_ID, printAckRec.getStationNum());

				final CBSStmt updatedAckPatient1Stmt = cbsStmtDao.get(ackPatient1Stmt.getId());
				assertNotNull(updatedAckPatient1Stmt);
				ackPatient1Stmt.setModifiedDate(updatedAckPatient1Stmt.getModifiedDate());
				ackPatient1Stmt.setPrintAckId(printAckRec.getId());
				ackPatient1Stmt.setPrintDate(ackPrintDate);
				ackPatient1Stmt.setStatusId(CBSDAOImplUtils.STATUS_ACK_ID);
				cbsDAOImplUtils.compareCBSStmt(updatedAckPatient1Stmt, ackPatient1Stmt);

				final CBSStmt updatedAckPatient2Stmt = cbsStmtDao.get(ackPatient2Stmt.getId());
				assertNotNull(updatedAckPatient2Stmt);
				ackPatient2Stmt.setModifiedDate(updatedAckPatient2Stmt.getModifiedDate());
				ackPatient2Stmt.setPrintAckId(printAckRec.getId());
				ackPatient2Stmt.setPrintDate(ackPrintDate);
				ackPatient2Stmt.setStatusId(CBSDAOImplUtils.STATUS_ACK_ID);
				cbsDAOImplUtils.compareCBSStmt(updatedAckPatient2Stmt, ackPatient2Stmt);

				final List<Long> printAckRecIdList = new ArrayList<Long>(1);
				printAckRecIdList.add(printAckRec.getId());
				final int[] numCbsStmtUpdatedAr = cbsStmtDao.undoPrintAcknowledgements(printAckRecIdList,
						CBSDAOImplUtils.STATUS_SENT_ID);
				assertNotNull(numCbsStmtUpdatedAr);
				assertEquals(1, numCbsStmtUpdatedAr.length);
				assertEquals(2, numCbsStmtUpdatedAr[0]);

				final CBSStmt updatedAckPatient1StmtUndone = cbsStmtDao.get(ackPatient1Stmt.getId());
				assertNotNull(updatedAckPatient1StmtUndone);
				ackPatient1Stmt.setModifiedDate(updatedAckPatient1StmtUndone.getModifiedDate());
				ackPatient1Stmt.setPrintAckId(0);
				ackPatient1Stmt.setPrintDate(null);
				ackPatient1Stmt.setStatusId(CBSDAOImplUtils.STATUS_SENT_ID);
				cbsDAOImplUtils.compareCBSStmt(updatedAckPatient1StmtUndone, ackPatient1Stmt);

				final CBSStmt updatedAckPatient2StmtUndone = cbsStmtDao.get(ackPatient2Stmt.getId());
				assertNotNull(updatedAckPatient2StmtUndone);
				ackPatient2Stmt.setModifiedDate(updatedAckPatient2StmtUndone.getModifiedDate());
				ackPatient2Stmt.setPrintAckId(0);
				ackPatient2Stmt.setPrintDate(null);
				ackPatient2Stmt.setStatusId(CBSDAOImplUtils.STATUS_SENT_ID);
				cbsDAOImplUtils.compareCBSStmt(updatedAckPatient2StmtUndone, ackPatient2Stmt);

			}

		}).run(jdbcTemplate);

	}

	/**
	 * Tests the getStatementCountWithStatus() method of CBSStmtDAOImpl
	 */
	@Test
	public final void testGetStatementCountWithStatus() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final int numStmtPerStatusToCreate = 2;
				final Long[] expectedNumStatuses = new Long[6];

				final long stationNumLong = 12345L;

				for (int statusId = 1; statusId <= expectedNumStatuses.length; statusId++) {
					expectedNumStatuses[statusId - 1] = cbsStmtDao.getStatementCountWithStatus(statusId)
							+ numStmtPerStatusToCreate;
				}

				for (int statusId = 1, key = 1; statusId <= expectedNumStatuses.length; statusId++) {

					for (int i = 1; i <= numStmtPerStatusToCreate; i++) {
						final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(key++, statusId,
								stationNumLong);
						getCbsStmtIdS().add(cbsStmtDao.save(cbsStmt));
						getAccountIdS().add(cbsStmt.getAccountId());
					}
				}

				for (int statusId = 1; statusId <= expectedNumStatuses.length; statusId++) {
					assertEquals(cbsStmtDao.getStatementCountWithStatus(statusId), expectedNumStatuses[statusId - 1]);
				}
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the updateMessageIdForStatus(), updateStatusForMessageId() and
	 * updateAllStatusFromTo() methods of CBSStmtDAOImpl
	 */
	@Test
	public final void testUpdateStatus() {
		//TODO: this test is hanging, fix
		if(true) {
			return;
		}
		
		(new SendCBSServiceTestWrapper() {

			@Override
			protected void test() {
				final long stationNumLong = 12345L;

				final int stmtStatusNewId = 2;
				final int stmtStatusSentId = 3;
				final int messageStatusInitialId = 1;

				// Create some statements in new status
				final int numStmts = 3;
				final List<CBSStmt> cbsStmtL = new ArrayList<CBSStmt>();
				cbsStmtL.addAll(getExistingNewStmts());
				for (int i = 0; i < numStmts; i++) {
					final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(i, stmtStatusNewId, stationNumLong);
					cbsDAOImplUtils.saveCBSStmt(cbsStmt);
					getCbsStmtIdS().add(cbsStmt.getId());
					getAccountIdS().add(cbsStmt.getAccountId());
					cbsStmtL.add(cbsStmt);
				}

				// Create the CBSMessage record
				final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();
				final int batchRunId = 1;
				final String fileName = "cbsTestFileNameSent.txt";

				final long messageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, messageStatusInitialId,
						batchRunId, fileName);
				getCbsMessageIdS().add(messageId);

				final long numUpdated = cbsStmtDao.updateMessageIdForStatus(stmtStatusNewId, messageId);
				assertEquals(numUpdated, cbsStmtL.size());

				for (CBSStmt cbsStmt : cbsStmtL) {
					CBSStmt updatedCbsStmt = cbsStmtDao.get(cbsStmt.getId());
					assertNotNull(updatedCbsStmt);
					assertEquals(updatedCbsStmt.getMessageId(), messageId);
				}

				final long numFromTo = cbsStmtDao.updateAllStatusFromTo(stmtStatusNewId, stmtStatusSentId);
				assertEquals(numFromTo, cbsStmtL.size());
				for (CBSStmt cbsStmt : cbsStmtL) {
					CBSStmt updatedCbsStmt = cbsStmtDao.get(cbsStmt.getId());
					assertNotNull(updatedCbsStmt);
					assertEquals(updatedCbsStmt.getStatusId(), stmtStatusSentId);
				}

				final long numStatusUpdated = cbsStmtDao.updateStatusForMessageId(stmtStatusNewId, messageId);
				assertEquals(numStatusUpdated, cbsStmtL.size());

				for (CBSStmt cbsStmt : cbsStmtL) {
					CBSStmt updatedCbsStmt = cbsStmtDao.get(cbsStmt.getId());
					assertNotNull(updatedCbsStmt);
					assertEquals(updatedCbsStmt.getStatusId(), stmtStatusNewId);
				}
			}

		}).run(jdbcTemplate);

	}

	private abstract class SendCBSServiceTestWrapper extends CBSDAOTestWrapper {

		private List<CBSStmt> existingNewStmts;

		public List<CBSStmt> getExistingNewStmts() {
			return existingNewStmts;
		}

		public void run(JdbcTemplate jdbcTemplate) {
			final int stmtStatusNewId = 2;

			// Get the current list of statements in new, so we can put them
			// back after the test
			final List<Long> existingNewIds = jdbcTemplate
					.query("SELECT id FROM CBSStmt WHERE statusId=" + stmtStatusNewId, new RowMapper<Long>() {
						@Override
						public Long mapRow(ResultSet rs, int rowNum) throws SQLException {
							return rs.getLong("id");
						}
					});
			existingNewStmts = new ArrayList<CBSStmt>(existingNewIds.size());
			for (long id : existingNewIds) {
				existingNewStmts.add(cbsStmtDao.get(id));
			}

			try {
				super.run(jdbcTemplate);
			} finally {
				for (CBSStmt cbsStmt : existingNewStmts) {
					// Revert status and messageId to previous state
					cbsStmtDao.save(cbsStmt);
				}
			}
		}

	}
}
