package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSSiteTransDAO;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * Unit tests the CBSSiteTransDAOImpl class.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class CBSSiteTransDAOImplTest {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSSiteTransDAO cbsSiteTransDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the save() and get() methods of CBSSiteTransDAOImpl
	 */
	@Test
	public final void testSaveGet() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final int cbsStmtKey = 1;
				final int statusId = 1;
				final long stationNumLong = 12345L;

				// Initialize the simple test CBSStmt object. The object graph
				// of
				// one CBSStmt, one CBSSiteStmt, one CBSSitePatient and two
				// CBSSiteTrans objects will be set up, but the ids of all
				// objects
				// will not be initialized. A CBSAccount is created in the
				// database.
				final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(cbsStmtKey, statusId, stationNumLong);

				List<CBSSiteTrans> originalTransL = new ArrayList<CBSSiteTrans>();
				List<CBSSiteTrans> updatedTransL = cbsStmt.getSiteStmtL().get(0).getSiteTransL();
				for (CBSSiteTrans cbsSiteTrans : updatedTransL) {
					originalTransL.add(cbsDAOImplUtils.copyCBSSiteTrans(cbsSiteTrans));
				}

				// Create the CBSStmt in the database.
				// cbsSiteTransDaoImpl.save()
				// will be called. Necessary to create dependent CBSAccount,
				// CBSStmt
				// and CBSSiteStmt records.
				cbsDAOImplUtils.saveCBSStmt(cbsStmt);
				getCbsStmtIdS().add(cbsStmt.getId());
				getAccountIdS().add(cbsStmt.getAccountId());

				int index = 0;
				for (CBSSiteTrans cbsSiteTrans : updatedTransL) {
					assertTrue(cbsSiteTrans.getId() > 0);
					// assertNotNull(createdBy);
					assertNotNull(cbsSiteTrans.getCreatedDate());
					// assertNotNull(modifiedBy);
					assertNotNull(cbsSiteTrans.getModifiedDate());

					// Fields not set before save
					CBSSiteTrans originalTrans = originalTransL.get(index++);
					originalTrans.setId(cbsSiteTrans.getId());
					originalTrans.setSiteStmtId(cbsSiteTrans.getSiteStmtId());
					originalTrans.setCreatedBy(cbsSiteTrans.getCreatedBy());
					originalTrans.setCreatedDate(cbsSiteTrans.getCreatedDate());
					originalTrans.setModifiedBy(cbsSiteTrans.getModifiedBy());
					originalTrans.setModifiedDate(cbsSiteTrans.getModifiedDate());

					// Assert relevant fields equal with original
					cbsDAOImplUtils.compareCBSSiteTrans(cbsSiteTrans, originalTrans);

					// Compare with database
					final CBSSiteTrans databaseTrans = cbsSiteTransDao.get(cbsSiteTrans.getId());
					assertNotNull(databaseTrans);
					cbsDAOImplUtils.compareCBSSiteTrans(cbsSiteTrans, databaseTrans);
				}
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the getAllByCBSSiteStmtID() method of CBSSiteTransDAOImpl
	 */
	@Test
	public final void testGetAllByCBSSiteStmtID() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final int statusId = 1;
				final long stationNum1 = 1L;
				final long stationNum2 = 2L;

				// Create a couple of statements
				final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(1, statusId, stationNum1);
				cbsDAOImplUtils.saveCBSStmt(cbsStmt);
				getCbsStmtIdS().add(cbsStmt.getId());
				getAccountIdS().add(cbsStmt.getAccountId());

				final CBSStmt cbsStmt2 = cbsDAOImplUtils.createDoubleTestCBSStmt(2, statusId, stationNum1, stationNum2);
				cbsDAOImplUtils.saveCBSStmt(cbsStmt2);
				getCbsStmtIdS().add(cbsStmt2.getId());
				getAccountIdS().add(cbsStmt2.getAccountId());

				callGetAllByCBSSiteStmtID(cbsStmt);
				callGetAllByCBSSiteStmtID(cbsStmt2);
			}

		}).run(jdbcTemplate);
	}

	@Test
	public final void testGetMessageStatementTransForSite() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final long stationNum1 = 1L;
				final long stationNum2 = 2L;
				final long stationNum3 = 3L;

				final int stmtStatusNewId = 2;
				final int stmtStatusSentId = 3;

				final int messageStatusInitialId = 1;
				final int messageStatusSuccessId = 2;

				final int sentBatchRunId = 1;
				final String sentFileName = "cbsTestFileNameSent.txt";

				final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();

				// Create an already sent statement for station 1
				final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(1, stmtStatusSentId, stationNum1);
				cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
				getCbsStmtIdS().add(sentCBSStmt1.getId());
				getAccountIdS().add(sentCBSStmt1.getAccountId());

				// Create the CBSMessage record for sent statement
				final long sentMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, messageStatusSuccessId,
						sentBatchRunId, sentFileName);
				getCbsMessageIdS().add(sentMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(sentCBSStmt1.getId(), sentMessageId);

				// Test with only one message and one stmt
				final List<CBSStmt> cbsStmtL1 = new ArrayList<CBSStmt>();
				cbsStmtL1.add(sentCBSStmt1);
				callMessageStatementTransForSite(sentMessageId, "" + stationNum1, cbsStmtL1);

				// Create a NEW statement for station 1
				final CBSStmt newCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(2, stmtStatusNewId, stationNum1);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt1);
				getCbsStmtIdS().add(newCBSStmt1.getId());
				getAccountIdS().add(newCBSStmt1.getAccountId());

				// Create a NEW statement for station 1 and station2. station 1
				// is primary
				final CBSStmt newCBSStmt1and2 = cbsDAOImplUtils.createDoubleTestCBSStmt(3, stmtStatusNewId, stationNum1,
						stationNum2);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt1and2);
				getCbsStmtIdS().add(newCBSStmt1and2.getId());
				getAccountIdS().add(newCBSStmt1and2.getAccountId());

				// Create a NEW statement for station 2 and station3. station 2
				// is primary
				final CBSStmt newCBSStmt2and3 = cbsDAOImplUtils.createDoubleTestCBSStmt(4, stmtStatusNewId, stationNum2,
						stationNum3);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt2and3);
				getCbsStmtIdS().add(newCBSStmt2and3.getId());
				getAccountIdS().add(newCBSStmt2and3.getAccountId());

				// Create the CBSMessage record for NEW statement
				final int newBatchRunId = 2;
				final String newFileName = "cbsTestFileNameNew.txt";
				final long newMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, messageStatusInitialId,
						newBatchRunId, newFileName);
				getCbsMessageIdS().add(newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1.getId(), newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1and2.getId(), newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt2and3.getId(), newMessageId);

				final List<CBSStmt> cbsStmtL2 = new ArrayList<CBSStmt>();
				cbsStmtL2.add(newCBSStmt1);
				cbsStmtL2.add(newCBSStmt1and2);
				callMessageStatementTransForSite(newMessageId, "" + stationNum1, cbsStmtL2);

				final List<CBSStmt> cbsStmtL3 = new ArrayList<CBSStmt>();
				cbsStmtL3.add(newCBSStmt2and3);
				callMessageStatementTransForSite(newMessageId, "" + stationNum2, cbsStmtL3);

				final List<CBSStmt> cbsStmtL4 = new ArrayList<CBSStmt>();
				callMessageStatementTransForSite(newMessageId, "" + stationNum3, cbsStmtL4);
			}

		}).run(jdbcTemplate);
	}

	private void callGetAllByCBSSiteStmtID(final CBSStmt cbsStmt) {
		final List<CBSSiteTrans> cbsSiteTransL = cbsSiteTransDao.getAllByCBSSiteStmtID(cbsStmt.getId());
		assertNotNull(cbsSiteTransL);
		assertEquals(cbsSiteTransL.size(), getTransCount(cbsStmt));
		for (CBSSiteTrans cbsSiteTrans : cbsSiteTransL) {
			assertNotNull(cbsSiteTrans);
			CBSSiteTrans expectedTrans = getTransWithId(cbsStmt, cbsSiteTrans.getId());
			assertNotNull(expectedTrans);
			cbsDAOImplUtils.compareCBSSiteTrans(cbsSiteTrans, expectedTrans);
		}
	}

	private int getTransCount(final CBSStmt cbsStmt) {
		int transCount = 0;
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			transCount += cbsSiteStmt.getSiteTransL().size();
		}
		return transCount;
	}

	private int getTransCountForStation(final List<CBSStmt> cbsStmtL, final String stationNum) {
		int transCount = 0;
		for (CBSStmt cbsStmt : cbsStmtL) {
			if (isStationPrimary(cbsStmt, stationNum)) {
				transCount += getTransCount(cbsStmt);
			}
		}
		return transCount;
	}

	private boolean isStationPrimary(final CBSStmt cbsStmt, final String stationNum) {
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			if (cbsSiteStmt.getIsPrimary().isTrue()) {
				return (stationNum.equals(cbsSiteStmt.getStationNum()));
			}
		}
		return false;
	}

	private void callMessageStatementTransForSite(final long sentMessageId, final String stationNum,
			final List<CBSStmt> cbsStmtL) {
		final List<CBSSiteTrans> cbsSiteTransL = cbsSiteTransDao.getMessageStatementTransForSite(sentMessageId,
				stationNum);
		assertNotNull(cbsSiteTransL);
		assertEquals(cbsSiteTransL.size(), getTransCountForStation(cbsStmtL, stationNum));
		for (CBSSiteTrans cbsSiteTrans : cbsSiteTransL) {
			assertNotNull(cbsSiteTrans);
			CBSSiteTrans expectedTrans = getTransWithId(cbsStmtL, cbsSiteTrans.getId());
			assertNotNull(expectedTrans);
			cbsDAOImplUtils.compareCBSSiteTrans(cbsSiteTrans, expectedTrans);
		}
	}

	private CBSSiteTrans getTransWithId(final List<CBSStmt> cbsStmtL, final long transId) {
		CBSSiteTrans foundTrans = null;
		for (CBSStmt cbsStmt : cbsStmtL) {
			for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
				for (CBSSiteTrans cbsSiteTrans : cbsSiteStmt.getSiteTransL()) {
					if (cbsSiteTrans.getId() == transId) {
						assertNull(foundTrans); // Should only find it once.
						foundTrans = cbsSiteTrans;
					}
				}
			}
		}
		return foundTrans;

	}

	private CBSSiteTrans getTransWithId(final CBSStmt cbsStmt, final long transId) {
		CBSSiteTrans foundTrans = null;
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			for (CBSSiteTrans cbsSiteTrans : cbsSiteStmt.getSiteTransL()) {
				if (cbsSiteTrans.getId() == transId) {
					assertNull(foundTrans); // Should only find it once.
					foundTrans = cbsSiteTrans;
				}
			}
		}
		return foundTrans;
	}

}
