package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.util.Date;

import javax.sql.DataSource;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSSitePatientDAO;
import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * Unit tests the CBSSitePatientDAOImpl class.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class CBSSitePatientDAOImplTest {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSSitePatientDAO cbsSitePatientDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the save() and get() methods of CBSSitePatientDAOImpl
	 */
	@Test
	public final void testSaveGet() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final int cbsStmtKey = 1;
				final int statusId = 1;
				final long stationNumLong = 12345L;

				// Initialize the simple test CBSStmt object. The object graph
				// of one
				// CBSStmt, one CBSSiteStmt, one CBSSitePatient and two
				// CBSSiteTrans
				// objects will be set up, but
				// the ids of all objects will not be initialized. A CBSAccount
				// is
				// created in the database.
				final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(cbsStmtKey, statusId, stationNumLong);

				// Assume functioning CBSSitePatient
				final CBSSitePatient cbsSitePatient = cbsStmt.getSiteStmtL().get(0).getSitePatient();

				final CBSSitePatient originalCbsSitePatient = cbsDAOImplUtils.copyCBSSitePatient(cbsSitePatient);

				// Create the CBSStmt in the database.
				// cbsSitePatientDaoImpl.save() will
				// be called.
				// Necessary to create dependent CBSAccount, CBSStmt and
				// CBSSiteStmt
				// records.
				cbsDAOImplUtils.saveCBSStmt(cbsStmt);
				getCbsStmtIdS().add(cbsStmt.getId());
				getAccountIdS().add(cbsStmt.getAccountId());

				final long id = cbsSitePatient.getId();
				final long siteStmtId = cbsSitePatient.getSiteStmtId();
				final String createdBy = cbsSitePatient.getCreatedBy();
				final Date createdDate = cbsSitePatient.getCreatedDate();
				final String modifiedBy = cbsSitePatient.getModifiedBy();
				final Date modifiedDate = cbsSitePatient.getModifiedDate();

				assertTrue(id > 0);
				assertTrue(siteStmtId > 0);
				// assertNotNull(createdBy);
				assertNotNull(createdDate);
				// assertNotNull(modifiedBy);
				assertNotNull(modifiedDate);

				// Fields not set before save
				originalCbsSitePatient.setId(id);
				originalCbsSitePatient.setSiteStmtId(siteStmtId);
				originalCbsSitePatient.setCreatedBy(createdBy);
				originalCbsSitePatient.setCreatedDate(createdDate);
				originalCbsSitePatient.setModifiedBy(modifiedBy);
				originalCbsSitePatient.setModifiedDate(modifiedDate);

				// Assert relevant fields equal with original
				cbsDAOImplUtils.compareCBSSitePatients(cbsSitePatient, originalCbsSitePatient);

				final CBSSitePatient newCbsSitePatient = cbsSitePatientDao.get(id);

				assertNotNull(newCbsSitePatient);

				cbsDAOImplUtils.compareCBSSitePatients(cbsSitePatient, newCbsSitePatient);
			}

		}).run(jdbcTemplate);
	}

}
