package gov.va.cpss.dao.impl;

import static org.junit.Assert.*;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSPatAcntBalMsgDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.bal.PatAcntBalMsg;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml"})
/**
 * Unit test for CBSPatAcntBalMsgDAOImpl.
 * 
 * @author Andrew Vance
 *
 */
public final class CBSPatAcntBalMsgDAOImplTest {
	
	@Autowired
	private CBSPatAcntBalMsgDAO cbsPatAcntBalMsgDao;
	
	@Autowired
	private ProcessStatusDAO processStatusDao;
	
	private List<PatAcntBalMsg> patAcntBalMsgL;	
	
	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	
	@Before
	public final void beforeTest() {
		insertTestData();
	}
	
	@After
	public final void afterTest() {
		cleanupTestData();
	}
	
	@Test
	public final void testGet() {
		final PatAcntBalMsg patAcntBalMsgExpected = patAcntBalMsgL.get(0);
		
		final PatAcntBalMsg patAcntBalMsgRetrieved = cbsPatAcntBalMsgDao.get(patAcntBalMsgExpected.getId());
		
		assertPatAcntBalMsgEquals(patAcntBalMsgExpected, patAcntBalMsgRetrieved);
	}
	
	@Test
	public final void testGetCBSPatAcntBalMsgInStatus() {
		// getStatusId() is okay to use because it was supplied by ProcessStatusDAO in insertTestData()
		final long targetStatusId = patAcntBalMsgL.get(0).getStatusId();
		
		final List<PatAcntBalMsg> msgExpectedL = patAcntBalMsgL.stream().filter(msg -> msg.getStatusId() == targetStatusId).collect(Collectors.toList());
		
		final List<PatAcntBalMsg> msgRetrievedL = cbsPatAcntBalMsgDao.getCBSPatAcntBalMsgInStatus((int)targetStatusId);
		
		// DB may have more than only test data in it, hence >= instead of ==
		assertTrue(msgRetrievedL.size() >= msgExpectedL.size());
		
		final List<Long> msgRetrievedIdL = msgRetrievedL.stream().map(PatAcntBalMsg::getId).collect(Collectors.toList());		
		assertTrue(msgRetrievedIdL.containsAll(msgExpectedL.stream().map(PatAcntBalMsg::getId).collect(Collectors.toList())));
		
		// Use the IDs of all PatAcntBalMsg that do NOT have targetStatusId to verify only correct PatAcntBalMsg were obtained.
		final List<PatAcntBalMsg> msgNotExpectedL = patAcntBalMsgL.stream().filter(msg -> msg.getStatusId() != targetStatusId).collect(Collectors.toList());
		msgNotExpectedL.forEach(msg -> assertFalse(msgRetrievedL.contains(msg.getId())));
	}
	
	@Test
	public final void testUpdate() {
		final PatAcntBalMsg msgOriginal = new PatAcntBalMsg();
		msgOriginal.setBatchRunId(12);
		msgOriginal.setFileName("fakefileee.test");
		msgOriginal.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		msgOriginal.setTotNumPatient(131);
		msgOriginal.setId(cbsPatAcntBalMsgDao.insert(msgOriginal));
		
		final PatAcntBalMsg msgOriginalRetrieved = cbsPatAcntBalMsgDao.get(msgOriginal.getId());
		assertPatAcntBalMsgEquals(msgOriginal, msgOriginalRetrieved);
		
		// Test update
		msgOriginal.setBatchRunId(22);
		msgOriginal.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.REPLACED));
		msgOriginal.setFileName("updatedfilename.test");		
		cbsPatAcntBalMsgDao.update(msgOriginal);
		
		final PatAcntBalMsg msgUpdatedRetrieved = cbsPatAcntBalMsgDao.get(msgOriginal.getId());
		assertPatAcntBalMsgEquals(msgOriginal, msgUpdatedRetrieved);
		
		// Test updateTotalPatients
		final long newTotalPatients = 1738L;
		cbsPatAcntBalMsgDao.updateTotalPatients(newTotalPatients, msgOriginal.getId());
		final PatAcntBalMsg msgTotalPatientsUpdatedRetrieved = cbsPatAcntBalMsgDao.get(msgOriginal.getId());
		assertEquals(newTotalPatients, msgTotalPatientsUpdatedRetrieved.getTotNumPatient());
	}
	
	private final void insertTestData() {
		final PatAcntBalMsg patAcntBalMsg1 = new PatAcntBalMsg();
		patAcntBalMsg1.setBatchRunId(1);
		patAcntBalMsg1.setFileName("fakefile.test");
		patAcntBalMsg1.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		patAcntBalMsg1.setTotNumPatient(11);
		patAcntBalMsg1.setId(cbsPatAcntBalMsgDao.insert(patAcntBalMsg1));
		
		final PatAcntBalMsg patAcntBalMsg2 = new PatAcntBalMsg();
		patAcntBalMsg2.setBatchRunId(12);
		patAcntBalMsg2.setFileName("fakefi2le2.test");
		patAcntBalMsg2.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.SUCCESS));
		patAcntBalMsg2.setTotNumPatient(121);
		patAcntBalMsg2.setId(cbsPatAcntBalMsgDao.insert(patAcntBalMsg2));
		
		final PatAcntBalMsg patAcntBalMsg3 = new PatAcntBalMsg();
		patAcntBalMsg3.setBatchRunId(12);
		patAcntBalMsg3.setFileName("fakefile2.test");
		patAcntBalMsg3.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		patAcntBalMsg3.setTotNumPatient(121);
		patAcntBalMsg3.setId(cbsPatAcntBalMsgDao.insert(patAcntBalMsg3));
		
		patAcntBalMsgL = new ArrayList<PatAcntBalMsg>();
		patAcntBalMsgL.add(patAcntBalMsg1);
		patAcntBalMsgL.add(patAcntBalMsg2);
		patAcntBalMsgL.add(patAcntBalMsg3);
	}
	
	private final void cleanupTestData() {
		patAcntBalMsgL.forEach(patAcntBalMsg -> {
			final String sql = "DELETE FROM PatAcntBalMsg WHERE ID=?";
			
			jdbcTemplate.update(new PreparedStatementCreator() {
				@Override
				public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
					final PreparedStatement ps = con.prepareStatement(sql);
					ps.setLong(1, patAcntBalMsg.getId());
					return ps;
				}
			});
		});
	}
	
	private static final void assertPatAcntBalMsgEquals(final PatAcntBalMsg expected, final PatAcntBalMsg retrieved) {
		assertEquals(expected.getId(), retrieved.getId());
		assertEquals(expected.getBatchRunId(), retrieved.getBatchRunId());
		assertEquals(expected.getStatusId(), retrieved.getStatusId());
		assertEquals(expected.getFileName(), retrieved.getFileName());
		assertEquals(expected.getTotNumPatient(), retrieved.getTotNumPatient());
	}
}
