package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSMessageDAO;
import gov.va.cpss.model.cbs.CBSMessage;

/**
 * Unit tests the CBSMessageDAOImpl class.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class CBSMessageDAOImplTest {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSMessageDAO cbsMessageDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the insert() and get() methods of CBSMessageDAOImpl
	 */
	@Test
	public final void testInsertGet() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final int statusId = 1;
				final long batchRunId = 1;
				final String fileName = "cbsTestFileName.txt,sbsTestFileName.txt";

				// Assume functioning CBSMessage
				final CBSMessage cbsMessage = new CBSMessage();

				cbsMessage.setStatusId(statusId);
				cbsMessage.setBatchRunId(batchRunId);
				cbsMessage.setFileName(fileName);

				final long id = cbsMessageDao.insert(cbsMessage);
				getCbsMessageIdS().add(id);

				assertTrue(id > 0);

				assertEquals(cbsMessage.getStatusId(), statusId);
				assertEquals(cbsMessage.getBatchRunId(), batchRunId);
				assertEquals(cbsMessage.getFileName(), fileName);

				assertEquals(cbsMessage.getId(), id);
				// assertNotNull(cbsMessage.getCreatedBy());
				assertNotNull(cbsMessage.getCreatedDate());
				// assertNotNull(cbsMessage.getModifiedBy());
				assertNotNull(cbsMessage.getModifiedDate());

				final CBSMessage newCBSMessage = cbsMessageDao.get(id);

				assertNotNull(newCBSMessage);

				cbsDAOImplUtils.compareCBSMessages(cbsMessage, newCBSMessage);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the update() method of CBSMessageDAOImpl
	 */
	@Test
	public final void testUpdate() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				// Original
				final int statusId = 1;
				final long batchRunId = 1;
				final String fileName = "cbsTestFileName.txt,sbsTestFileName.txt";

				// Assume functioning CBSMessage
				final CBSMessage cbsMessage = new CBSMessage();
				cbsMessage.setStatusId(statusId);
				cbsMessage.setBatchRunId(batchRunId);
				cbsMessage.setFileName(fileName);

				// Original
				final long id = cbsMessageDao.insert(cbsMessage);
				getCbsMessageIdS().add(id);
				final String createdBy = cbsMessage.getCreatedBy();
				final Date createdDate = cbsMessage.getCreatedDate();
				final String modifiedBy = cbsMessage.getModifiedBy();

				// Test Data
				// Updated
				final int updatedStatusId = 2;
				final long updatedBatchRunId = 2;
				final String updatedFileName = "cbsTestFileName2.txt,sbsTestFileName2.txt";

				cbsMessage.setStatusId(updatedStatusId);
				cbsMessage.setBatchRunId(updatedBatchRunId);
				cbsMessage.setFileName(updatedFileName);

				cbsMessageDao.update(cbsMessage);

				final CBSMessage newCBSMessage = cbsMessageDao.get(id);

				assertNotNull(newCBSMessage);

				assertEquals(newCBSMessage.getId(), id);
				assertEquals(newCBSMessage.getStatusId(), updatedStatusId);
				assertEquals(newCBSMessage.getBatchRunId(), updatedBatchRunId);
				assertEquals(newCBSMessage.getFileName(), updatedFileName);
				assertEquals(newCBSMessage.getCreatedBy(), createdBy);
				assertEquals(newCBSMessage.getCreatedDate(), createdDate);
				assertEquals(newCBSMessage.getModifiedBy(), modifiedBy);
				assertEquals(newCBSMessage.getModifiedDate(), cbsMessage.getModifiedDate());
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the getCBSMessagesInStatus() method of CBSMessagesDAOImpl
	 */
	@Test
	public final void testGetCBSMessagesInStatus() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final Map<Integer, List<CBSMessage>> cbsMessages = getCurrentCBSMessages();

				final int[] statusIds = { 1, 2, 3 }; // Status ids to test

				final long batchRunId = 1;
				final String fileName = "cbsTestFileName.txt,sbsTestFileName.txt";

				for (int statusId : statusIds) {
					List<CBSMessage> cbsMessagesL = cbsMessages.get(statusId);
					if (cbsMessagesL == null) {
						cbsMessagesL = new ArrayList<CBSMessage>();
						cbsMessages.put(statusId, cbsMessagesL);
					}

					// Add two new records for this status
					getCbsMessageIdS()
							.add(cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, statusId, batchRunId, fileName));
					getCbsMessageIdS()
							.add(cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, statusId, batchRunId, fileName));

					List<CBSMessage> cbsMessagesInStatus = cbsMessageDao.getCBSMessagesInStatus(statusId);
					assertNotNull(cbsMessagesInStatus);

					compareCBSMessageLists(cbsMessagesInStatus, cbsMessagesL);
				}
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Build and return a Map of all the CBSMessage records in the database,
	 * organized by statusId.
	 * 
	 * @return A Map where keys are statusIds of existing CBSMessage records and
	 *         values are Lists of CBSMessages with that status.
	 */
	private Map<Integer, List<CBSMessage>> getCurrentCBSMessages() {
		final Map<Integer, List<CBSMessage>> currentCBSMessages = new HashMap<Integer, List<CBSMessage>>();

		final List<Long> cbsMessageIds = jdbcTemplate.query("SELECT id FROM CBSMessage ORDER BY id",
				new RowMapper<Long>() {
					@Override
					public Long mapRow(ResultSet rs, int row) throws SQLException {
						return new Long(rs.getLong(1));
					}
				});

		for (long cbsMessageId : cbsMessageIds) {
			final CBSMessage cbsMessage = cbsMessageDao.get(cbsMessageId);

			final int statusId = cbsMessage.getStatusId();

			List<CBSMessage> cbsMessagesL = currentCBSMessages.get(statusId);
			if (cbsMessagesL == null) {
				cbsMessagesL = new ArrayList<CBSMessage>();
				currentCBSMessages.put(statusId, cbsMessagesL);
			}

			cbsMessagesL.add(cbsMessage);
		}

		return currentCBSMessages;
	}

	/**
	 * Compares two lists of CBSMessage objects and asserts that the properties
	 * of the objects from each list are equal. Objects expected to be identical
	 * should be at the same index in each list.
	 * 
	 * @param list1
	 * @param list2
	 */
	private void compareCBSMessageLists(List<CBSMessage> list1, List<CBSMessage> list2) {
		assertEquals(list1.size(), list2.size());

		int index = 0;
		for (CBSMessage cbsMessage1 : list1) {
			CBSMessage cbsMessage2 = list2.get(index++);

			assertNotNull(cbsMessage1);
			assertNotNull(cbsMessage2);

			cbsDAOImplUtils.compareCBSMessages(cbsMessage1, cbsMessage2);
		}
	}

}
