package gov.va.cpss.dao.impl;

import java.util.HashSet;
import java.util.Set;

import org.apache.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;

/**
 * Provides a wrapper for DAO tests using test data in the CBS tables. Tests
 * using this wrapper should provide an implementation for the test() method,
 * where the custom test should be implemented. Within the test() method, the
 * ids of test records added to the CBSStmt, CBSMessage and CBSAccount tables
 * during the test should be added to the corresponding Id sets and the wrapper
 * will ensure that the records are deleted at the end of the test.
 * 
 * @author Brad Pickle
 *
 */
public abstract class CBSDAOTestWrapper {

	protected Logger logger = Logger.getLogger(this.getClass().getName());
	
	private final Set<Long> cbsStmtIdS = new HashSet<Long>();

	private final Set<Long> cbsMessageIdS = new HashSet<Long>();

	private final Set<Long> accountIdS = new HashSet<Long>();

	private final Set<Long> printAckRecIdS = new HashSet<Long>();

	/**
	 * Returns the Set used to manage the set of record ids for test records
	 * created in the CBSStmt table. Ids in this Set at the end of the test will
	 * be deleted from the table.
	 * 
	 * @return The Set of ids of test records added to the CBSStmt table. Tests
	 *         using this wrapper should add/remove ids of the corresponding
	 *         records as they are inserted/deleted from the CBSStmt table
	 *         during the test.
	 */
	public Set<Long> getCbsStmtIdS() {
		return cbsStmtIdS;
	}

	/**
	 * Returns the Set used to manage the set of record ids for test records
	 * created in the CBSMessage table. Ids in this Set at the end of the test
	 * will be deleted from the table.
	 * 
	 * @return The Set of ids of test records added to the CBSMessage table.
	 *         Tests using this wrapper should add/remove ids of the
	 *         corresponding records as they are inserted/deleted from the
	 *         CBSMessage table during the test.
	 */
	public Set<Long> getCbsMessageIdS() {
		return cbsMessageIdS;
	}

	/**
	 * Returns the Set used to manage the set of record ids for test records
	 * created in the CBSAccount table. Ids in this Set at the end of the test
	 * will be deleted from the table.
	 * 
	 * @return The Set of ids of test records added to the CBSAccount table.
	 *         Tests using this wrapper should add/remove ids of the
	 *         corresponding records as they are inserted/deleted from the
	 *         CBSAccount table during the test.
	 */
	public Set<Long> getAccountIdS() {
		return accountIdS;
	}

	/**
	 * Returns the Set used to manage the set of record ids for test records
	 * created in the CBSAccount table. Ids in this Set at the end of the test
	 * will be deleted from the table.
	 * 
	 * @return The Set of ids of test records added to the CBSAccount table.
	 *         Tests using this wrapper should add/remove ids of the
	 *         corresponding records as they are inserted/deleted from the
	 *         CBSAccount table during the test.
	 */
	public Set<Long> getPrintAckRecIdS() {
		return printAckRecIdS;
	}

	/**
	 * Runs the test within the wrapped context. Tests using this wrapper should
	 * kick the tests off with this method. A JdbcTemplate should be provided to
	 * be used to delete the test records.
	 * 
	 * @param jdbcTemplate
	 *            JdbcTemplate to use to delete the test records.
	 */
	public void run(final JdbcTemplate jdbcTemplate) {
		try {
			test();
		} finally {
			for (Long cbsStmtId : cbsStmtIdS) {
				jdbcTemplate.update("DELETE FROM CBSStmt WHERE id=" + cbsStmtId);
			}
			for (Long cbsMessageId : cbsMessageIdS) {
				jdbcTemplate.update("DELETE FROM CBSMessage WHERE id=" + cbsMessageId);
			}
			for (Long accountId : accountIdS) {
				jdbcTemplate.update("DELETE FROM CBSAccount WHERE id=" + accountId);
			}
			for (Long printAckRecId : printAckRecIdS) {
				jdbcTemplate.update("DELETE FROM PrintAckRec WHERE id=" + printAckRecId);
			}
		}
	}

	/**
	 * The test method that should contain the tests to run. An implementation
	 * for this method should be provided by tests using this wrapper.
	 */
	protected abstract void test();
}
