package gov.va.cpss.dao.impl;

import static org.junit.Assume.assumeTrue;

import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.junit.After;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.BatchConfigDAO;
import gov.va.cpss.dao.BatchJobDAO;
import gov.va.cpss.dao.BatchRunDAO;
import gov.va.cpss.dao.PSDetailsDAO;
import gov.va.cpss.dao.PSPatientDAO;
import gov.va.cpss.dao.PSReceivedDAO;
import gov.va.cpss.dao.PSSiteDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BatchConfig;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSReceived;
import gov.va.cpss.model.fps.PSSite;

/**
 * Abstract class used to create unit tests for CBS DAOs. Subclasses must implement the insertTestData() method, and the supplied 
 * cleanupTestData() method will then automatically remove all test data from the database.
 * 
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public abstract class AbstractCbsDaoImplTest {
	@Autowired
	protected CBSDAOImplUtils cbsDaoImplUtils;
	
	@Autowired
	protected BatchConfigDAO batchConfigDao;
	
	@Autowired
	protected BatchJobDAO batchJobDao;
	
	@Autowired
	protected BatchRunDAO batchRunDao;
	
	@Autowired
	protected PSSiteDAO psSiteDao;
	
	@Autowired
	protected PSReceivedDAO psReceivedDao;
	
	@Autowired
	protected PSPatientDAO psPatientDao;
	
	@Autowired
	protected PSDetailsDAO psDetailsDao;
	
	@Autowired
	protected ProcessStatusDAO processStatusDao;
	
	protected List<BatchRun> testBatchRunL;
	
	protected List<BatchConfig> testBatchConfigL;
	
	protected List<BatchJob> testBatchJobL;
	
	protected List<PSSite> testPsSiteL;
	
	protected List<PSReceived> testPsReceivedL;
	
	protected List<PSPatient> testPsPatientL;
	
	protected List<PSDetails> testPsDetailsL;
	
	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;
	
	protected JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}
	
	public AbstractCbsDaoImplTest() {
		initializeTestDataLists();
	}
	
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		
		insertTestData();
	}
	
	@After
	public final void afterTest() {
		if(runIntegrationTest) {
			cleanupTestData();
		}		
	}
	
	private final void initializeTestDataLists() {
		
		testBatchRunL = new ArrayList<BatchRun>();
		
		testBatchConfigL = new ArrayList<BatchConfig>();
		
		testBatchJobL = new ArrayList<BatchJob>();
		
		testPsSiteL = new ArrayList<PSSite>();
		
		testPsReceivedL = new ArrayList<PSReceived>();
		
		testPsPatientL = new ArrayList<PSPatient>();
		
		testPsDetailsL = new ArrayList<PSDetails>();
	}
	
	private final void cleanupTestData() {
		testPsDetailsL.forEach(psDetails -> {
			final String sql = "DELETE FROM PSDetails WHERE ID = ?";
			
			jdbcTemplate.update(sql, psDetails.getId());
		});
		
		testPsReceivedL.forEach(psReceived -> {
			final String sql = "DELETE FROM PSReceived WHERE ID = ?";
			
			jdbcTemplate.update(sql, psReceived.getId());
		});
		
		testPsPatientL.forEach(psPatient -> {
			final String sql = "DELETE FROM PSPatient WHERE ID = ?";
			
			jdbcTemplate.update(sql, psPatient.getId());
		});
		
		testPsSiteL.forEach(psSite -> {
			final String sql = "DELETE FROM PSSite WHERE ID = ?";
			
			jdbcTemplate.update(sql, psSite.getId());
		});
		
		testBatchRunL.forEach(batchRun -> {
			final String sql = "DELETE FROM BatchRun WHERE ID = ?";
			
			jdbcTemplate.update(sql, batchRun.getId());
		});
		
		testBatchJobL.forEach(batchJob -> {
			String sql = "DELETE FROM BatchJob WHERE id=?";
			jdbcTemplate.update(sql, batchJob.getId());
		});
		
		testBatchConfigL.forEach(batchConfig -> {
			String sql = "DELETE FROM BatchConfig WHERE batchJobId=?";
			jdbcTemplate.update(sql, batchConfig.getBatchJobId());
		});
	}
	
	protected abstract void insertTestData();
}
