package gov.va.cpss.dao.impl;

import static org.junit.Assume.assumeTrue;

import java.util.ArrayList;
import java.util.List;

import org.junit.After;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.APSDetailsDAO;
import gov.va.cpss.dao.APSPatientDAO;
import gov.va.cpss.dao.APSPaymentDAO;
import gov.va.cpss.dao.APSReceivedDAO;
import gov.va.cpss.dao.APSSiteDAO;
import gov.va.cpss.dao.APSSitePatientDAO;
import gov.va.cpss.dao.APSSiteStmtDAO;
import gov.va.cpss.dao.APSStmtDAO;
import gov.va.cpss.dao.BatchJobDAO;
import gov.va.cpss.dao.BatchRunDAO;
import gov.va.cpss.dao.BatchRunProcessDAO;
import gov.va.cpss.dao.BatchStatusDAO;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.apps.APSDetails;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSPayment;
import gov.va.cpss.model.apps.APSReceived;
import gov.va.cpss.model.apps.APSSite;
import gov.va.cpss.model.apps.APSSitePatient;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.cbs.CBSAccount;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/apps-dao-test-context.xml" })
/**
 * Abstract class used to great unit tests for APS DAOs. Subclasses must implement the insertTestData() method, and the supplied 
 * cleanupTestData() method will then automatically remove all test data from the database.
 * 
 * @author Andrew Vance
 *
 */
public abstract class AbstractApsDaoImplTest {
	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;
	
	@Autowired
	protected APPSDAOImplUtils appsDaoImplUtils;
	
	@Autowired
	protected APSDetailsDAO apsDetailsDao;
	
	@Autowired
	protected APSPatientDAO apsPatientDao;
	
	@Autowired
	protected APSPaymentDAO apsPaymentDao;
	
	@Autowired
	protected APSReceivedDAO apsReceivedDao;
	
	@Autowired
	protected APSSiteDAO apsSiteDao;
	
	@Autowired
	protected APSSitePatientDAO apsSitePatientDao;
	
	@Autowired
	protected APSSiteStmtDAO apsSiteStmtDao;
	
	@Autowired
	protected APSStmtDAO apsStmtDao;
	
	@Autowired
	protected BatchRunDAO batchRunDao;
	
	@Autowired
	protected BatchJobDAO batchJobDao;
	
	@Autowired
	protected BatchRunProcessDAO batchRunProcessDao;
	
	@Autowired
	protected BatchStatusDAO batchStatusDao;
	
	@Autowired
	protected CBSAccountDAO cbsAccountDao;
	
	@Autowired
	protected ProcessStatusDAO processStatusDao;

	protected List<APSDetails> testApsDetailsL;
	protected List<APSPatient> testApsPatientL;
	protected List<APSPayment> testApsPaymentL;
	protected List<APSReceived> testApsReceivedL;
	protected List<APSSite> testApsSiteL;
	protected List<APSSitePatient> testApsSitePatientL;
	protected List<APSSiteStmt> testApsSiteStmtL;
	protected List<APSStmt> testApsStmtL;
	protected List<BatchRun> testBatchRunL;
	protected List<BatchJob> testBatchJobL;
	protected List<BatchRunProcess> testBatchRunProcessL;
	protected List<CBSAccount> testCbsAccountL;
	
	/**
	 * Public constructor in which the test data lists are expected to be initialized. Constructors in any subclasses
	 * should include a call to super(), or find another way to initialize the data lists.
	 */
	public AbstractApsDaoImplTest() {
		initializeTestDataLists();
	}
	
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		
		insertTestData();
	}
	
	@After
	public final void afterTest() {
		if(runIntegrationTest) {
			cleanupTestData();
		}		
	}
	
	/**
	 * Creates empty lists of all relevant data types.
	 */
	private final void initializeTestDataLists() {
		// TODO: Can improve performance slightly by requiring subclasses to instantiate the lists only when they need it. Andrew Vance 3/17/2017
		testApsDetailsL = new ArrayList<APSDetails>();
		testApsPatientL = new ArrayList<APSPatient>();
		testApsPaymentL = new ArrayList<APSPayment>();
		testApsReceivedL = new ArrayList<APSReceived>();
		testApsSiteL = new ArrayList<APSSite>();
		testApsSitePatientL = new ArrayList<APSSitePatient>();
		testApsSiteStmtL = new ArrayList<APSSiteStmt>();
		testApsStmtL = new ArrayList<APSStmt>();
		testCbsAccountL = new ArrayList<CBSAccount>();
		testBatchRunProcessL = new ArrayList<BatchRunProcess>();
		testBatchRunL = new ArrayList<BatchRun>();
	}
	
	/**
	 * Removes all test data from database by iterating through every testApsFooL. Note that due to foreign key constraints, the 
	 * order of these removals is very strict and should only be changed with caution.
	 */
	protected final void cleanupTestData() {
		// TODO: Performance could be gained from doing a batch delete, by CBSDAO objects aren't 
		// required to have a batch delete method. Andrew Vance 3/17/2017 
		if(testApsPatientL != null) {
			testApsPatientL.forEach(apsPatient -> apsPatientDao.deleteById(apsPatient.getId()));
		}
		
		if(testApsReceivedL != null) {
			testApsReceivedL.forEach(apsReceived -> apsReceivedDao.deleteById(apsReceived.getId()));
		}
		
		if(testApsDetailsL != null) {
			testApsDetailsL.forEach(apsDetails -> apsDetailsDao.deleteById(apsDetails.getId()));
		}
		
		if(testApsSitePatientL != null) {
			testApsSitePatientL.forEach(apsSitePatient -> apsSitePatientDao.deleteById(apsSitePatient.getId()));
		}
		
		if(testApsSiteStmtL != null) {
			testApsSiteStmtL.forEach(apsSiteStmt -> apsSiteStmtDao.deleteById(apsSiteStmt.getId()));
		}
		
		if(testApsSiteL != null) {
			testApsSiteL.forEach(apsSite -> apsSiteDao.deleteById(apsSite.getId()));
		}
		
		if(testApsStmtL != null) {
			testApsStmtL.forEach(apsStmt -> apsStmtDao.deleteById(apsStmt.getId()));
		}
		
		if(testBatchRunProcessL != null) {
			testBatchRunProcessL.forEach(batchRunProcess -> batchRunProcessDao.deleteById(batchRunProcess.getId()));
		}
		
		if(testBatchRunL != null) {
			testBatchRunL.forEach(batchRun -> appsDaoImplUtils.deleteBatchRun(batchRun));
		}
		
		if(testCbsAccountL != null) {
			testCbsAccountL.forEach(cbsAccount -> appsDaoImplUtils.deleteCbsAccount(cbsAccount.getId()));
		}
	}
	
	/**
	 * Implementations should insert data into the database and add the APSFoo objects to its respective testApsFooL. 
	 * e.g. testApsStmtL.add((APSStmt)stmt). Typically, the cleanupTestData() will delete the record by ID, so it is
	 * important that the model also includes the ID from the database. A simple way to achieve this is apsFoo.setId(apsFooDao.save(apsFoo));
	 */
	protected abstract void insertTestData();
}
