package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.time.Instant;
import java.time.Year;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.apps.APSPayment;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;
import gov.va.cpss.model.cbs.CBSAccount;

/**
 * Unit tests for the APSStmtDAOImpl class.
 * 
 * @author Andrew Vance
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/apps-dao-test-context.xml" })
public final class APSStmtDAOImplTest extends AbstractApsDaoImplTest {

	@Test
	public final void testGet() {
		final APSStmt apsStmtExpected = testApsStmtL.get(0);

		final APSStmt apsStmtRetrieved = apsStmtDao.get(apsStmtExpected.getId());

		assertNotNull(apsStmtRetrieved);

		assertApsStmtEquals(apsStmtExpected, apsStmtRetrieved);
	}

	@Test
	public final void testGetAllStatementsForStatus() {
		final APSStmt apsStmtExpected = testApsStmtL.get(0);

		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getAllStatementsForStatus(
				processStatusDao.getStatusType(apsStmtExpected.getStatusId()).getStatus());

		// Ensure statements retrieved only have proper status
		apsStmtRetrievedL.forEach(stmt -> assertEquals(apsStmtExpected.getStatusId(), stmt.getStatusId()));

		// Ensure statements retrieved contain at least the test APSStmt with the right status, and not the one
		// with the wrong status
		final List<Long> apsStmtRetrievedIdL = apsStmtRetrievedL.stream().map(APSStmt::getId).collect(Collectors.toList());
		assertTrue(apsStmtRetrievedIdL.contains(apsStmtExpected.getId()));
		assertFalse(apsStmtRetrievedIdL.contains(testApsStmtL.get(1).getId()));
	}

	@Test
	public final void testDeleteByGenProcessId() {

		//Create test CBSAccount
		final CBSAccount cbsAccount = appsDaoImplUtils.createTestCbsAccount();
		cbsAccount.setId(cbsAccountDao.save(cbsAccount.getIcn()));
		testCbsAccountL.add(cbsAccount);

		//Create test BatchRunProcess
		final BatchRunProcess batchRunProcessGen = appsDaoImplUtils.createTestBatchRunProcess(testBatchRunL.get(0));
		batchRunProcessGen.setId(batchRunProcessDao.save(batchRunProcessGen));
		testBatchRunProcessL.add(batchRunProcessGen);

		final APSStmt apsStmtToDelete = appsDaoImplUtils.createTestAPSStmt(cbsAccount);
		apsStmtToDelete.setGenBatchRunId(batchRunProcessGen.getId());
		apsStmtToDelete.setId(apsStmtDao.save(apsStmtToDelete));
		testApsStmtL.add(apsStmtToDelete);

		final APSStmt apsStmtRetrieved = apsStmtDao.get(apsStmtToDelete.getId());
		assertNotNull(apsStmtRetrieved);
		assertEquals(apsStmtToDelete.getId(), apsStmtRetrieved.getId());
		assertEquals(apsStmtToDelete.getGenBatchRunId(), apsStmtRetrieved.getGenBatchRunId());

		apsStmtDao.deleteByGenProcessId(batchRunProcessGen.getId());		
		assertNull(apsStmtDao.get(apsStmtToDelete.getId()));
		testApsStmtL.remove(apsStmtToDelete);
		
		// Make sure extra things weren't deleted
		final List<APSStmt> apsRetrievedL = apsStmtDao.getExistingStatements(testApsStmtL);
		assertNotNull(apsRetrievedL);
		assertEquals(testApsStmtL.size(), apsRetrievedL.size());
	}

	@Test
	public void testGetExistingStatements() {
		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getExistingStatements(testApsStmtL);
		assertNotNull(apsStmtRetrievedL);
		assertApsStmtListEquals(testApsStmtL, apsStmtRetrievedL);
	}

	@Test
	public void testGetGenCountWithStatus() {
		final APSStmt apsStmtExpected = testApsStmtL.get(0);

		final long genCount = apsStmtDao.getGenCountWithStatus(apsStmtExpected.getGenBatchRunId(), 
				processStatusDao.getStatusType(apsStmtExpected.getStatusId()).getStatus());

		assertEquals(1, genCount);
	}

	@Test
	public void testUpdateStatusToSent() {
		final List<APSStmt> apsStmtInputL = new ArrayList<APSStmt>();
		apsStmtInputL.add(testApsStmtL.get(0));
		apsStmtInputL.add(testApsStmtL.get(2));

		final int[] updated = apsStmtDao.updateStatusToSent(apsStmtInputL, testBatchRunProcessL.get(0).getId());
		assertEquals(apsStmtInputL.size(), updated.length); 

		// Status gets updated in DB but not in our models
		apsStmtInputL.forEach(stmt -> stmt.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.SENT)));

		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getExistingStatements(apsStmtInputL);
		assertNotNull(apsStmtRetrievedL);
		assertApsStmtListEquals(apsStmtInputL, apsStmtRetrievedL);
	}

	@Test
	public void batchUpdate() {
		testApsStmtL.forEach(stmt -> {
			stmt.setProcessDate(Date.from(Instant.now()));
			stmt.setStatementDate(Date.from(Instant.now()));
			stmt.setTotalAmountReceived(new AITCDollar(19292));
		});

		final int[] updated = apsStmtDao.batchUpdate(testApsStmtL);
		assertEquals(testApsStmtL.size(), updated.length);

		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getExistingStatements(testApsStmtL);
		assertNotNull(apsStmtRetrievedL);
		assertApsStmtListEquals(testApsStmtL, apsStmtRetrievedL);
	}

	@Test
	public void testUpdateStatusByGenProcessId() {
		final APSStmt apsStmtExpected = testApsStmtL.get(0);

		final Status newStatus = ProcessStatus.Status.REPLACED;

		final int updated = apsStmtDao.updateStatusByGenProcessId(apsStmtExpected.getGenBatchRunId(), ProcessStatus.Status.REPLACED);
		assertTrue(updated >= 1);

		final APSStmt apsStmtRetrieved = apsStmtDao.get(apsStmtExpected.getId());
		assertNotNull(apsStmtRetrieved);
		assertEquals(processStatusDao.getStatusFromEnum(newStatus).intValue(), apsStmtRetrieved.getStatusId());
	}

	@Test
	public void testUpdatePrintAckFields() {
		final BatchRunProcess expectedBatchRunProcess = testBatchRunProcessL.get(1);
		final int expectedStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.OTHER);

		apsStmtDao.updatePrintAckFields(testApsStmtL.stream().map(APSStmt::getId).collect(Collectors.toList()),
				expectedBatchRunProcess.getId(), Date.from(Instant.now()), expectedStatusId);

		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getExistingStatements(testApsStmtL);
		assertNotNull(apsStmtRetrievedL);		
		apsStmtRetrievedL.forEach(stmt -> {
			assertEquals(expectedBatchRunProcess.getId(), stmt.getAckBatchRunId());
			assertEquals(expectedStatusId, stmt.getStatusId());
		});
	}

	@Test
	public void testUndoPrintAcknowledgements() {

		final List<Long> batchRunProcessIdAckList = testBatchRunProcessL.stream().map(BatchRunProcess::getId).collect(Collectors.toList());
		final int sentStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.SENT);

		final int[] updated = apsStmtDao.undoPrintAcknowledgements(batchRunProcessIdAckList, sentStatusId);		
		assertEquals(updated.length, batchRunProcessIdAckList.size());

		// Only target the APS statements whose batchRunProcessIdAck is part of the testBatchRunProcessL.
		final List<APSStmt> apsStmtTargetL = testApsStmtL.stream()
				.filter(stmt -> batchRunProcessIdAckList.contains(stmt.getAckBatchRunId())).collect(Collectors.toList());
		// The undoPrintAcknowledgements() method does not update the model, and getExistingStatements() relies on the statusId field.
		apsStmtTargetL.forEach(stmt -> stmt.setStatusId(sentStatusId));
		
		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getExistingStatements(apsStmtTargetL);
		assertNotNull(apsStmtRetrievedL);
		assertEquals(apsStmtTargetL.size(), apsStmtRetrievedL.size());

		apsStmtRetrievedL.forEach(stmt -> {
			assertEquals(sentStatusId, stmt.getStatusId());
			assertEquals(0, stmt.getAckBatchRunId());
			assertNull(stmt.getPrintDate());
		});
	}
	
	@Test
	public void testGetStatementsForPrintAckDetails() {
		// getStatementsForPrintAck() checks for APSStmts in status SENT or ACK, so make sure all statements have
		// one of those statuses.
		testApsStmtL.forEach(stmt -> stmt.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.SENT)));
		final int[] update = apsStmtDao.updateStatusToSent(testApsStmtL, testBatchRunProcessL.get(0).getId());
		assertEquals(update.length, testApsStmtL.size());

		final List<APSStmt> apsStmtExpectedL = apsStmtDao.getExistingStatements(testApsStmtL);
		assertNotNull(apsStmtExpectedL);
		assertTrue(apsStmtExpectedL.size() > 0);
		
		final List<APPSADFileRecord> recordL = new ArrayList<APPSADFileRecord>();
		for(CBSAccount account : testCbsAccountL) {
			final APPSADFileRecord record = new APPSADFileRecord();
			record.setPatientAccount(account.getIcn());
			recordL.add(record);
		}
		
		final List<Long> apsStmtRetrievedL = apsStmtDao.getStatementIdsForPrintAck(recordL,
				APSStmtDAOImpl.getYear(testApsStmtL.get(0).getStatementDate()), 
				processStatusDao.getStatusFromEnum(ProcessStatus.Status.SENT), 
				processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK),
				new Date(0), -1); //Method only checks if not equal, so dummy values are fine
		assertNotNull(apsStmtRetrievedL);
		
		assertEquals(apsStmtExpectedL.size(), apsStmtRetrievedL.size());
	}
	
	@Test
	public final void testGetApsStmtsWithPaging() {
		final int pageSize = 3;
		final Status targetStatus = Status.NEW;
		final long expectedSize = testApsStmtL.stream()
				.filter(stmt -> stmt.getStatusId() == processStatusDao.getStatusFromEnum(targetStatus)).count();
		
		final List<APSStmt> apsStmtRetrievedL = apsStmtDao.getApsStmtsWithPaging(1, pageSize, Year.now().getValue(), Status.NEW);
		assertNotNull(apsStmtRetrievedL);		
		
		assertEquals(expectedSize, apsStmtRetrievedL.size());		
	}

	@Override
	protected void insertTestData() {
		// Insert CBS account
		final String icn1 = "icnForJUnitTest1";
		final CBSAccount cbsAccount1 = new CBSAccount();
		cbsAccount1.setIcn(icn1);
		cbsAccount1.setId(cbsAccountDao.save(icn1));
		testCbsAccountL.add(cbsAccount1);

		final String icn2 = "icnForJUnitTest2";
		final CBSAccount cbsAccount2 = new CBSAccount();
		cbsAccount2.setIcn(icn2);
		cbsAccount2.setId(cbsAccountDao.save(icn2));
		testCbsAccountL.add(cbsAccount2);

		final String icn3 = "icnForJUnitTest3";
		final CBSAccount cbsAccount3 = new CBSAccount();
		cbsAccount3.setIcn(icn3);
		cbsAccount3.setId(cbsAccountDao.save(icn3));
		testCbsAccountL.add(cbsAccount3);		

		// Insert batchRun
		final BatchRun batchRun = appsDaoImplUtils.createTestBatchRun();
		batchRun.setId(batchRunDao.save(batchRun));
		testBatchRunL.add(batchRun);

		// Insert Batch Run Processes
		final BatchRunProcess brpGen = appsDaoImplUtils.createTestBatchRunProcess(batchRun);
		brpGen.setId(batchRunProcessDao.save(brpGen));
		testBatchRunProcessL.add(brpGen);

		final BatchRunProcess brpSend = appsDaoImplUtils.createTestBatchRunProcess(batchRun);
		brpSend.setId(batchRunProcessDao.save(brpSend));
		testBatchRunProcessL.add(brpSend);

		final BatchRunProcess brpAck = appsDaoImplUtils.createTestBatchRunProcess(batchRun);
		brpAck.setId(batchRunProcessDao.save(brpAck));
		testBatchRunProcessL.add(brpAck);

		// Insert APSStmts
		final APSStmt apsStmt1 = appsDaoImplUtils.createTestAPSStmt(cbsAccount1);
		apsStmt1.setGenBatchRunId(brpGen.getId());
		apsStmt1.setSendBatchRunId(brpGen.getId());
		apsStmt1.setAckBatchRunId(brpGen.getId());
		apsStmt1.setId(apsStmtDao.save(apsStmt1));
		testApsStmtL.add(apsStmt1);		

		final APSStmt apsStmt2 = appsDaoImplUtils.createTestAPSStmt(cbsAccount2);
		apsStmt2.setGenBatchRunId(brpGen.getId());
		apsStmt2.setSendBatchRunId(brpGen.getId());
		apsStmt2.setAckBatchRunId(brpGen.getId());
		apsStmt2.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK));
		apsStmt2.setId(apsStmtDao.save(apsStmt2));
		testApsStmtL.add(apsStmt2);	

		final APSStmt apsStmt3 = appsDaoImplUtils.createTestAPSStmt(cbsAccount3);
		apsStmt3.setId(apsStmtDao.save(apsStmt3));
		testApsStmtL.add(apsStmt3);

		// Insert APSSiteStmts
		testApsStmtL.forEach(stmt -> {
			final APSSiteStmt apsSiteStmt = appsDaoImplUtils.createTestAPSSiteStmt(stmt);
			apsSiteStmt.setId(apsSiteStmtDao.save(apsSiteStmt));
			testApsSiteStmtL.add(apsSiteStmt);
		});
		
		// Insert APSPayments
		testApsSiteStmtL.forEach(siteStmt -> {
			final APSPayment apsPayment = appsDaoImplUtils.createTestAPSPayment(siteStmt);
			apsPayment.setId(apsPaymentDao.save(apsPayment));
			testApsPaymentL.add(apsPayment);
		});
	}

	private static void assertApsStmtEquals(final APSStmt expected, final APSStmt retrieved) {
		assertEquals(expected.getId(), retrieved.getId());
		assertEquals(expected.getAccountId(), retrieved.getAccountId());
		assertEquals(expected.getStatementDate().getTime(), retrieved.getStatementDate().getTime(), 10000);
		assertEquals(expected.getTotalAmountReceived().getDouble(), retrieved.getTotalAmountReceived().getDouble(), 0);
		assertEquals(expected.getProcessDate().getTime(), retrieved.getProcessDate().getTime(), 1000);
		assertEquals(expected.getGenBatchRunId(), retrieved.getGenBatchRunId());
		assertEquals(expected.getSendBatchRunId(), retrieved.getSendBatchRunId());
		assertEquals(expected.getAckBatchRunId(), retrieved.getAckBatchRunId());
		assertEquals(expected.getAccountNumDisp(), retrieved.getAccountNumDisp());
		assertEquals(expected.getReplacedStmtId(), retrieved.getReplacedStmtId());
		assertEquals(expected.getPrintDate().getTime(), retrieved.getPrintDate().getTime(), 1000);
		assertEquals(expected.getStatusId(), retrieved.getStatusId());
	}

	/**
	 * Asserts that two List<APSStmt> are the same size and contain exactly the same APSStmt IDs. This does not verify that the 
	 * individual APSStmt are equal to one another, so, for example, expected and retrieved may have different modifiedBy fields.
	 * @param expected
	 * @param retrieved
	 */
	private static void assertApsStmtListEquals(final List<APSStmt> expected, final List<APSStmt> retrieved) {
		assertEquals(expected.size(), retrieved.size());
		assertTrue(retrieved.stream().map(APSStmt::getId).collect(Collectors.toList())
				.containsAll(expected.stream().map(APSStmt::getId).collect(Collectors.toList())));
	}
}