package gov.va.cpss.cobol;

import static org.junit.Assert.assertTrue;

import java.util.Arrays;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import org.junit.Test;

/**
 *  MoneyTest Class
 * This MoneyTest class is used for unit test for Money conversion
 */
public class MoneyTest {

    private enum ConversionTest {

        POSITIVE_0("00000360{", "36.00", 36.0),
        POSITIVE_1("00000360A", "36.01", 36.01),
        POSITIVE_2("00000360B", "36.02", 36.02),
        POSITIVE_3("00000360C", "36.03", 36.03),
        POSITIVE_4("00000360D", "36.04", 36.04),
        POSITIVE_5("00000360E", "36.05", 36.05),
        POSITIVE_6("00000360F", "36.06", 36.06),
        POSITIVE_7("00000360G", "36.07", 36.07),
        POSITIVE_8("00000360H", "36.08", 36.08),
        POSITIVE_9("00000360I", "36.09", 36.09),
        NEGATIVE_0("00000360}", "-36.00", -36.0),
        NEGATIVE_1("00000360J", "-36.01", -36.01),
        NEGATIVE_2("00000360K", "-36.02", -36.02),
        NEGATIVE_3("00000360L", "-36.03", -36.03),
        NEGATIVE_4("00000360M", "-36.04", -36.04),
        NEGATIVE_5("00000360N", "-36.05", -36.05),
        NEGATIVE_6("00000360O", "-36.06", -36.06),
        NEGATIVE_7("00000360P", "-36.07", -36.07),
        NEGATIVE_8("00000360Q", "-36.08", -36.08),
        NEGATIVE_9("00000360R", "-36.09", -36.09);

        private String cobol;
        private String valueS;
        private double valueD;

        public String getCobol() {
            return cobol;
        }

        public String getValueS() {
            return valueS;
        }

        public double getValueD() {
            return valueD;
        }

        private ConversionTest(String cobol, String valueS, double valueD) {
            this.cobol = cobol;
            this.valueS = valueS;
            this.valueD = valueD;
        }
    }

    final List<ConversionTest> enumL = Arrays.asList(ConversionTest.values());

    @Test
    public final void testGetDouble() {

        for (ConversionTest t : enumL) {
            final Money m = new Money(t.getCobol());
            assertEquals(t.getValueS(), m.getDoubleAsString());
        }
    }

    @Test
    public final void testGetCobol9() {

        for (ConversionTest t : enumL) {
            final Money m = new Money(t.getValueD(), 9);
            assertEquals(t.getCobol(), m.getCobol());
        }
    }

    @Test
    public final void testGetCobol11() {

        for (ConversionTest t : enumL) {
            final Money m = new Money(t.getValueD(), 11);
            assertEquals("00" + t.getCobol(), m.getCobol());
        }
    }

    @Test
    public final void testMaxValuesCobol11() {

            final Money maxM = new Money(999_999_999.99d, 11);
            assertEquals("9999999999I", maxM.getCobol());
            
            final Money minM = new Money(-999_999_999.99d, 11);
            assertEquals("9999999999R", minM.getCobol());
    }
    
    @Test
    public final void testMaxGetDouble() {
        final Money maxM = new Money("9999999999I");
        assertEquals("999999999.99", maxM.getDoubleAsString());
    	
        final Money minM = new Money("9999999999R");
        assertEquals("-999999999.99", minM.getDoubleAsString());
    }

    @Test
    public final void testBadCobolFormat() {
        try {
            new Money("00000360Z");
            fail("Invalid format did not throw exception.");
        } catch (NumberFormatException e) {
            assertTrue(e.getMessage().contains("Unexpected Overpunched Cobol Format"));
        }
    }

    @Test
    public final void testTooRestrictiveDoubleFormat() {
        try {
            new Money(1234567.00, 2);
            fail("Size too small format did not throw exception.");
        } catch (IndexOutOfBoundsException e) {
            assertTrue(e.getMessage().contains("Cobol format requires larger size than specified"));
        }
    }
    
    @Test
    public final void testInvalidNumberFormat() {
        try {
            new Money("0XYZ0360L");
            fail("Invalid number format did not throw exception.");
        } catch (NumberFormatException e) {
            assertTrue(e.getMessage().contains("For input string"));
        }
    }
    
}
