package gov.va.cpss.service.impl.vfs;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.VFS;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;
import org.apache.commons.vfs2.impl.StandardFileSystemManager;
import org.apache.log4j.Logger;

/**
 * Provides a factory and pool of FileSystemManager objects that ensures a
 * unique instance of the Apache VFS FileSystemManager for each Thread, since
 * using the singleton returned by VFS.getFileSystemManager() may not be
 * thread-safe.
 * 
 * A client invokes the get() method to get an instance of the
 * FileSystemManager. The instance returned will depend upon the given Thread
 * that invokes the get() method. The factory will instantiate one
 * FileSystemManager per Thread, and always return that instance for all calls
 * to get() by that Thread.
 * 
 * A ThreadLocal<Long> is used to identify a Thread and determine which
 * FileSystemManager to provide to the Thread. This potentially introduces a
 * memory leak within the context of Application Servers like WebLogic. The
 * application server creates and manages the Thread life cycles and allocation
 * of the Threads to the application, and the Threads will typically outlive the
 * Application if the application is redeployed without the Application Server
 * restarting.
 * 
 * Subsequent deployments of the Application will be within the context of a
 * different ClassLoader and a new ThreadLocal<Long>. The current implementation
 * of ThreadLocal uses a protected ThreadLocalMap member of Thread to store the
 * Long instance for the given Thread. The entries in ThreadLocalMap for prior
 * application deployments will remain, because the application does not have
 * control of the Threads when it is cleaning up and therefore cannot force the
 * Threads to call remove() on the ThreadLocal<Long> to remove the entries in
 * the ThreadLocalMap.
 * 
 * A work around would be to save a reference to all Threads that invoke get(),
 * then on clean up use Reflection to access the protected ThreadLocalMap to
 * remove the Long reference. This would be a brittle solution that relies on
 * the current implementation of ThreadLocal, which could break with newer
 * versions of Java. I chose not to do this with this implementation, and chose
 * to use just a Long to minimize the amount of memory required for the stale
 * ThreadLocalMap references and also to avoid leaving references to application
 * specific classes that would force the application's ClassLoader to stay
 * loaded by the lingering ThreadLocalMap reference which would be a bigger
 * problem. The assumption is that even if the Application Server were to not be
 * restarted for years, this memory leak will be insignificant.
 * 
 * @author Brad Pickle
 *
 */
public class FileSystemManagerFactory {

	private static final Logger logger = Logger.getLogger(FileSystemManagerFactory.class.getCanonicalName());

	/**
	 * A unique id attached to each Thread to uniquely identify it.
	 */
	private ThreadLocal<Long> threadLocalFileSystemManagerId = new ThreadLocal<Long>() {
		private AtomicLong nextId = new AtomicLong();

		@Override
		protected Long initialValue() {
			return nextId.getAndIncrement();
		}
	};

	/**
	 * Map of all Proxied FileSystemManager instances instantiated by this
	 * factory.
	 */
	private Map<Long, ProxyFileSystemManager<DefaultFileSystemManager>> fileSystemManagerMap = new ConcurrentHashMap<Long, ProxyFileSystemManager<DefaultFileSystemManager>>();

	/**
	 * Returns a FileSystemManager. The FileSystemManager will be unique for
	 * each Thread. A given Thread will always receive the same
	 * FileSystemManager instance on multiple calls to get().
	 * 
	 * @return A unique FileSystemManager for the given calling Thread.
	 */
	public FileSystemManager get() {
		final long fileSystemManagerId = threadLocalFileSystemManagerId.get();

		ProxyFileSystemManager<DefaultFileSystemManager> fsm = fileSystemManagerMap.get(fileSystemManagerId);
		if (fsm == null) {
			fsm = new ProxyFileSystemManager<DefaultFileSystemManager>(createFileSystemManager());
			fileSystemManagerMap.put(fileSystemManagerId, fsm);
		} else if (fsm.getFileSystemManager() == null) {
			fsm.setFileSystemManager(createFileSystemManager());
		}

		return fsm;
	}

	/**
	 * Close all FileSystemManager instances managed by this factory.
	 */
	public void closeAllFileSystemManagers() {
		final List<ProxyFileSystemManager<DefaultFileSystemManager>> currentFSMList = new ArrayList<ProxyFileSystemManager<DefaultFileSystemManager>>(
				fileSystemManagerMap.size());

		synchronized (fileSystemManagerMap) {
			currentFSMList.addAll(fileSystemManagerMap.values());
			fileSystemManagerMap.clear();
		}

		for (ProxyFileSystemManager<DefaultFileSystemManager> fsm : currentFSMList) {
			try {
				if (fsm != null) {
					fsm.close();
				}
			} catch (Exception e) {
				logger.error("Error closing file system manager:", e);
			}
		}
	}

	/**
	 * Creates a new and initialized FileSystemManager.
	 * 
	 * @return A new FileSystemManager.
	 */
	private DefaultFileSystemManager createFileSystemManager() {
		StandardFileSystemManager manager = new StandardFileSystemManager();

		manager.setLogger(LogFactory.getLog(VFS.class));

		try {
			manager.init();
		} catch (FileSystemException e) {
			logger.fatal("Could not initialize thread-local FileSystemManager.", e);
		}

		return manager;
	}

}
