package gov.va.cpss.service.impl.vfs;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.Selectors;
import org.apache.commons.vfs2.provider.sftp.SftpFileSystemConfigBuilder;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Service;

import com.jcraft.jsch.JSch;

import gov.va.cpss.service.FileData;
import gov.va.cpss.service.SftpConnectionProperties;
import gov.va.cpss.service.SftpService;
import gov.va.cpss.service.SftpStreamSession;

/**
 * Implementation of SftpService using Apache VFS.
 * 
 * @author Brad Pickle
 */
@Service
public class ApacheVfsSftpServiceImpl implements SftpService, InitializingBean, DisposableBean {
	
	// Pass com.craft.jsch log messages to log4J
	{
		JSch.setLogger(new JschLogger());
	}

	/** The Constant FILE_URL_PREFIX. */
	public static final String FILE_URL_PREFIX = "file" + SftpConnectionProperties.COLON;

	public static final String CURRENT_DIR_INDICATOR = ".";

	private static final Logger logger = Logger.getLogger(ApacheVfsSftpServiceImpl.class.getCanonicalName());

	private SftpConnectionProperties sftpConnectionProperties;

	private FileSystemManagerFactory fsmFactory;

	public SftpConnectionProperties getSftpConnectionProperties() {
		return sftpConnectionProperties;
	}

	public void setSftpConnectionProperties(SftpConnectionProperties sftpConnectionProperties) {
		this.sftpConnectionProperties = sftpConnectionProperties;
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		fsmFactory = new FileSystemManagerFactory();
	}

	@Override
	public void destroy() throws Exception {
		fsmFactory.closeAllFileSystemManagers();
		fsmFactory = null;
	}

	private FileSystemManager getFSM() {
		return fsmFactory.get();
	}

	private FileSystemOptions createFSOptions() throws FileSystemException {
		logger.debug("Method: createFileSystemOptions Message: Attempting  to create FileSystemOptions.");
		FileSystemOptions fsOptions = new FileSystemOptions();
		try {
			SftpFileSystemConfigBuilder.getInstance().setStrictHostKeyChecking(fsOptions, "no");
			SftpFileSystemConfigBuilder.getInstance().setUserDirIsRoot(fsOptions, true);
			SftpFileSystemConfigBuilder.getInstance().setPreferredAuthentications(fsOptions, "publickey,password");

			final String privateKey = sftpConnectionProperties.getPrivateKey();
			
			if ((privateKey != null) && !privateKey.trim().equals("")) {
				SftpFileSystemConfigBuilder.getInstance().setIdentities(fsOptions, new File[] { new File(privateKey) });
			}
		} catch (FileSystemException e) {
			logger.debug("Method: createFileSystemOptions Message: Failed to create FileSystemOptions", e);
		}

		logger.debug("Method: createFileSystemOptions Message: Created FileSystemOptions.");
		return fsOptions;
	}

	private String getRemoteFilePath(String targetFileName, String targetDirectory) {
		final String targetDirectoryNotEmpty = ((targetDirectory == null) || (targetDirectory.length() == 0))
				? CURRENT_DIR_INDICATOR : targetDirectory;

		final String separatorSlash = (targetDirectoryNotEmpty.endsWith(SftpConnectionProperties.SLASH) ? ""
				: SftpConnectionProperties.SLASH);

		return targetDirectoryNotEmpty + separatorSlash + targetFileName;
	}

	private String getUriForRemoteFilePath(String filepath) {
		final String filePathNotNull = ((filepath == null) || (filepath.length() == 0)) ? SftpConnectionProperties.SLASH
				: filepath;

		final String separatorSlash = filePathNotNull.startsWith(SftpConnectionProperties.SLASH) ? ""
				: SftpConnectionProperties.SLASH;

		return sftpConnectionProperties.getEncodedBaseUri() + separatorSlash + filePathNotNull;
	}

	private String getUriForLocalFilePath(String filepath) {
		return FILE_URL_PREFIX + (((filepath == null) || (filepath.length() == 0)) ? CURRENT_DIR_INDICATOR : filepath);
	}

	/**
	 * Transfer the specified file path to the default directory on the server.
	 * If the filepath is to a file the file will be sent to the default
	 * directory on the server. IF the filepath is a directory then all files
	 * will be sent to the default directory on the server.
	 * 
	 * @param filepath
	 *            The path of the file or directory to send to the server.
	 * 
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpFileToServer(final String filepath) {
		return ftpFileToServer(filepath, ".");
	}

	/**
	 * Transfer the specified file path to the specified directory on the
	 * server. If the filepath is to a file the file will be sent to the default
	 * directory on the server. IF the filepath is a directory then all files
	 * will be sent to the default directory on the server.
	 * 
	 * @param filepath
	 *            The path of the file or directory to send to the server.
	 * @param targetDirectory
	 *            The relative target directory on the server.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpFileToServer(final String filepath, final String targetDirectory) {

		boolean successful = false;

		try {
			final File localFile = new File(filepath);
			if (!localFile.exists())
				throw new RuntimeException("Local file does not exists:" + filepath);

			successful = ftpFileToServerWithName(filepath, localFile.getName(), targetDirectory);

		} catch (Exception e) {
			logger.error("Unexpected exception copying file to sFTP destination: ", e);
		}

		return successful;
	}

	/**
	 * Send the specified local filepath to the server as the specified target
	 * filename in the target directory on the server.
	 * 
	 * @param filepath
	 *            The local filepath to send.
	 * @param targetFilename
	 *            The target filename on the server.
	 * @param targetDirectory
	 *            The target directory for the target filename on the server.
	 * @return Boolean flag indicating if successful or not.
	 */
	@Override
	public boolean ftpFileToServerWithName(final String filepath, final String targetFilename,
			final String targetDirectory) {

		boolean successful = false;

		FileObject localFileObject = null;
		FileObject remoteFileObject = null;

		try {
			final File localFile = new File(filepath);
			if (!localFile.exists())
				throw new RuntimeException("Local file does not exists:" + filepath);

			final String remoteFilePath = getRemoteFilePath(targetFilename, targetDirectory);

			localFileObject = getFSM().resolveFile(getUriForLocalFilePath(localFile.getAbsolutePath()));

			remoteFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteFilePath), createFSOptions());

			remoteFileObject.copyFrom(localFileObject, Selectors.SELECT_ALL);

			successful = true;

		} catch (Exception e) {
			logger.error("Unexpected exception copying file to sFTP destination: ", e);
		} finally {
			if (localFileObject != null) {
				try {
					localFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing local file object.", e);
				}
			}
			if (remoteFileObject != null) {
				try {
					remoteFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
		}

		return successful;
	}

	/**
	 * Send an empty file to the sftp server in the specified target directory
	 * with the specified name.
	 * 
	 * @param targetFilename
	 *            The name of the resulting empty file on the sftp server.
	 * @param targetDirectory
	 *            The target directory on the sftp server.
	 * @return Boolean flag indicating successful or not.
	 */
	@Override
	public boolean ftpEmptyFileToServerWithName(final String targetFilename, final String targetDirectory) {
		return ftpStringAsFileToServerWithName("", targetFilename, targetDirectory);
	}

	/**
	 * Send the specified string content as the specified file content to the
	 * sftp server int he specified target directory with the specified name.
	 * 
	 * @param content
	 *            The content of the target file.
	 * @param targetFilename
	 *            The name of the resulting file on the sftp server.
	 * @param targetDirectory
	 *            The target directory on the sftp server.
	 * @return Boolean flag indicating successful or not.
	 */
	@Override
	public boolean ftpStringAsFileToServerWithName(final String content, final String targetFilename,
			final String targetDirectory) {

		boolean successful = false;

		FileObject remoteFileObject = null;

		try {
			final String remoteFilePath = getRemoteFilePath(targetFilename, targetDirectory);

			remoteFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteFilePath), createFSOptions());

			OutputStream remoteFileOS = remoteFileObject.getContent().getOutputStream();

			remoteFileOS.write(content.getBytes());

			successful = true;

		} catch (Exception e) {
			logger.error("Unexpected exception copying file to sFTP destination: ", e);
		} finally {
			if (remoteFileObject != null) {
				try {
					remoteFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
		}

		return successful;
	}

	/**
	 * Remove the specified filename from the specified target directory on the
	 * server.
	 * 
	 * @param filename
	 *            The name of the file to remove.
	 * @param targetDirectory
	 *            The directory path where the file should reside.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpRemoveFileFromDirectory(final String filename, final String targetDirectory) {

		boolean successful = false;

		FileObject remoteFileObject = null;

		try {
			final String remoteFilePath = getRemoteFilePath(filename, targetDirectory);

			remoteFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteFilePath), createFSOptions());

			if (remoteFileObject.exists()) {
				remoteFileObject.delete();
			}

			successful = true;

		} catch (Exception e) {
			logger.error("Unexpected exception copying file to sFTP destination: ", e);
		} finally {
			if (remoteFileObject != null) {
				try {
					remoteFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
		}

		return successful;
	}

	/**
	 * Move the specified filename to the specified target directory.
	 * 
	 * @param originalFilename
	 *            The name of the file to move.
	 * @param targetDirectory
	 *            The directory path to move the file.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpMoveFileToDirectory(final String originalFilename, final String targetDirectory) {

		return ftpMoveFileFromDirectoryToDirectory(originalFilename, CURRENT_DIR_INDICATOR, originalFilename,
				targetDirectory);
	}

	/**
	 * Move the specified filename to the specified target directory.
	 * 
	 * @param originalFilename
	 *            The name of the file to move.
	 * @param originalDirectory
	 *            The directory where the file to move is originally located.
	 * @param targetDirectory
	 *            The directory path to move the file.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpMoveFileFromDirectoryToDirectory(final String originalFilename, final String originalDirectory,
			final String targetDirectory) {

		return ftpMoveFileFromDirectoryToDirectory(originalFilename, originalDirectory, originalFilename,
				targetDirectory);
	}

	/**
	 * Rename the specified filename with the new filename in the specified
	 * target directory on the server.
	 * 
	 * @param originalFilename
	 *            The original filename on the server.
	 * @param newFilename
	 *            The new filename on the server.
	 * @param targetDirectory
	 *            The target directory where the files reside on the server.
	 * @return Boolean flag indicating successful rename or not.
	 */
	@Override
	public boolean ftpRenameFileInDirectory(final String originalFilename, final String newFilename,
			final String targetDirectory) {

		return ftpMoveFileFromDirectoryToDirectory(originalFilename, targetDirectory, newFilename, targetDirectory);
	}

	/**
	 * Move the specified filename to the specified target directory and rename
	 * to the specified name.
	 * 
	 * @param originalFilename
	 *            The name of the file to move.
	 * @param originalDirectory
	 *            The directory where the file to move is originally located.
	 * @param newFilename
	 *            The new filename on the server.
	 * @param targetDirectory
	 *            The directory path to move the file.
	 * @return Boolean flag indicating success or failure.
	 */
	private boolean ftpMoveFileFromDirectoryToDirectory(String originalFilename, String originalDirectory,
			String newFilename, String targetDirectory) {

		boolean successful = false;

		FileObject remoteOriginalFileObject = null;
		FileObject remoteTargetFileObject = null;

		try {
			final String remoteOriginalFilePath = getRemoteFilePath(originalFilename, originalDirectory);
			final String remoteTargetFilePath = getRemoteFilePath(newFilename, targetDirectory);

			remoteOriginalFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteOriginalFilePath),
					createFSOptions());
			remoteTargetFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteTargetFilePath),
					createFSOptions());

			if (!remoteOriginalFileObject.exists())
				throw new RuntimeException("Remote file does not exists:" + originalFilename);

			remoteOriginalFileObject.moveTo(remoteTargetFileObject);

			successful = true;

		} catch (Exception e) {
			logger.error("Unexpected exception moving file on sFTP server: ", e);
		} finally {
			if (remoteOriginalFileObject != null) {
				try {
					remoteOriginalFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
			if (remoteTargetFileObject != null) {
				try {
					remoteTargetFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
		}

		return successful;
	}

	/**
	 * Get a list of files that reside in the specified directory path.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @return List of file names.
	 */
	@Override
	public List<String> ftpGetFileListInDirectory(final String targetDirectory) {
		return ftpGetFileListInDirectory(targetDirectory, false);
	}

	/**
	 * Get a list of files that reside in the specified directory path.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @param throwExceptions
	 *            If true, any exceptions will be re-thrown after being logged.
	 * @return List of file names.
	 * @throws Exception
	 */
	@Override
	public List<String> ftpGetFileListInDirectory(final String targetDirectory, boolean throwExceptions) {

		List<String> fileL = null;
		FileObject[] children = null;

		FileObject remoteTargetDirectoryObject = null;

		try {
			remoteTargetDirectoryObject = getFSM().resolveFile(getUriForRemoteFilePath(targetDirectory),
					createFSOptions());

			children = remoteTargetDirectoryObject.getChildren();

			if (children == null) {
				fileL = new ArrayList<String>(0);
			} else {
				fileL = new ArrayList<String>(children.length);
				for (FileObject child : children) {
					fileL.add(child.getName().getBaseName());
				}
			}

		} catch (Exception e) {
			logger.error("Unexpected exception listing directory on sFTP server: ", e);
			if (throwExceptions)
				throw new RuntimeException(e);
		} finally {
			if (remoteTargetDirectoryObject != null) {
				try {
					remoteTargetDirectoryObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
			if (children != null) {
				for (FileObject child : children) {
					if (child != null) {
						try {
							child.close();
						} catch (Exception e) {
							logger.error("Error closing remote file object.", e);
						}
					}
				}
			}
		}

		return fileL;
	}

	/**
	 * Get a list of files that reside in the specified directory path that have
	 * the specified extension. The extension check will ignore case.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public List<String> ftpGetFileListWithExtensionInDirectory(final String targetDirectory, final String extension) {
		return ftpGetFileListWithExtensionInDirectory(targetDirectory, extension, false);
	}

	/**
	 * Get a list of files that reside in the specified directory path that have
	 * the specified extension. The extension check will ignore case.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @param throwExceptions
	 *            If true, any exceptions will be re-thrown after being logged.
	 * @return List of file names sorted by their modified dates
	 */
	@Override
	public List<String> ftpGetFileListWithExtensionInDirectory(final String targetDirectory, final String extension,
			boolean throwExceptions) {

		List<String> fileL = null;

		FileObject remoteTargetDirectoryObject = null;

		FileObject[] children = null;

		try {
			remoteTargetDirectoryObject = getFSM().resolveFile(getUriForRemoteFilePath(targetDirectory),
					createFSOptions());

			children = remoteTargetDirectoryObject.getChildren();

			if (children == null) {
				fileL = new ArrayList<String>(0);
			} else {
				final List<FileData> dataList = new ArrayList<FileData>(children.length);
				final String extensionUpper = extension.toUpperCase();
				for (FileObject child : children) {
					final String childFilename = child.getName().getBaseName();
					if (childFilename.toUpperCase().endsWith(extensionUpper)) {
						dataList.add(new FileData(childFilename, child.getContent().getLastModifiedTime()));
					}
				}

				// Sort the list by date modified
				Collections.sort(dataList);

				fileL = new ArrayList<String>(dataList.size());
				for (FileData fileData : dataList) {
					fileL.add(fileData.getFileName());
				}
			}

		} catch (Exception e) {
			logger.error("Unexpected exception listing directory on sFTP server: ", e);
			if (throwExceptions)
				throw new RuntimeException(e);
		} finally {
			if (remoteTargetDirectoryObject != null) {
				try {
					remoteTargetDirectoryObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
			if (children != null) {
				for (FileObject child : children) {
					if (child != null) {
						try {
							child.close();
						} catch (Exception e) {
							logger.error("Error closing remote file object.", e);
						}
					}
				}
			}
		}

		return fileL;
	}

	/**
	 * Check if the specified filename exists in the target directory.
	 * 
	 * @param filename
	 *            The name of the file to check existence.
	 * @param targetDirectory
	 *            The target directory path to check for the specified file.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpFileExistsInDirectory(final String filename, final String targetDirectory) {
		return ftpFileExistsInDirectory(filename, targetDirectory, false);
	}

	/**
	 * Check if the specified filename exists in the target directory.
	 * 
	 * @param filename
	 *            The name of the file to check existence.
	 * @param targetDirectory
	 *            The target directory path to check for the specified file.
	 * @param throwExceptions
	 *            If true, any exceptions will be re-thrown after being logged.
	 * @return Boolean flag indicating success or failure.
	 */
	@Override
	public boolean ftpFileExistsInDirectory(final String filename, final String targetDirectory,
			boolean throwExceptions) {

		boolean successful = false;

		List<String> fileL = ftpGetFileListInDirectory(targetDirectory, throwExceptions);

		if (fileL != null) {
			successful = fileL.contains(filename);
		}

		return successful;
	}

	/**
	 * Get the size of the specified file or null if does not exist.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @return Long Size of the file or null if does not exist.
	 */
	@Override
	public Long ftpGetFileSizeInDirectory(final String filename, final String targetDirectory) {

		Long size = null;

		FileObject remoteFileObject = null;

		try {
			final String remoteFilePath = getRemoteFilePath(filename, targetDirectory);

			remoteFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteFilePath), createFSOptions());

			if (remoteFileObject.exists()) {
				size = remoteFileObject.getContent().getSize();
			}

		} catch (Exception e) {
			logger.error("Unexpected exception getting file size from sFTP destination: ", e);
		} finally {
			if (remoteFileObject != null) {
				try {
					remoteFileObject.close();
				} catch (Exception e) {
					logger.error("Error closing remote file object.", e);
				}
			}
		}

		return size;

	}

	/**
	 * Open a stream for the specified file.
	 * 
	 * @param filename
	 *            The file to open.
	 * @return SftpStreamSession bean that contains the session and input stream
	 *         or null if error.
	 * @throws Exception
	 */
	@Override
	public SftpStreamSession openFileStream(final String filename, final String directory) throws Exception {

		boolean successful = false;

		SftpStreamSession sss = null;

		FileObject remoteFileObject = null;

		InputStream remoteFileInputStream = null;

		try {
			final String remoteFilePath = getRemoteFilePath(filename, directory);

			remoteFileObject = getFSM().resolveFile(getUriForRemoteFilePath(remoteFilePath), createFSOptions());

			if (remoteFileObject.exists()) {
				sss = new SftpStreamSession(new ApacheVfsSftpSessionImpl(remoteFileObject),
						remoteFileObject.getContent().getInputStream());
				successful = true;
			}

		} catch (Exception e) {
			logger.error("Unexpected exception reading file from sFTP destination: ", e);
		} finally {
			if (!successful) {
				if (remoteFileInputStream != null) {
					try {
						remoteFileInputStream.close();
					} catch (Exception e) {
						logger.error("Error closing remote file input stream.", e);
					}
				}
				if (remoteFileObject != null) {
					try {
						remoteFileObject.close();
					} catch (Exception e) {
						logger.error("Error closing remote file object.", e);
					}
				}
			}
		}

		return sss;
	}

}
