package gov.va.cpss.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.ArrayUtils;

import org.apache.log4j.Logger;

import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.CBSSitePatientDAO;
import gov.va.cpss.dao.VistaAccountDAO;
import gov.va.cpss.dao.VistaAcntHistDAO;

import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.DfnStationNumPair;
import gov.va.cpss.model.cbs.CBSAccount;
import gov.va.cpss.model.icn.VistaAccount;
import gov.va.cpss.model.icn.VistaAcntHist;

//import static gov.va.cpss.Constants.*;

/**
 * 
 * Business Service Implementation to Update ICN Across CBSS
 * 
 * (Sprint 2 - Story 325577 Update ICNs / Task 325578 Business service
 * implementation to update ICN across CBSS.)
 * 
 * This service can take in a collection of ICN Histories, and find new and old
 * ICN's, so to update ICN across CBSS; or, if there is no input ICN History, it
 * will get ICN History from the database, and find new and old ICN's to update.
 * 
 * Copyright HPE / VA June 6, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 *
 *
 *          Sprint 5 - Epic 349068, Task 370266 Implement ICN Update Service -
 *          Update based on new design.
 *
 *          This service can take in a collection of VistA Account Histories,
 *          and find new and old ICN's, so to update ICN across CBSS; or, if
 *          there is no input VistA Account History, it will get VistA Account
 *          History from the database, and find new and old ICN's to update.
 *
 *          Copyright HPE / VA August 26, 2016
 * 
 * @author Yiping Yao
 * @version 2.0.0
 *
 *          12/2/2016 - Brad: - Modified to allow oldVistaAcntNum to have null
 *          distinct from 0 - Added support for multiple updates to the same
 *          VistaAccount within the same batch run. - Changed various internal
 *          Lists/Arrays to Maps and Sets to improve performance.
 */
@SuppressWarnings("nls")
public class UpdateICNService
{
	/**
	 * Constants
	 * 
	 */
	// Messages
	public static final String MESSAGE_NULL_OR_EMPTY_ICNS = "The new and/or old ICN's are null or empty.";
	public static final String MESSAGE_SAME_ICNS = "The new ICN's are the same as the old ICN's";
	public static final String MESSAGE_NOT_FINDING_NEW_ICNS = "Did not find any new ICN's to update.";
	public static final String MESSAGE_NOT_FINDING_ALL_ACCOUNTS = "Did not find all accounts for the given ICN's.";
	public static final String MESSAGE_NOT_CREATING_ALL_ACCOUNTS = "Did not create all new accounts for the given ICN's.";
	public static final String MESSAGE_NOT_UPDAING_ALL_ACCOUNTS = "Did not update all accounts for the given ICN's.";
	// public static final String MESSAGE_NOT_UPDAING_ALL_SITE_PATIENTS = "Did not update all site patients for the given ICN's.";

	// Private Logger
	private static final Logger logger = Logger.getLogger(UpdateICNService.class.getCanonicalName());

	// DAO's
	private CBSAccountDAO cbsAccountDAO;
	private VistaAccountDAO vistaAccountDAO;
	private VistaAcntHistDAO vistaAcntHistDAO;
	private CBSSitePatientDAO cbsSitePatientDAO;

	/**
	 * @return the cbsAccountDAO
	 */
	public CBSAccountDAO getCbsAccountDAO()
	{
		return this.cbsAccountDAO;
	}

	/**
	 * @param inCBSAccountDAO
	 *            the cbsAccountDAO to set
	 */
	public void setCbsAccountDAO(CBSAccountDAO inCBSAccountDAO)
	{
		this.cbsAccountDAO = inCBSAccountDAO;
	}

	/**
	 * @return the vistaAccountDAO
	 */
	public VistaAccountDAO getVistaAccountDAO()
	{
		return this.vistaAccountDAO;
	}

	/**
	 * @param inVistaAccountDAO
	 *            the vistaAccountDAO to set
	 */
	public void setVistaAccountDAO(VistaAccountDAO inVistaAccountDAO)
	{
		this.vistaAccountDAO = inVistaAccountDAO;
	}

	/**
	 * @return the vistaAcntHistDAO
	 */
	public VistaAcntHistDAO getVistaAcntHistDAO()
	{
		return this.vistaAcntHistDAO;
	}

	/**
	 * @param inVistaAcntHistDAO
	 */
	public void setVistaAcntHistDAO(VistaAcntHistDAO inVistaAcntHistDAO)
	{
		this.vistaAcntHistDAO = inVistaAcntHistDAO;
	}

	/**
	 * @return the cbsSitePatientDAO
	 */
	public CBSSitePatientDAO getCbsSitePatientDAO()
	{
		return this.cbsSitePatientDAO;
	}

	/**
	 * @param inCBSSitePatientDAO
	 *            the cbsSitePatientDAO to set
	 */
	public void setCbsSitePatientDAO(CBSSitePatientDAO inCBSSitePatientDAO)
	{
		this.cbsSitePatientDAO = inCBSSitePatientDAO;
	}

	/**
	 * The service implementation of updating ICN's.
	 * 
	 * Note:
	 * 
	 * The service returns a collection of numbers of updated rows for each new
	 * ICN. But due to the limitation of the current Oracle JDBC drive, it only
	 * returns a value of -2, which means Statement.SUCCESS_NO_INFO.
	 * 
	 * @param inVistaAcntHistories
	 * @return Result
	 */
	@SuppressWarnings("null")
    public Result<Integer[]> updateICN(List<VistaAcntHist> inVistaAcntHistories)
	{
		logger.info("Update ICN Service starts.");

		Result<Integer[]> result = new Result<>();
		int[] numberOfUpdates = null;
		VistaAcntHist vistaAcntHistory = null;
		List<VistaAcntHist> vistaAcntHistories = null;
		List<VistaAcntHist> noprocessVistaAcntHistories = null;
		// long[] ids = null;
		// String[] oldICNs = null;
		// String[] newICNs = null;
		Set<String> ICNs = null;

		result.setSuccessful(false);

		try
		{
			// Step 1 - Retrieve new VistA Account History:
			// If there is no input VistA Account History,
			// getting new ICN's from VistA Account History in DB.
			if (inVistaAcntHistories == null || inVistaAcntHistories.isEmpty())
			{
				vistaAcntHistories = this.vistaAcntHistDAO.select(Status.NEW);
			}
			else
			{
				vistaAcntHistories = inVistaAcntHistories;
			}

			if (vistaAcntHistories != null && !vistaAcntHistories.isEmpty())
			{
				logger.debug("Number of new ICN's before purge: " + vistaAcntHistories.size());

				noprocessVistaAcntHistories = new ArrayList<>();

				// Remove the VistA Account History that has the same old ICN
				// and new ICN, and add to the unprocessed VistA Account History
				// List to process later.
				for (Iterator<VistaAcntHist> iterator = vistaAcntHistories.iterator(); iterator.hasNext();)
				{
					vistaAcntHistory = iterator.next();

					if ( vistaAcntHistory.getNewIcn().equals(vistaAcntHistory.getOldIcn()) ||
					     Status.NEW != vistaAcntHistory.getStatus() )
					{
						noprocessVistaAcntHistories.add(vistaAcntHistory);
						iterator.remove();
					}
				}

				logger.debug("Number of new ICN's after purge: " + vistaAcntHistories.size());
				logger.debug("Number of unprocessed ICN's after purge: " + noprocessVistaAcntHistories.size());

				// After removing the same new and old ICN's,
				// if there is nothing left in the VistA Account History
				// collection, stop; otherwise populate (the ID's,) new ICN's,
				// and old ICN's arrays.
				// Combined old and new ICN's into one Set, since oldICNs was no
				// longer being used and algorithm changed to also create
				// missing old ICNs. - Brad 1/4/2017
				if (!vistaAcntHistories.isEmpty())
				{
					// ids = new long[vistaAcntHistories.size()];
					// newICNs = new String[vistaAcntHistories.size()];
					// oldICNs = new String[vistaAcntHistories.size()];
					//
					// for (int i = 0; i < vistaAcntHistories.size(); i++) {
					// ids[i] = vistaAcntHistories.get(i).getId();
					// newICNs[i] = vistaAcntHistories.get(i).getNewIcn();
					// oldICNs[i] = vistaAcntHistories.get(i).getOldIcn();
					// }
					ICNs = new HashSet<>();

					for (VistaAcntHist vistaAcntHist : vistaAcntHistories)
					{
						addICN(ICNs, vistaAcntHist.getNewIcn());
						addICN(ICNs, vistaAcntHist.getOldIcn());
					}

					// Also creating accounts for the non-process rows.
					for (VistaAcntHist vistaAcntHist : noprocessVistaAcntHistories)
					{
						// new and old are identical
						// addICN(ICNs, vistaAcntHist.getNewIcn());
						addICN(ICNs, vistaAcntHist.getOldIcn());
					}

					result.setSuccessful(true);
				}
				else
				{
					result.setSuccessful(false);
					result.setMessage(MESSAGE_NOT_FINDING_NEW_ICNS);
				}
			}
			else
			{
				result.setSuccessful(false);
				result.setMessage(MESSAGE_NOT_FINDING_NEW_ICNS);
			}

			if (result.isSuccessful() && (ICNs != null))
			{
				result.setSuccessful(false);

				// Step 2 - Check if ICN's are in CBS Accounts,
				// if not, create and insert new CBS Account.
				Map<String, CBSAccount> allCBSAccounts = null;

				List<CBSAccount> existingCBSAccounts = this.cbsAccountDAO.batchSelect(ICNs.toArray(new String[0]));

				if (existingCBSAccounts == null || existingCBSAccounts.isEmpty())
				{
					// All CBS Accounts
					allCBSAccounts = insertAndCreateCBSAccounts(ICNs, (Map<String, CBSAccount>) null);

					result.setSuccessful(true);

					if (allCBSAccounts == null || allCBSAccounts.isEmpty())
					{
						result.setSuccessful(false);
						result.setMessage(MESSAGE_NOT_CREATING_ALL_ACCOUNTS);
					}
				}
				else
				{
					allCBSAccounts = new HashMap<>();

					// All existing CBS Accounts
					for (CBSAccount cbsAccount : existingCBSAccounts)
					{
						allCBSAccounts.put(cbsAccount.getIcn(), cbsAccount);
					}

					// Create and insert new CBS Accounts that are missing;
					// then create the combined all CBS Accounts with the new
					// and existing,
					if ((ICNs.size() - existingCBSAccounts.size()) > 0)
					{
						// Find ICNs that don't have CBSAccounts yet
						final Set<String> insertICNs = new HashSet<>();

						for (String icn : ICNs)
						{
							if (!allCBSAccounts.containsKey(icn))
							{
								insertICNs.add(icn);
							}
						}

						if (!insertICNs.isEmpty())
						{
							if (insertAndCreateCBSAccounts(insertICNs, allCBSAccounts) != null)
							{
								result.setSuccessful(true);
							}
						}

						if (!result.isSuccessful())
						{
							result.setMessage(MESSAGE_NOT_CREATING_ALL_ACCOUNTS);
						}
					}
					else
					{
						result.setSuccessful(true);
					}
				}

				// Step 3 - Retrieve Vista Account record by DFN / StationNum
				if (result.isSuccessful())
				{
					result.setSuccessful(false);

					Map<DfnStationNumPair, VistaAccount> vistaAccountMap = new HashMap<>();
					Map<DfnStationNumPair, VistaAccount> updateVistaAccounts = new HashMap<>();

					for (Iterator<VistaAcntHist> iterator = vistaAcntHistories.iterator(); iterator.hasNext();)
					{
						vistaAcntHistory = iterator.next();

						final String oldICN = vistaAcntHistory.getOldIcn();
						final String newICN = vistaAcntHistory.getNewIcn();
						final long dfn = vistaAcntHistory.getDfn();
						final String stationNum = vistaAcntHistory.getStationNum();

						final long newCBSAccountID = getCBSAccountID(allCBSAccounts, newICN);

						final DfnStationNumPair dfnStationNumPair = new DfnStationNumPair(dfn, stationNum);
						VistaAccount vistaAccount = vistaAccountMap.get(dfnStationNumPair);

						if (vistaAccount == null)
						{
							vistaAccount = this.vistaAccountDAO.getVistaAccount(vistaAcntHistory.getDfn(),
									vistaAcntHistory.getStationNum());

							if (vistaAccount != null)
							{
								vistaAccountMap.put(dfnStationNumPair, vistaAccount);
							}
						}

						if (vistaAccount != null)
						{
							if (oldICN.equals(vistaAccount.getIcn()))
							{
								// Step 3 a. - Set
								// VistaAccountHistory.oldCbssAcntId =
								// VistaAccount.cbssAcntId
								vistaAcntHistory.setOldCbssAcntId(vistaAccount.getCbssAcntId());
							}
							else
							{
								// Remove VistA Account History that has been
								// already processed.

								// oldCbssAcntId may have been set by MVI
								// Interface, so don't unset it - Brad -
								// 12/2/2016
								// vistaAcntHistory.setOldCbssAcntId(0);
								noprocessVistaAcntHistories.add(vistaAcntHistory);
								iterator.remove();
							}

							// Only update VistaAccount if ICN is different from
							// newICN - Brad - 12/2/2016
							// Changed updateVistaAccounts from List to
							// Map<DfnStationNumPair,VistaAccount>
							// to prevent the same VistaAccount from being
							// updated with the batchUpdate
							// that is done later. This assumes that the
							// VistaAcntHist records are being
							// processed in the order received so that the last
							// update to a given
							// DFN/SiteNum is what will be updated in
							// VistaAccount - Brad - 12/2/2016
							if (!newICN.equals(vistaAccount.getIcn()))
							{
								// Step 3 b. - Set VistaAccount.ICN = new ICN
								// and VistaAccount.cbssAcntId = cbssAcntId in
								// Step 2
								vistaAccount.setIcn(newICN);
								vistaAccount.setCbssAcntId(newCBSAccountID);

								updateVistaAccounts.put(dfnStationNumPair, vistaAccount);
							}
						}
					}

					// Step 3 c. - Update Vista Accounts
					numberOfUpdates = this.vistaAccountDAO.update(new ArrayList<>(updateVistaAccounts.values()));

					result.setSuccessful(true);

					if (numberOfUpdates == null || (numberOfUpdates.length != updateVistaAccounts.size()))
					{
						result.setSuccessful(false);
						result.setMessage(MESSAGE_NOT_UPDAING_ALL_ACCOUNTS);
					}
				}

				// (Step 2 - Update ICN's in CBS Account)
				// Sprint 5 - This step is no longer needed.
				// Yiping Yao
				// August 29, 2016
				// numberOfUpdates = this.cbsAccountDAO.batchUpdate(newICNs,
				// oldICNs);
				//
				// if (numberOfUpdates == null || numberOfUpdates.length !=
				// oldICNs.length)
				// {
				// result.setSuccessful(false);
				// result.setMessage(MESSAGE_NOT_UPDAING_ALL_ACCOUNTS);
				// }
				//
				// (Step 3 - Update ICN's in CBS Site Patient)
				// Sprint 5 - This step is now done in Generate CBS process.
				// Yiping Yao
				// August 29, 2016
				// if (result.isSuccessful())
				// {
				// numberOfUpdates = this.cbsSitePatientDAO.batchUpdate(newICNs,
				// oldICNs);
				//
				// if (numberOfUpdates == null || numberOfUpdates.length !=
				// oldICNs.length)
				// {
				// result.setSuccessful(false);
				// result.setMessage(MESSAGE_NOT_UPDAING_ALL_SITE_PATIENTS);
				// }
				// }
			}

			// Step 4 - Update VistA Account History with status of PROCESSED
			if (result.isSuccessful())
			{
				result.setSuccessful(false);

				if (noprocessVistaAcntHistories != null && !noprocessVistaAcntHistories.isEmpty())
				{
					long[] noprocessIDs = new long[noprocessVistaAcntHistories.size()];

					for (int i = 0; i < noprocessIDs.length; i++)
					{
						noprocessIDs[i] = noprocessVistaAcntHistories.get(i).getId();
					}

					this.vistaAcntHistDAO.update(noprocessIDs, Status.OTHER);
				}

				// this.vistaAcntHistDAO.update(ids, Status.PROCESSED);
				numberOfUpdates = this.vistaAcntHistDAO.update(vistaAcntHistories, Status.PROCESSED);

				result.setSuccessful(true);

				if (numberOfUpdates == null || numberOfUpdates.length == 0)
				{
					result.setSuccessful(false);
					result.setMessage(MESSAGE_NOT_UPDAING_ALL_ACCOUNTS);
				}
			}
		}
		catch (Exception ex)
		{
			logger.error("An error occurred in updateICN.", ex);
			result.setSuccessful(false);
			result.setMessage(ex.getMessage());
		}
		finally
		{
			// Step 5 - Finally
			if (result.isSuccessful())
			{
				// result.setResults(Arrays.stream( numberOfUpdates
				// ).boxed().toArray( Integer[]::new )); // Java 8
				result.set(ArrayUtils.toObject(numberOfUpdates));
			}
			else
			{
				logger.debug(result.getMessage());
			}
		}

		logger.info("Update ICN Service ends.");

		return result;
	}

	/**
	 * Insert and Create CBS Accounts from ICN's
	 * 
	 * Changed inNewICNs from String[] to Set to avoid step of removing
	 * duplicates - Brad - 12/2/2016
	 * 
	 * Changed return type from List <CBSAccount> to Map<String,CBSAccount> -
	 * Brad - 12/2/2016
	 * 
	 * Changed to allow the return Map to be passed in to avoid having a later
	 * merge step - Brad - 1/4/2017
	 * 
	 * Changed to use new batchInsertAndReturnCBSAccounts to avoid copying ICN
	 * Set to Array - Brad - 1/4/2017
	 * 
	 * @param inNewICNs
	 * @param inCbsAccounts
	 *            The CBSAccounts will be added to this Map and this Map will be
	 *            returned. If null, a new Map will be allocated.
	 * @return The passed in Map or a new Map (if null) with ICNs as keys and
	 *         CBSAccounts as values. Null is returned if the correct number of
	 *         ICNs are not created.
	 */
	private Map<String, CBSAccount> insertAndCreateCBSAccounts(Set<String> inNewICNs,
			Map<String, CBSAccount> inCbsAccounts) {
		// Assume the order of the returned keys (account numbers) is the same
		// as the ICN's.
		List<CBSAccount> newCBSAccounts = this.cbsAccountDAO.batchInsertAndReturnCBSAccounts(inNewICNs);

		if (newCBSAccounts == null || newCBSAccounts.size() == 0 || newCBSAccounts.size() != inNewICNs.size()) {
			return null;
		}

		Map<String, CBSAccount> cbsAccounts = (inCbsAccounts == null) ? new HashMap<>(newCBSAccounts.size()) : inCbsAccounts;

		for (CBSAccount cbsAccount : newCBSAccounts) {
			if (cbsAccount.getId() > 0) {
				cbsAccounts.put(cbsAccount.getIcn(), cbsAccount);
			}
		}

		logger.debug("Number of CBS Accounts created for new ICN's: " + cbsAccounts.size());

		return cbsAccounts;
	}

	/**
	 * Find and return the corresponding CBS Account ID from ICN.
	 * 
	 * Changed inCBSAccounts from List to Map to improve search performance -
	 * Brad - 12/2/2016
	 * 
	 * @param inCBSAccounts
	 *            Map of CBSAccounts by ICN
	 * @param inICN
	 *            ICN to find
	 * @return CBS Account ID for given ICN. 0 if not found.
	 */
	private static long getCBSAccountID(Map<String, CBSAccount> inCBSAccounts, String inICN) {
		long cbsAccountID = 0;

		CBSAccount cbsAccount = inCBSAccounts.get(inICN);
		if (cbsAccount != null) {
			cbsAccountID = cbsAccount.getId();
		}

		return cbsAccountID;
	}

	private static void addICN(Set<String> ICNs, String icn) {
		if ((icn != null) && !icn.isEmpty())
			ICNs.add(icn);
	}
}
