package gov.va.cpss.service;

import java.util.List;

/**
 * Service class for handling activities relating to sftp.
 * 
 * Converted to interface to support both Spring Integration and Apache VFS
 * implementations. Original Spring Integration implementation moved to
 * gov.va.cpss.service.impl.si.SpringIntegrationSftpServiceImpl, which will
 * eventually be removed since Spring Integration is currently not approved by
 * TRM. - Brad Pickle - 10/3/2016
 * 
 * @author DNS   
 */
public interface SftpService {

	/**
	 * Transfer the specified file path to the default directory on the server.
	 * If the filepath is to a file the file will be sent to the default
	 * directory on the server. IF the filepath is a directory then all files
	 * will be sent to the default directory on the server.
	 * 
	 * @param filepath
	 *            The path of the file or directory to send to the server.
	 * 
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpFileToServer(final String filepath);

	/**
	 * Transfer the specified file path to the specified directory on the
	 * server. If the filepath is to a file the file will be sent to the default
	 * directory on the server. IF the filepath is a directory then all files
	 * will be sent to the default directory on the server.
	 * 
	 * @param filepath
	 *            The path of the file or directory to send to the server.
	 * @param targetDirectory
	 *            The relative target directory on the server.
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpFileToServer(final String filepath, final String targetDirectory);

	/**
	 * Sent the specified local filepath to the server as the specified target
	 * filename in the target directory on the server.
	 * 
	 * @param filepath
	 *            The local filepath to send.
	 * @param targetFilename
	 *            The target filename on the server.
	 * @param targetDirectory
	 *            The target directory for the target filename on the server.
	 * @return Boolean flag indicating if successful or not.
	 */
	boolean ftpFileToServerWithName(final String filepath, final String targetFilename, final String targetDirectory);

	/**
	 * Send an empty file to the sftp server in the specified target directory
	 * with the specified name.
	 * 
	 * @param targetFilename
	 *            The name of the resulting empty file on the sftp server.
	 * @param targetDirectory
	 *            The target directory on the sftp server.
	 * @return Boolean flag indicating successful or not.
	 */
	boolean ftpEmptyFileToServerWithName(final String targetFilename, final String targetDirectory);

	/**
	 * Send the specified string content as the specified file content to the
	 * sftp server int he specified target directory with the specified name.
	 * 
	 * @param content
	 *            The content of the target file.
	 * @param targetFilename
	 *            The name of the resulting file on the sftp server.
	 * @param targetDirectory
	 *            The target directory on the sftp server.
	 * @return Boolean flag indicating successful or not.
	 */
	boolean ftpStringAsFileToServerWithName(final String content, final String targetFilename,
			final String targetDirectory);

	/**
	 * Remove the specified filename from the specified target directory on the
	 * server.
	 * 
	 * @param filename
	 *            The name of the file to remove.
	 * @param targetDirectory
	 *            The directory path where the file should reside.
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpRemoveFileFromDirectory(final String filename, final String targetDirectory);

	/**
	 * Move the specified filename to the specified target directory.
	 * 
	 * @param originalFilename
	 *            The name of the file to move.
	 * @param targetDirectory
	 *            The directory path to move the file.
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpMoveFileToDirectory(final String originalFilename, final String targetDirectory);

	/**
	 * Move the specified filename to the specified target directory.
	 * 
	 * @param originalFilename
	 *            The name of the file to move.
	 * @param targetDirectory
	 *            The directory path to move the file.
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpMoveFileFromDirectoryToDirectory(final String originalFilename, final String originalDirectory,
			final String targetDirectory);

	/**
	 * Rename the specified filename with the new filename in the specified
	 * target directory on the server.
	 * 
	 * @param originalFilename
	 *            The original filename on the server.
	 * @param newFilename
	 *            The new filename on the server.
	 * @param targetDirectory
	 *            The target directory where the files reside on the server.
	 * @return Boolean flag indicating successful rename or not.
	 */
	boolean ftpRenameFileInDirectory(final String originalFilename, final String newFilename,
			final String targetDirectory);

	/**
	 * Get a list of files that reside in the specified directory path.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @return Boolean flag indicating success or failure.
	 */
	List<String> ftpGetFileListInDirectory(final String targetDirectory);

	/**
	 * Get a list of files that reside in the specified directory path.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @param throwExceptions
	 *            If true, any exceptions will be re-thrown after being logged.
	 * @return Boolean flag indicating success or failure.
	 */
	List<String> ftpGetFileListInDirectory(final String targetDirectory, boolean throwExceptions);

	/**
	 * Get a list of files that reside in the specified directory path that have
	 * the specified extension. The extension check will ignore case.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @return Boolean flag indicating success or failure.
	 */
	List<String> ftpGetFileListWithExtensionInDirectory(final String targetDirectory, final String extension);

	/**
	 * Get a list of files that reside in the specified directory path that have
	 * the specified extension. The extension check will ignore case.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @param throwExceptions
	 *            If true, any exceptions will be re-thrown after being logged.
	 * @return List of file names sorted by their modified dates
	 */
	List<String> ftpGetFileListWithExtensionInDirectory(final String targetDirectory, final String extension,
			boolean throwExceptions);

	/**
	 * Check if the specified filename exists in the target directory.
	 * 
	 * @param filename
	 *            The name of the file to check existence.
	 * @param targetDirectory
	 *            The target directory path to check for the specified file.
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpFileExistsInDirectory(final String filename, final String targetDirectory);

	/**
	 * Check if the specified filename exists in the target directory.
	 * 
	 * @param filename
	 *            The name of the file to check existence.
	 * @param targetDirectory
	 *            The target directory path to check for the specified file.
	 * @param throwExceptions
	 *            If true, any exceptions will be re-thrown after being logged.
	 * @return Boolean flag indicating success or failure.
	 */
	boolean ftpFileExistsInDirectory(final String filename, final String targetDirectory, boolean throwExceptions);

	/**
	 * Get the size of the specified file or null if does not exist.
	 * 
	 * @param targetDirectory
	 *            The directory path to get associated file listing.
	 * @return Long Size of the file or null if does not exist.
	 */
	Long ftpGetFileSizeInDirectory(final String filename, final String targetDirectory);

	/**
	 * Open a stream for the specified file.
	 * 
	 * @param filename
	 *            The file to open.
	 * @return SftpStreamSession bean that contains the session and input stream
	 *         or null if error.
	 * @throws Exception
	 */
	SftpStreamSession openFileStream(final String filename, final String directory) throws Exception;

}
