package gov.va.cpss.service;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.stereotype.Service;

import gov.va.cpss.dao.CBSMessageDAO;
import gov.va.cpss.dao.CBSSiteStmtDAO;
import gov.va.cpss.dao.CBSSiteTransDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * Service class for handling activities relating to the Send CBS batch job.
 * 
 * @author Brad Pickle
 */
@Service
public class SendCBSService {

	private static final Logger serviceLogger = Logger.getLogger(SendCBSService.class.getCanonicalName());

	private CBSMessageDAO cbsMessageDAO;

	private ProcessStatusDAO processStatusDAO;

	private CBSStmtDAO cbsStmtDAO;

	private CBSSiteStmtDAO cbsSiteStmtDAO;

	private CBSSiteTransDAO cbsSiteTransDAO;

	private String stagingDirectory;

	public SendCBSService() {
	}

	public CBSMessageDAO getCbsMessageDAO() {
		return cbsMessageDAO;
	}

	public void setCbsMessageDAO(CBSMessageDAO cbsMessageDAO) {
		this.cbsMessageDAO = cbsMessageDAO;
	}

	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	public CBSStmtDAO getCbsStmtDAO() {
		return cbsStmtDAO;
	}

	public void setCbsStmtDAO(CBSStmtDAO cbsStmtDAO) {
		this.cbsStmtDAO = cbsStmtDAO;
	}

	public CBSSiteStmtDAO getCbsSiteStmtDAO() {
		return cbsSiteStmtDAO;
	}

	public void setCbsSiteStmtDAO(CBSSiteStmtDAO cbsSiteStmtDAO) {
		this.cbsSiteStmtDAO = cbsSiteStmtDAO;
	}

	public CBSSiteTransDAO getCbsSiteTransDAO() {
		return cbsSiteTransDAO;
	}

	public void setCbsSiteTransDAO(CBSSiteTransDAO cbsSiteTransDAO) {
		this.cbsSiteTransDAO = cbsSiteTransDAO;
	}

	public String getStagingDirectory() {
		return stagingDirectory;
	}

	public void setStagingDirectory(String stagingDirectory) {
		this.stagingDirectory = stagingDirectory;
	}

	public String getCBSOutputResource(String cbsOutputFileName) {
		return stagingDirectory + "/" + cbsOutputFileName;
	}

	/**
	 * Determines if there are consolidated statements (CBSStmt) in NEW status
	 * and if so, initializes a CBSMessage record in INITIAL status to be used
	 * by the Send CBS batch job. Any existing CBSMessage records in INITIAL
	 * status will be set to ERROR, effectively replacing any running or
	 * abandoned jobs with this new one. Any new consolidated statements will be
	 * updated with the messageId of the new CBSMessage record.
	 * 
	 * This process should be run in an isolated transaction to ensure that only
	 * one Send CBS job will own the consolidated statements.
	 * 
	 * NOTE: Using {@link #startIsolatedSendCBSJob(BatchRun)
	 * startIsolatedSendCBSJob} instead. No longer checking for other Send CBS
	 * instance and setting messageId on CBSMessage records has been moved into
	 * a task following the check for Generate CBS running.
	 * 
	 * @param batchRun
	 *            BatchRun record associated with this run.
	 * @return CBSMessage A new initialized CBSMessage for the batch run.
	 *         Returns null if there are no consolidated statements to process.
	 */
	@Deprecated
	public CBSMessage startSendCBSJob(final BatchRun batchRun) {

		final Integer initialStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		if (initialStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.INITIAL);
			return null;
		}
		serviceLogger.debug("initialStatusId:" + initialStatusId);

		final Integer errorStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
			return null;
		}
		serviceLogger.debug("errorStatusId:" + errorStatusId);

		final Integer newStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);
		if (newStmtStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.NEW);
			return null;
		}
		serviceLogger.debug("newStmtStatusId:" + newStmtStatusId);

		final List<CBSMessage> cbsMessagesInInitialL = cbsMessageDAO.getCBSMessagesInStatus(initialStatusId);
		for (CBSMessage cbsMessageInInitial : cbsMessagesInInitialL) {
			cbsMessageInInitial.setStatusId(errorStatusId);
			cbsMessageDAO.update(cbsMessageInInitial);
		}

		CBSMessage cbsMessage = null;
		if (cbsStmtDAO.getStatementCountWithStatus(newStmtStatusId) > 0) {
			cbsMessage = new CBSMessage();
			cbsMessage.setStatusId(initialStatusId);
			cbsMessage.setBatchRunId(batchRun.getId());
			cbsMessage.setFileName(generateFileName(batchRun));

			cbsMessageDAO.insert(cbsMessage);
			final long cbsMessageId = cbsMessage.getId();

			cbsStmtDAO.updateMessageIdForStatus(newStmtStatusId, cbsMessageId);
		}

		return cbsMessage;
	}

	/**
	 * Initializes a CBSMessage record in INITIAL status to be used by the Send
	 * CBS batch job.
	 * 
	 * @param batchRun
	 *            BatchRun record associated with this run.
	 * @return CBSMessage A new initialized CBSMessage for the batch run.
	 */
	public CBSMessage startIsolatedSendCBSJob(final BatchRun batchRun) {

		final Integer initialStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		if (initialStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing job. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.INITIAL);
			serviceLogger.error(error.toString());
			throw new RuntimeException(error.toString());
		}
		serviceLogger.debug("initialStatusId:" + initialStatusId);

		final Integer errorStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
			return null;
		}
		serviceLogger.debug("errorStatusId:" + errorStatusId);

		final List<CBSMessage> cbsMessagesInInitialL = cbsMessageDAO.getCBSMessagesInStatus(initialStatusId);
		for (CBSMessage cbsMessageInInitial : cbsMessagesInInitialL) {
			cbsMessageInInitial.setStatusId(errorStatusId);
			cbsMessageDAO.update(cbsMessageInInitial);
		}

		CBSMessage cbsMessage = new CBSMessage();
		cbsMessage.setStatusId(initialStatusId);
		cbsMessage.setBatchRunId(batchRun.getId());
		cbsMessage.setFileName(generateFileName(batchRun));

		cbsMessageDAO.insert(cbsMessage);

		return cbsMessage;
	}

	private String generateFileName(BatchRun batchRun) {
		return String.format("CBS-%tm%<td%<tY.txt,SBS-%<tm%<td%<tY.txt", batchRun.getStartDate());
	}

	public List<CBSStmt> getMessageStatementsForSite(final long cbsMessageId, final String stationNum,
			final boolean loadSiteStatements) {
		final List<CBSStmt> cbsStmtL = cbsStmtDAO.getMessageStatementsForSite(cbsMessageId, stationNum);

		if (loadSiteStatements) {

			final Map<Long, CBSStmt> cbsStmtById = new HashMap<Long, CBSStmt>();
			for (CBSStmt cbsStmt : cbsStmtL) {
				cbsStmt.setSiteStmtL(new LinkedList<CBSSiteStmt>());
				cbsStmtById.put(cbsStmt.getId(), cbsStmt);
			}

			// Load CBSSiteStmt and CBSSitePatient records and merge into
			// CBSStmt objects
			final List<CBSSiteStmt> cbsSiteStmtL = cbsSiteStmtDAO.getMessageStatementSitesForSite(cbsMessageId,
					stationNum);
			final Map<Long, CBSSiteStmt> cbsSiteStmtById = new HashMap<Long, CBSSiteStmt>();
			
			for (CBSSiteStmt cbsSiteStmt : cbsSiteStmtL) {
				cbsSiteStmt.setSiteTransL(new LinkedList<CBSSiteTrans>());
				cbsSiteStmtById.put(cbsSiteStmt.getId(), cbsSiteStmt);
				cbsStmtById.get(cbsSiteStmt.getStmtId()).getSiteStmtL().add(cbsSiteStmt);				
			}

			// Load CBSSiteTrans records and merge into CBSSiteStmt objects
			final List<CBSSiteTrans> cbsSiteTransL = cbsSiteTransDAO.getMessageStatementTransForSite(cbsMessageId,
					stationNum);
			for (CBSSiteTrans cbsSiteTrans : cbsSiteTransL) {
				cbsSiteStmtById.get(cbsSiteTrans.getSiteStmtId()).getSiteTransL().add(cbsSiteTrans);
			}
		}

		return cbsStmtL;
	}
	
	public boolean endSendCBSJob(JobExecution execution, CBSMessage cbsMessage) {

		boolean jobSuccess = false;

		if (execution == null) {

			serviceLogger.error("Job ended with null execution");
			errorSendCBSJob(cbsMessage);

		} else if (execution.getExitStatus() != ExitStatus.COMPLETED) {

			if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
				serviceLogger
						.error("Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY));
			} else {
				serviceLogger.error("Job ended with unknown error: " + execution.getExitStatus());
			}
			errorSendCBSJob(cbsMessage);
		} else {

			// Job was successful!
			jobSuccess = true;
		}

		serviceLogger.info("End Send CBS Job");

		return jobSuccess;
	}

	private void errorSendCBSJob(CBSMessage cbsMessage) {

		// Change CBSMessage status to ERROR
		final Integer errorStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
		} else {
			cbsMessage.setStatusId(errorStatusId);
			cbsMessageDAO.update(cbsMessage);

		}
	}

	public boolean deleteCBSOutputResource(String cbsFileName) {

		File file = new File(cbsFileName);

		try {
			return Files.deleteIfExists(file.toPath());
		} catch (IOException e) {
			final StringBuilder error = new StringBuilder();
			error.append("An error occurred while attempting to delete the local CBS file, ");
			error.append(cbsFileName);
			error.append(", error: ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());

			serviceLogger.error(error.toString());
			return false;
		}

	}

	/**
	 * Update all statement status to sent status for the specified message ID.
	 * 
	 * @param cbsMessageId
	 *            The message ID for the status.
	 * @return Long value that is the number of statement status updated.
	 */
	public long updateAllStatementStatusToSentForMessageId(long cbsMessageId) {

		long totalPatientsIncluded = -1;

		final Integer sentStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SENT);
		if (sentStmtStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.SENT);
		} else {

			// Update CBSStmt status to SENT
			totalPatientsIncluded = cbsStmtDAO.updateStatusForMessageId(sentStmtStatusId, cbsMessageId);
		}

		return totalPatientsIncluded;
	}

	/**
	 * Attempt to set the message record with the specified ID to the specified
	 * status.
	 * 
	 * @param cbsMessageId
	 *            The ID of the message to set.
	 * @param status
	 *            The status value to set the message.
	 * @return Boolean flag indicating if successful or not.
	 */
	public boolean setMessageStatus(long cbsMessageId, ProcessStatus.Status status) {

		final Integer statusId = processStatusDAO.getStatusFromEnum(status);
		if (statusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + status);
			return false;
		}

		// Look up the CBSMessage record to make sure it has not been set to
		// ERROR by a subsequent Send CBS batch run
		CBSMessage cbsMessage = cbsMessageDAO.get(cbsMessageId);
		if (cbsMessage == null) {
			serviceLogger.error("Unable to obtain CBSMessage record for id: " + cbsMessageId);
			return false;
		}

		// Update CBSMessage status to SUCCESS
		cbsMessage.setStatusId(statusId);
		cbsMessageDAO.update(cbsMessage);

		return true;
	}

}
