package gov.va.cpss.service;

import static gov.va.cpss.model.ps.Constants.TOTAL_SITE_COUNT_KEY;
import static gov.va.cpss.model.ps.Constants.TOTAL_PATIENT_COUNT_KEY;
import static gov.va.cpss.model.ps.Constants.ERROR_SITES_KEY;
import static gov.va.cpss.model.ps.Constants.LINE_FEED;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicReference;

import org.springframework.batch.core.JobExecution;

import org.springframework.stereotype.Service;

import gov.va.cpss.model.apps.APSReceived;
import gov.va.cpss.model.apps.APSReceivedSite;
import gov.va.cpss.model.apps.APSSite;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSDetails;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.updatestation.StationInfo;

/**
 * Service class for handling activities relating to load APPS Payments Data.
 * 
 * Copyright DXC / VA
 * May 8, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@Service
@SuppressWarnings("nls")
public class LoadAPPSService extends APPSBaseService<APSReceived>
{
    // Station lists
    // (Using AtomicReference to alleviate the "race condition" raised by the Fortify scan.)
    private AtomicReference<List<StationInfo>> receivedStationsRef = new AtomicReference<>();
    private AtomicReference<List<StationInfo>> remainingStationsRef = new AtomicReference<>();


    //
    // Service methods - Load APPS
    //
    @Override
    public Result<APSReceived> startJob(final BatchRun batchRun, final String fileName)
    {
        Result<APSReceived> result = super.startJob(batchRun, fileName);

        if (result == null)
        {
            result = new Result<>();
        }

        this.receivedStationsRef.set(null);
        this.remainingStationsRef.set(null);

        APSReceived apsReceived = new APSReceived(batchRun.getId(), this.statusIdInitial, batchRun.getStartDate(), fileName);

        try
        {
            long receivedId = this.apsReceivedDAO.save(apsReceived);

            if (receivedId > 0)
            {
                apsReceived.setId(receivedId);

                result.set(apsReceived);
                result.setSuccessful(true);
            }
            else
            {
                result.setSuccessful(false);
            }
        }
        catch (Exception ex)
        {
            result.setMessage("Failed to save APSReceived: " + ex.getMessage());
            result.setSuccessful(false);
        }

        return result;
    }

    @Override
    @SuppressWarnings("unchecked")
    public String endJob(JobExecution execution, APSReceived apsReceived)
    {
        String message = null;
        String tempMessage = null;

        // NEW:
        // Remove sites that have been erred.
        if ( execution != null &&
             execution.getExecutionContext().containsKey(ERROR_SITES_KEY) &&
             execution.getExecutionContext().get(ERROR_SITES_KEY) != null &&
             !((List<String>) execution.getExecutionContext().get(ERROR_SITES_KEY)).isEmpty() )
        {
            List<String> errorSites = (List<String>) execution.getExecutionContext().get(ERROR_SITES_KEY);

            this.logger.debug("Deleting error sites: " + errorSites);

            message = deleteSites(errorSites);
        }

        this.receivedStationsRef.set(null);
        this.remainingStationsRef.set(null);

        tempMessage = super.endJob(execution, apsReceived);
 
        if (tempMessage != null)
        {
            message = (message == null) ? tempMessage : message + LINE_FEED + tempMessage;
        }

        return message;
    }

    @Override
    protected String processJobError(APSReceived apsReceived)
    {
        this.logger.info("Service: Process Load Job Error.");

        String message = null;

        // Be sure to delete any partial processed data.
        // Deleting the site entries will cascade delete through the child tables.
        message = deleteSite(apsReceived.getId());

        apsReceived.setStatusId(this.statusIdError);

        try
        {
            this.apsReceivedDAO.updateStatus(apsReceived);
        }
        catch (Exception ex)
        {
            message = (message == null) ? ex.getMessage() : message + LINE_FEED + ex.getMessage();
        }

        return message;
    }

    @Override
    protected String completeJob(JobExecution execution, APSReceived apsReceived)
    {
        this.logger.debug("Service: Complete Load Job.");

        String message = null;

        if (execution.getExecutionContext().containsKey(TOTAL_SITE_COUNT_KEY))
        {
            apsReceived.setNumOfSite(execution.getExecutionContext().getLong(TOTAL_SITE_COUNT_KEY));
        }
        else
        {
            message = "Unable to obtain total site count from process results.";
        }

        if (execution.getExecutionContext().containsKey(TOTAL_PATIENT_COUNT_KEY))
        {
            apsReceived.setNumOfPatient(execution.getExecutionContext().getLong(TOTAL_PATIENT_COUNT_KEY));
        }
        else
        {
            message += LINE_FEED + "Unable to obtain total patient count from process results.";
        }

        if (message != null)
        {
            this.logger.error(message);
        }
        else
        {
            // If successfully processed set the status to NEW to indicate ready
            // for processing.
            apsReceived.setStatusId(this.statusIdNew);

            try
            {
                this.apsReceivedDAO.updateResults(apsReceived);
            }
            catch (Exception ex)
            {
                message = "Failed to update: " + ex.getMessage();
            }
        }

        return message;
    }

    /**
     * Bulk / batch save a list of site and its children data.
     * 
     * This save method uses batch insert with generated keys for the
     * two tables, APSSite, APSPatient, but not APSDetails.
     * 
     * @param entries
     * @return
     */
    public String save(List<? extends Entry<APSSite, List<APSPatient>>> entries)
    {
        this.logger.info("Service: Save APPS Data Started.");

        final long methodStartTime = System.nanoTime();

        String message = null;

        // Save Sites
        List<APSSite> allSites = new ArrayList<>();

        for (Entry<APSSite, List<APSPatient>> entry : entries)
        {
            this.logger.debug("Saving APSSite: " + entry.getKey());

            allSites.add(entry.getKey());
        }

        if (!allSites.isEmpty())
        {
            final long saveStartTime = System.nanoTime();

            try
            {
                long[] allSitesIDs = saveSites(allSites);

                if (allSitesIDs == null || allSitesIDs.length != allSites.size())
                {
                    message = "Saving APSSite erred. Not all Sites saved.";
                    this.logger.error(message);
    
                    return message;
                }
            }
            catch (Exception ex)
            {
                message = "Failed to save Sites: " + ex.getMessage();
                this.logger.error(message);

                return message;
            }

            final long saveElapsedTime = System.nanoTime() - saveStartTime;

            this.logger.debug("Saving Sites: " + allSites.size() + ". " + printElapsedTime("save", saveElapsedTime));
        }

        allSites = null;

        // Set Site ID on Patients
        List<APSPatient> allPatients = new ArrayList<>();

        for (Entry<APSSite, List<APSPatient>> entry : entries)
        {
            final APSSite site = entry.getKey();
            List<APSPatient> sitePatients = entry.getValue();

            sitePatients.forEach(p -> p.setApsSite(site));

            allPatients.addAll(sitePatients);
        }

        // Save Patients
        if (!allPatients.isEmpty())
        {
            final long saveStartTime = System.nanoTime();

            try
            {
                long[] allPatientsIDs = savePatients(allPatients);

                if (allPatientsIDs == null || allPatientsIDs.length != allPatients.size())
                {
                    message = "Saving APSPatient erred. Not all Patients saved.";
                    this.logger.error(message);
    
                    return message;
                }
            }
            catch (Exception ex)
            {
                message = "Failed to save Patients: " + ex.getMessage();
                this.logger.error(message);

                return message;
            }

            final long saveElapsedTime = System.nanoTime() - saveStartTime;

            this.logger.debug("Saving Patients: " + allPatients.size() + ". " + printElapsedTime("save", saveElapsedTime));
        }

        // Set Patient ID on Details
        List<APSDetails> allDetailsList = new ArrayList<>();

        for (APSPatient patient : allPatients)
        {
            List<APSDetails> patientDetailsList = patient.getDetailsList();

            patientDetailsList.forEach(d -> d.setApsPatientId(patient.getId()));

            allDetailsList.addAll(patientDetailsList);
        }

        allPatients = null;

        // Save Details
        if (!allDetailsList.isEmpty())
        {
            final long saveStartTime = System.nanoTime();

            try
            {
                int[] numOfInserts = saveDetailsList(allDetailsList);

                if (numOfInserts == null || numOfInserts.length != allDetailsList.size())
                {
                    message = "Saving APSDetails erred. Not all Details saved.";
                    this.logger.error(message);
    
                    return message;
                }
            }
            catch (Exception ex)
            {
                message = "Failed to save Details: " + ex.getMessage();
                this.logger.error(message);

                return message;
            }

            final long saveElapsedTime = System.nanoTime() - saveStartTime;

            this.logger.debug("Saving Details: " + allDetailsList.size() + ". " + printElapsedTime("save", saveElapsedTime));
        }

        allDetailsList = null;
        
        final long methodElapsedTime = System.nanoTime() - methodStartTime;

        this.logger.info("Service: Save APPS Data Ended. " + printElapsedTime("method", methodElapsedTime));

        return message;
    }

    //
    // Other methods
    //
    //
    // C (Create / Save / Insert) R (Retrieve / Get / Select) U (Update) D (Delete)
    // methods for APSReceivedSite
    //
    public long saveReceivedSite(APSReceivedSite receivedSite, boolean isSiteSuccessful)
    {
        if (receivedSite != null)
        {
            receivedSite.setStatementDate(getSqlDateFromYear(this.previousYear));

            if (isSiteSuccessful)
            {
                receivedSite.setStatusId(this.statusIdInitial);
            }
            else
            {
                receivedSite.setStatusId(this.statusIdError);
            }
        }

        return this.apsReceivedSiteDAO.save(receivedSite);
    }

    public APSReceivedSite getReceivedSite(String facilityNumber)
    {
        List<APSReceivedSite> receivedSites = this.apsReceivedSiteDAO.select(facilityNumber, this.previousYear);

        if (receivedSites == null || receivedSites.isEmpty())
        {
            return null;
        }

        // Should only have one row.
        return receivedSites.get(0); 
    }

    public int updateReceivedSite(APSReceivedSite receivedSite, boolean isSiteSuccessful)
    {
        // If the Site is erred, set the status to error.
        if (!isSiteSuccessful)
        {
            receivedSite.setStatusId(this.statusIdError);
        }

        return this.apsReceivedSiteDAO.update(receivedSite.getId(), receivedSite.getNumOfMessages(), receivedSite.getStatusId());
    }

    public int deleteReceivedSite(APSReceivedSite receivedSite)
    {
        return this.apsReceivedSiteDAO.deleteById(receivedSite.getId());
    }

    /**
     * Save APSSite list and update the database with auto generated IDs.
     * 
     * @param sites
     *            The sites record.
     * @return The updated sites populated with the auto generated IDs.
     */
    public long[] saveSites(List<APSSite> sites)
    {
        return this.apsSiteDAO.batchSave(sites);
    }

    /**
     * Save APSPatient list and update the database with auto generated IDs.
     * 
     * @param patients
     *            The patients record.
     * @return The updated patients populated with the auto generated IDs.
     */
    public long[] savePatients(List<APSPatient> patients)
    {
        return this.apsPatientDAO.batchSave(patients);
    }

    public int[] saveDetailsList(List<APSDetails> detailsList)
    {
        if (detailsList == null || detailsList.isEmpty())
        {
            return null;
        }

        int[] numberOfSaves = null;

        try
        {
            numberOfSaves = this.apsDetailsDAO.batchInsert(detailsList);
        }
        catch (Exception ex)
        {
            this.logger.error("Error enountered when saving details: " + ex.getMessage());
        }

        return numberOfSaves;
    }

    /**
     * Delete the site via APSRecieved ID, which will cascading deleted all of its children.
     * 
     * @param receivedId
     *            The APSReceived ID.
     */
    public String deleteSite(long receivedId)
    {
        String message = null;

        try
        {
            this.apsSiteDAO.deleteByReceivedId(receivedId);
        }
        catch (Exception ex)
        {
            message = "Delete Site failed: " + ex.getMessage();
        }

        return message;
    }

    /**
     * Delete the sites via facility numbers, which will cascading deleted all of its children.
     * 
     * @param sites
     *            The APSSites.
     */
    public String deleteSites(List<String> sites)
    {
        String message = null;

        if (sites != null && !sites.isEmpty())
        {
            try
            {
                this.apsSiteDAO.deleteByFacilityNumbers(sites);
            }
            catch (Exception ex)
            {
                message = "Delete Sites failed: " + ex.getMessage();
            }
        }

        return message;
    }

    /**
     * Check if the Site is erred,and if so,
     * check to see if it has passed the rejection period.
     * 
     * @param statusId
     * @return
     */
    public boolean isSiteStillErred(APSReceivedSite receivedSite, int rejectionPeriodInHours)
    {
        boolean isSiteErred = false;

        if (isErrorStatus(receivedSite.getStatusId()))
        {
            Calendar rejectionTime = Calendar.getInstance();
            rejectionTime.add(Calendar.HOUR_OF_DAY, -rejectionPeriodInHours);

            isSiteErred = receivedSite.getReceivedDate().after(rejectionTime.getTime());
        }

        return isSiteErred;
    }

    /**
     * Return a list of stations that have sent the payment information.
     */
    public List<StationInfo> getPaymentRecievedStations()
    {
        if (this.receivedStationsRef.get() == null || this.remainingStationsRef.get() == null)
        {
            buildStationsLists();
        }
 
        return this.receivedStationsRef.get();
    }

    /**
     * Return a list of remaining stations that have not sent the payment information.
     */
    public List<StationInfo> getRemainingStations()
    {
        if (this.remainingStationsRef.get() == null || this.receivedStationsRef.get() == null)
        {
            buildStationsLists();
        }
 
        return this.remainingStationsRef.get();
    }

    /**
     * Build the lists of received stations and remaining stations.
     * 
     * synchronized modifier is added to alleviate the "race condition" raised by the Fortify scan.
     */
    private synchronized void buildStationsLists()
    {
        if (StationInfoUtil.getStationInfoMap() == null)
        {
            return;
        }

        List<StationInfo> receivedStations = new ArrayList<>();
        List<StationInfo> remainingStations = new ArrayList<>();

        List<APSSite> receivedSites = this.apsSiteDAO.getPaymentReceivedSites(this.previousYear, getSqlDateFromYear(this.currentYear));

        Collection<StationInfo> stations = StationInfoUtil.getStationInfoMap().values(); 
        APSSite tmpSite;

        if (receivedSites != null && !receivedSites.isEmpty())
        {
            for (StationInfo station : stations)
            {
                // Using Java 8 stream / filter / predicate, so not to use collection contains() and the model equals() methods.
                tmpSite = receivedSites.stream().filter(s -> s.getFacilityNum().trim().equalsIgnoreCase(station.getStationNum().trim())).findFirst().orElse(null);

                //this.logger.debug("Received site: " + (tmpSite == null ? tmpSite : tmpSite.getFacilityNum()) + ". Station: " + station.getStationNum() + ".");

                if (tmpSite != null)
                {
                    receivedStations.add(station);
                }
                else
                {
                    remainingStations.add(station);
                }
            }
        }
        else
        {
            remainingStations.addAll(stations);
        }

        this.receivedStationsRef.set(receivedStations);
        this.remainingStationsRef.set(remainingStations);
    }
}
