package gov.va.cpss.service;

import static gov.va.cpss.model.ps.Constants.AVAILABLE_CS_STATISTICS_KEY;
import static gov.va.cpss.model.ps.Constants.GENERATED_CS_STATISTICS_KEY;
import static gov.va.cpss.model.ps.Constants.LINE_FEED;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.batch.core.JobExecution;

import org.springframework.stereotype.Service;

import gov.va.cpss.model.apps.APSSite;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSDetails;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSSitePatient;
import gov.va.cpss.model.apps.APSPayment;
import gov.va.cpss.model.cbs.CBSAccount;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.ProcessStatus.Status;

/**
 * Service class for handling activities relating to Generate APPS Statements Data.
 * 
 * Copyright DXC / VA
 * May 9, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@Service
@SuppressWarnings({"nls", "static-method"})
public class GenerateAPPSService extends APPSBaseService<BatchRunProcess>
{
    //
    // Service methods - Generate APPS
    //
    /**
     * Start Generate APPS batch job.
     * 
     * @param batchRun
     * @return
     */
    @Override
    public Result<BatchRunProcess> startJob(final BatchRun batchRun, final String fileName)
    {
        Result<BatchRunProcess> result = super.startJob(batchRun, fileName);

        if (result == null)
        {
            result = new Result<>();
        }

        BatchRunProcess process = new BatchRunProcess();

        process.setBatchRunId(batchRun.getId());
        process.setFileName(fileName);
        process.setProcessDate(batchRun.getStartDate());
        process.setStatusId(this.statusIdInitial);

        try
        {
            long id = this.batchRunProcessDAO.save(process);

            process.setId(id);
 
            result.set(process);
            result.setSuccessful(true);
        }
        catch (Exception ex)
        {
            result.setMessage(ex.getMessage());
            result.setSuccessful(false);
        }

        return result;
    }

    /**
     * End Generate APPS batch job.
     * 
     * @param execution
     * @param process
     * @return
     */
    @Override
    public String endJob(JobExecution execution, BatchRunProcess process)
    {
        // Nothing specific for Generate APPS job.
        return super.endJob(execution, process);
    }

    /**
     * Process Generate APPS batch job error.
     *  
     * @param process
     */
    @Override
    protected String processJobError(BatchRunProcess process)
    {
        this.logger.info("Service: Process Generate Job Error.");

        String message = null;

        // Be sure to delete any partial processed data.
        // Deleting the statement entries will cascade delete through the child tables.
        message = deleteStatements(process.getId());

        try
        {
            process.setStatusId(this.statusIdError);
            this.batchRunProcessDAO.updateStatus(process);
        }
        catch (Exception ex)
        {
            message = (message == null) ? ex.getMessage() : message + LINE_FEED + ex.getMessage();
        }
 
        return message;
    }

    /**
     * Complete Generate APPS batch job.
     * 
     * @param execution
     * @param process
     * @return
     */
    @Override
    protected String completeJob(JobExecution execution, BatchRunProcess process)
    {
        this.logger.info("Service: Complete Generate Job.");

        String message = null;
        StringBuilder statisticsMsg = new StringBuilder();

        if (execution.getExecutionContext().containsKey(AVAILABLE_CS_STATISTICS_KEY))
        {
            statisticsMsg.append("Total APPS Patient Records Available / Read / Loaded: ");
            statisticsMsg.append(execution.getExecutionContext().getString(AVAILABLE_CS_STATISTICS_KEY));
            statisticsMsg.append(". ");
        }
        else
        {
            message = "Unable to obtain total patient records count from process results.";
        }

        if (execution.getExecutionContext().containsKey(GENERATED_CS_STATISTICS_KEY))
        {
            statisticsMsg.append("Total APPS Statements Generated: ");
            statisticsMsg.append(execution.getExecutionContext().getString(GENERATED_CS_STATISTICS_KEY));
            statisticsMsg.append(".");
        }
        else
        {
            message += LINE_FEED + "Unable to obtain total generated statement count from process results."; 
        }

        if (message != null)
        {
            this.logger.error(message);
        }
        else
        {
            // If successfully processed set the status to PROCESSED.
            process.setStatusId(this.statusIdProcessed);
            process.setOther(statisticsMsg.toString());
            this.batchRunProcessDAO.updateStatus(process);
        }

        return message;
    }

    /**
     * Read patient data with pagination.
     * 
     * @param entries
     * @return
     */
    public Result<List<APSPatient>> getNewPatientsWithPaging(int page, int pageSize)
    {
        Result<List<APSPatient>> result = new Result<>();
        
        try
        {
            List<APSPatient> patients = this.apsPatientDAO.getPatientsWithPaging(page, pageSize, this.previousYear, Status.NEW);

            result.set(patients);
            result.setSuccessful(true);
        }
        catch (Exception ex)
        {
            result.setMessage("Failed to get new Patients with paging: " + ex.getMessage());
            result.setSuccessful(false);
        }

        return result;
    }

    /**
     * Bulk / batch save a list of statements and its children data.
     * 
     * Only saves for Statement and Site Statement return generated keys;
     * the Site Patient and Payment do not, since there is no need.
     * 
     * @param entries
     * @return
     */
    public String save(List<? extends Entry<APSStmt, List<APSSiteStmt>>> entries)
    {
        this.logger.info("Service: Save CAPS Data Started.");

        final long methodStartTime = System.nanoTime();

        String message = null;

        long updateStartTime = 0L;
        long updateElapsedTime = 0L;
        long saveStartTime = 0L;
        long saveElapsedTime = 0L;

        // Save statements
        List<Entry<APSStmt, List<APSSiteStmt>>> existingEntries = new ArrayList<>();
        List<Entry<APSStmt, List<APSSiteStmt>>> initialEntries = new ArrayList<>();

        List<APSStmt> existingStatements = new ArrayList<>();
        List<APSStmt> initialStatements = new ArrayList<>();

        List<APSSiteStmt> updatedSiteStatements = new ArrayList<>();
        List<APSSiteStmt> initialSiteStatements = new ArrayList<>();
        List<APSSiteStmt> initialSiteStmtsFromExistingStmts = new ArrayList<>();
        
        for (Entry<APSStmt, List<APSSiteStmt>> entry : entries)
        {
            //this.logger.debug("Saving APSStmt: " + entry.getKey());

            if (entry.getKey().getStatusId() == this.statusIdOther)
            {
                existingEntries.add(entry);
                existingStatements.add(entry.getKey());

                // For existing Statement, get both existing updated and initial Site Statements,
                // which already have the associated Statement ID.
                updatedSiteStatements.addAll(entry.getValue().stream().filter(s -> s.getStatusId() == this.statusIdOther).collect(Collectors.toList()));
                initialSiteStmtsFromExistingStmts.addAll(entry.getValue().stream().filter(s -> s.getStatusId() == this.statusIdInitial).collect(Collectors.toList()));
            }
            else if (entry.getKey().getStatusId() == this.statusIdInitial)
            {
                initialEntries.add(entry);
                initialStatements.add(entry.getKey());
            }
        } // for

        // For existing Statements, do update
        if (!existingStatements.isEmpty())
        {
            updateStartTime = System.nanoTime();

            // Set existing Statements status back to INITIAL,
            // so that they can be included in consolidation again later.
            existingStatements.forEach(s -> s.setStatusId(this.statusIdInitial));

            try
            {
                updateStatements(existingStatements);
            }
            catch (Exception ex)
            {
                message = "Failed to update existing Statement: " + ex.getMessage();
                this.logger.error(message);

                return message;
            }

            updateElapsedTime = System.nanoTime() - updateStartTime;

            this.logger.debug("Updating existing Statements: " + existingStatements.size() + ". " + printElapsedTime("update", updateElapsedTime));
        }

        // For initial Statements, do save (insert)
        if (!initialStatements.isEmpty())
        {
            saveStartTime = System.nanoTime();

            long[] statementIDs = null;

            try
            {
                statementIDs = saveStatements(initialStatements);
            }
            catch (Exception ex)
            {
                message = "Failed to save initial Statements: " + ex.getMessage();
                this.logger.error(message);

                return message;
            }

            saveElapsedTime = System.nanoTime() - saveStartTime;

            this.logger.debug("Saving initial Statements: " + initialStatements.size() + ". " + printElapsedTime("save", saveElapsedTime));

            if (statementIDs == null || statementIDs.length != initialStatements.size())
            {
                message = "Saving initial Statements erred.";
                this.logger.error(message);

                return message;
            }

            // For initial Site Statements
            // For initial Statement, get initial Site Statements,
            // which do not have the associated Statement ID.
            for (int i = 0; i < statementIDs.length; i++)
            {
                final APSStmt statement = initialStatements.get(i);
                List<APSSiteStmt> siteStatements = initialEntries.get(i).getValue();
                siteStatements.forEach(p -> p.setStatementId(statement.getId()));

                initialSiteStatements.addAll(siteStatements);
            }
        }

        // For existing updated Site Statements, do update
        if (!updatedSiteStatements.isEmpty())
        {
            updateStartTime = System.nanoTime();

            try
            {
                updateSiteStatements(updatedSiteStatements);
            }
            catch (Exception ex)
            {
                message = "Failed to update Site Statements: " + ex.getMessage();
                this.logger.error(message);

                return message;
            }

            updateElapsedTime = System.nanoTime() - updateStartTime;

            this.logger.debug("Updating existing Site Statements: " + updatedSiteStatements.size() + ". " + printElapsedTime("update", updateElapsedTime));
        }

        // Add initial Site Statement from existing Statements
        // to the initial Site Statements collection.
        if (!initialSiteStmtsFromExistingStmts.isEmpty())
        {
            initialSiteStatements.addAll(initialSiteStmtsFromExistingStmts);
        }

        // For initial Site Statements, do save (insert)
        saveStartTime = System.nanoTime();

        long[] siteStatementIDs = null;

        try
        {
            siteStatementIDs = saveSiteStatements(initialSiteStatements);
        }
        catch (Exception ex)
        {
            message = "Failed to save Site Statements: " + ex.getMessage();
            this.logger.error(message);

            return message;
        }

        saveElapsedTime = System.nanoTime() - saveStartTime;

        this.logger.debug("Saving initial Site Statements: " + initialSiteStatements.size() + ". " + printElapsedTime("save", saveElapsedTime));

        if (siteStatementIDs == null || siteStatementIDs.length != initialSiteStatements.size())
        {
            message = "Saving initial SiteStatements erred.";
            this.logger.error(message);

            return message;
        }

        // Save Payments and Site Patients
        List<APSPayment> allPayments = new ArrayList<>();
        List<APSPayment> payments = null;
        List<APSSitePatient> allSitePatients = new ArrayList<>();
        APSSitePatient sitePatient = null;
        
        for (int i = 0; i < siteStatementIDs.length; i++)
        {
            final APSSiteStmt siteStatement = initialSiteStatements.get(i);
            payments = siteStatement.getPayments();
            sitePatient = siteStatement.getPatient();

            payments.forEach(d -> d.setSiteStmtId(siteStatement.getId()));
            sitePatient.setSiteStmtId(siteStatement.getId());

            allPayments.addAll(payments);
            allSitePatients.add(sitePatient);
        }

        saveStartTime = System.nanoTime();

        int[] numOfInserts = null;

        try
        {
            numOfInserts = savePayments(allPayments);
        }
        catch (Exception ex)
        {
            message = "Failed to save Payments: " + ex.getMessage();
            this.logger.error(message);

            return message;
        }

        saveElapsedTime = System.nanoTime() - saveStartTime;

        this.logger.debug("Saving Payments: " + allPayments.size() + ". " + printElapsedTime("save", saveElapsedTime));

        if (numOfInserts == null || numOfInserts.length != allPayments.size())
        {
            message = "Saving Payments erred.";
            this.logger.error(message);

            return message;
        }

        saveStartTime = System.nanoTime();

        try
        {
            numOfInserts = saveSitePatients(allSitePatients);
        }
        catch (Exception ex)
        {
            message = "Failed to save Site Patients: " + ex.getMessage();
            this.logger.error(message);

            return message;
        }

        saveElapsedTime = System.nanoTime() - saveStartTime;

        this.logger.debug("Saving Site Patients: " + allSitePatients.size() + ". " + printElapsedTime("save", saveElapsedTime));

        if (numOfInserts == null || numOfInserts.length != allSitePatients.size())
        {
            message = "Saving SitePatients erred.";
            this.logger.error(message);

            return message;
        }

        final long methodElapsedTime = System.nanoTime() - methodStartTime;

        this.logger.info("Service: Save CAPS Data Ended. " + printElapsedTime("method", methodElapsedTime));

        return message;
    }


    //
    // Other methods
    //
    public Result<List<APSSite>> getSitesForPatients(final List<APSPatient> patients)
    {
        Result<List<APSSite>> result = new Result<>();

        try
        {
            // Get the APSSite from database.
            List<APSSite> sites = this.apsSiteDAO.getByPatients(patients);

            if (sites != null && !sites.isEmpty())
            {
                result.set(sites);
                result.setSuccessful(true);
            }
            else
            {
                result.setMessage("Failed to get Sites.");
                result.setSuccessful(false);
            }
        }
        catch (Exception ex)
        {
            result.setMessage("Failed to get Sites for Patients: " + ex.getMessage());
            result.setSuccessful(false);
        }

        return result;
    }

    public Result<List<APSDetails>> getDetailsForPatients(final List<APSPatient> patients)
    {
        Result<List<APSDetails>> result = new Result<>();

        try
        {
            // Get the APSDetails from database.
            List<APSDetails> detailsList = this.apsDetailsDAO.getByPatients(patients);

            if (detailsList != null && !detailsList.isEmpty())
            {
                result.set(detailsList);
                result.setSuccessful(true);
            }
            else
            {
                result.setMessage("Failed to get Details.");
                result.setSuccessful(false);
            }
        }
        catch (Exception ex)
        {
            result.setMessage("Failed to get Details for Patients: " + ex.getMessage());
            result.setSuccessful(false);
        }

        return result;
    }

    /**
     * Update the NEW received data to PROCESSED status.
     * 
     * @return The number of received rows that were updated.
     */
    public String updateAPSReceivedStatus(final List<Long> inIds, final Status status)
    {
        String message = null;

        Long[] tempIds = inIds.toArray(new Long[inIds.size()]);
        long[] ids = Arrays.stream(tempIds).mapToLong(Long::longValue).toArray();

        try
        {
            int[] numOfUpdates = this.apsReceivedDAO.update(ids, status);
 
            if (numOfUpdates == null || numOfUpdates.length != ids.length)
            {
                message = "Update APSReceived erred.";
            }
        }
        catch (Exception ex)
        {
            message = "Update APSReceived status failed: " + ex.getMessage();
        }

        return message;
    }

    /**
     * Update APSStmt list.
     * 
     * @param statements
     *            The statements record.
     * @return The number of updated statements.
     */
    public int[] updateStatements(List<APSStmt> statements)
    {
        return this.apsStmtDAO.batchUpdate(statements);
    }

    /**
     * 
     * @param processId
     * @param statusId
     * @return
     */
    public String updateStatementsStatus(long processId, Status status)
    {
        String message = null;

        try
        {
            this.apsStmtDAO.updateStatusByGenProcessId(processId, status);
        }
        catch (Exception ex)
        {
            message = "Update Statement status failed: " + ex.getMessage();
        }

        return message;
    }

    /**
     * Save APSStmt list and update the database with auto generated IDs.
     * 
     * @param statements
     *            The statements record.
     * @return The updated statements populated with the auto generated IDs.
     */
    public long[] saveStatements(List<APSStmt> statements)
    {
        return this.apsStmtDAO.batchSave(statements);
    }

    /**
     * Save APSSiteStmt list and update the database with auto generated IDs.
     * 
     * @param siteStatements
     *            The siteStatements record.
     * @return The updated siteStatements populated with the auto generated IDs.
     */
    public long[] saveSiteStatements(List<APSSiteStmt> siteStatements)
    {
        return this.apsSiteStmtDAO.batchSave(siteStatements);
    }

    /**
     * Update APSSiteStmt list.
     * 
     * @param siteStatements
     *            The site statements record.
     * @return The number of updated site statements.
     */
    public int[] updateSiteStatements(List<APSSiteStmt> siteStatements)
    {
        return this.apsSiteStmtDAO.batchUpdate(siteStatements);
    }

    /**
     * Save APSSitePatient list and update the database with auto generated IDs.
     * 
     * @param sitePatients
     *            The sitePatients record.
     * @return The updated sitePatients populated with the auto generated IDs.
     */
    public int[] saveSitePatients(List<APSSitePatient> sitePatients)
    {
        return this.apsSitePatientDAO.batchInsert(sitePatients);
    }

    /**
     * Save an APSPayment list and update the database with auto generated IDs.
     * 
     * @param payments
     *            The payments record.
     * @return The updated payments populated with the auto generated IDs.
     */
    public int[] savePayments(List<APSPayment> payments)
    {
        return this.apsPaymentDAO.batchInsert(payments);
    }

    /**
     * Delete the statements via job process ID, which will cascading deleted all of its children.
     * 
     * @param processId
     *            The job process ID.
     */
    public String deleteStatements(long processId)
    {
        String message = null;

        try
        {
            this.apsStmtDAO.deleteByGenProcessId(processId);
        }
        catch (Exception ex)
        {
            message = "Delete Statements failed: " + ex.getMessage();
        }

        return message;
    }

    /**
     * Query the number of consolidated statements with the specific status.
     * 
     * @return The number of consolidated statements with the specific status
     *         or null if error.
     */
    public Long getStatementCountWithStatus(final long inGenProcessId, final Status inStatus)
    {
        Long count = null;

        count = this.apsStmtDAO.getGenCountWithStatus(inGenProcessId, inStatus);

        if (count == null)
        {
            this.logger.error("Invalid count when checking for count of statements");
        }

        return count;
    }

    /**
     * Create Payment list from Details list.
     * 
     * @param inDetailsList
     * @return
     * 
     */
    public Entry<AITCDollar, List<APSPayment>> createPayments(final List<APSDetails> inDetailsList)
    {
        if (inDetailsList == null || inDetailsList.isEmpty())
        {
            return null;
        }

        List<APSPayment> payments = new ArrayList<>();
        double totalAmount = 0.0;

        for (APSDetails details : inDetailsList)
        {
            totalAmount += details.getTransactionAmount().getDouble();

            APSPayment payment = new APSPayment(details);

            payments.add(payment);
        }

        return new AbstractMap.SimpleEntry<>(new AITCDollar(totalAmount), payments);
    }

    /**
     * Creates Site Statements from Site / Patient / Details data structure.
     * 
     * @param inSites
     * @return
     * 
     */
    public Result<List<APSSiteStmt>> createSiteStatements(final List<Entry<APSSite, List<APSPatient>>> inSites)
    {
        Result<List<APSSiteStmt>> result = new Result<>();

        if (inSites == null || inSites.isEmpty())
        {
            result.setMessage("Null or empty Sites and cannot create Site Statements without Sites.");
            result.setSuccessful(false);

            return result;
        }

        List<APSSiteStmt> siteStmts = new ArrayList<>();

        for (Entry<APSSite, List<APSPatient>> siteEntry : inSites)
        {
            APSSite site = siteEntry.getKey();
            List<APSPatient> patients = siteEntry.getValue();

            for (APSPatient patient : patients)
            {
                APSSiteStmt siteStmt = new APSSiteStmt();
                APSSitePatient sitePatient = new APSSitePatient(patient);
                Entry<AITCDollar, List<APSPayment>> payments = createPayments(patient.getDetailsList());

                // Set Site Statement
                // (The Primary and Primary Address will be set during the consolidation process.)
                siteStmt.setFacilityNum(site.getFacilityNum());
                siteStmt.setFacilityPhoneNum(site.getFacilityPhoneNum());
                siteStmt.setStatementDate(site.getStatementDate());
                siteStmt.setProcessDate(site.getProcessDate());
                siteStmt.setLastBillPrepDate(patient.getLastBillPrepDate());
                siteStmt.setArAddress(patient.isArAddress());
                siteStmt.setPatient(sitePatient);
                siteStmt.setTotalAmountReceived(payments.getKey());
                siteStmt.setPayments(payments.getValue());

                // Set status to INITIAL, so to distinguish from existing Site Statement,
                // and will be inserted later.
                siteStmt.setStatusId(this.statusIdInitial);

                // Add to the site statements list
                siteStmts.add(siteStmt);
            }
        }

        result.set(siteStmts);
        result.setSuccessful(true);

        return result;
    }

    /**
     * Create initial Statements from Site Statements.
     * 
     * @param inSiteStmts
     * @return
     * 
     */
    public Result<List<APSStmt>> createInitialStatements(final long batchProcessId, final List<APSSiteStmt> inSiteStmts)
    {
        Result<List<APSStmt>> result = new Result<>();

        if (inSiteStmts == null || inSiteStmts.isEmpty())
        {
            result.setSuccessful(false);
            result.setMessage("Null or empty Site Statements. Cannot create initial Statements without Site Satements.");

            return result;
        }

        List<APSStmt> stmts = new ArrayList<>();

        // Group Site Statements by Patient's ICN
        Map<String, List<APSSiteStmt>> groupedSiteStmts = inSiteStmts.stream().collect(Collectors.groupingBy(s -> s.getPatient().getIcn()));

        // Get CBSAccounts from ICN's
        //List<String> icns = new ArrayList<>(groupedSiteStmts.keySet());
        Set<String> icns = groupedSiteStmts.keySet();
        List<CBSAccount> cbsAccounts = this.cbsAccountDAO.batchSelect(Arrays.copyOf(icns.toArray(), icns.size(), String[].class));

        try
        {
            if (cbsAccounts == null || cbsAccounts.isEmpty())
            {
                // Create all missing CBSAccount
                cbsAccounts = this.cbsAccountDAO.batchInsertAndReturnCBSAccounts(icns);
            }
            else if (cbsAccounts.size() != icns.size())
            {
                Set<String> icnsWithoutCBSAccounts = new HashSet<>(); 

                for (String icn : icns)
                {
                    List<CBSAccount> tmpCBSAccounts = cbsAccounts.stream().filter(a -> a.getIcn().equals(icn)).collect(Collectors.toList());
    
                    // The ICN has no CBSAccount
                    if (tmpCBSAccounts == null || tmpCBSAccounts.isEmpty())
                    {
                        icnsWithoutCBSAccounts.add(icn);
                    }
                }

                // Create missing CBSAccount for ICN's without CBSAccounts
                List<CBSAccount> missingCBSAccounts = this.cbsAccountDAO.batchInsertAndReturnCBSAccounts(icnsWithoutCBSAccounts);
    
                cbsAccounts.addAll(missingCBSAccounts);
            }
        }
        catch (Exception ex)
        {
            result.setMessage("Failed to insert CBSAccount: " + ex.getMessage());
            result.setSuccessful(false);

            return result;
        }

        for (Entry<String, List<APSSiteStmt>> siteStmtEntry : groupedSiteStmts.entrySet())
        {
            String icn = siteStmtEntry.getKey();
            long accountId = cbsAccounts.stream().filter(a -> a.getIcn().equals(icn)).collect(Collectors.toList()).get(0).getId();
            List<APSSiteStmt> siteStmts = siteStmtEntry.getValue();

            // Set Primary and Primary Address on Site Statements.
            setPrimaryAndPrimaryAddress(siteStmts);

            APSStmt stmt = new APSStmt();

            // Set Statement
            stmt.setAccountId(accountId);
            stmt.setSiteStmts(siteStmts);
            stmt.setGenBatchRunId(batchProcessId);

            // Set status to INITIAL, so will be inserted later.
            stmt.setStatusId(this.statusIdInitial);

            stmts.add(stmt);
        }

        result.set(stmts);
        result.setSuccessful(true);

        return result;
    }

    /**
     * Consolidate Statements.
     * 
     * @param inStmts
     * @return
     * 
     */
    @SuppressWarnings("null")
    public Result<List<APSStmt>> consolidateStatements(final List<APSStmt> inStmts)
    {
        Result<List<APSStmt>> result = new Result<>();

        if (inStmts == null || inStmts.isEmpty())
        {
            result.setSuccessful(false);
            result.setMessage("Null or empty Statements.");

            return result;
        }

        List<APSStmt> consolidatedStmts = new ArrayList<>();
        List<APSStmt> allExistingStmts = null;
        List<APSSiteStmt> allExistingSiteStmts = null;

        try
        {
            // Check to see if Statements already exist.
            allExistingStmts = this.apsStmtDAO.getExistingStatements(inStmts);

            // If Statements exist, get the corresponding Site Statements
            if (allExistingStmts != null && !allExistingStmts.isEmpty())
            {
                Long[] stmtIDs = new Long[allExistingStmts.size()];
                int i = 0;

                for (APSStmt existingStmt : allExistingStmts)
                {
                    stmtIDs[i] = Long.valueOf(existingStmt.getId());
                    i++;
                }

                allExistingSiteStmts = this.apsSiteStmtDAO.getExistingSiteStatements(stmtIDs);
            }
        }
        catch (Exception ex)
        {
            result.setMessage("Failed to get existing Statements or Site Statements: " + ex.getMessage());
            result.setSuccessful(false);

            return result;
        }

        // Consolidate Statements / Site Statements
        for (APSStmt inStmt : inStmts)
        {
            List<APSSiteStmt> existingSiteStmts = null;
            List<APSSiteStmt> consolidateSiteStmts = null;
            double totalAmountReceived = 0.0;

            // Consolidate from existing Statements (allExistingStmts).
            if (allExistingStmts != null && !allExistingStmts.isEmpty())
            {
                List<APSStmt> existingStmts = allExistingStmts.stream().filter(s -> s.getAccountId() == inStmt.getAccountId()).collect(Collectors.toList());

                if (existingStmts != null && !existingStmts.isEmpty())
                {
                    // Should only contain one Statement
                    APSStmt existingStmt = existingStmts.get(0);

                    // Get existing Site Statements for this existing Statement
                    // If there is existing Statement, then the Site Statement should / will not be null. 
                    existingSiteStmts = allExistingSiteStmts.stream().filter(s -> s.getStatementId() == existingStmt.getId()).collect(Collectors.toList());

                    // Get the Primary, Primary Address, and AR Address Site Statements from existing Site Statements, if any, for later consolidation.
                    consolidateSiteStmts = existingSiteStmts.stream().filter(s -> s.isPrimary() || s.isPrimaryAddress() || s.isArAddress()).collect(Collectors.toList());

                    // Get Total Amount Received from the existing Statement - already consolidated from its associated existing Site Statements.
                    totalAmountReceived = existingStmt.getTotalAmountReceived().getDouble();

                    // Set Statement ID from existing Statement
                    inStmt.setId(existingStmt.getId());

                    // Set Statement ID on initial Site Statements
                    inStmt.getSiteStmts().stream().forEach(s -> s.setStatementId(existingStmt.getId()));

                    // Set dates from existing Statement
                    inStmt.setStatementDate(existingStmt.getStatementDate());
                    inStmt.setProcessDate(existingStmt.getProcessDate());

                    // Set Status to OTHER for existing Statements, so will be updated later.
                    inStmt.setStatusId(this.statusIdOther);
                } // If - Existing Statement
            } // If - All existing Statements


            //
            // Consolidate / combined with initial Statements (inStmts).
            //

            // Calculate the total amount received from all (existing + initial) Sites.
            // The for-loop here is for initial Site Statements
            // (before existing Site Statements are added).
            for (APSSiteStmt siteStmt : inStmt.getSiteStmts())
            {
                totalAmountReceived += siteStmt.getTotalAmountReceived().getDouble();
            } // For

            // Set Statement
            inStmt.setTotalAmountReceived(new AITCDollar(totalAmountReceived));

            // Add existing Primary, Primary Address, and AR Address Site Statements,
            // if any, to the Site Statements collection,
            // so that the consolidation will be done on all Site Statements.
            if (consolidateSiteStmts != null && !consolidateSiteStmts.isEmpty())
            {
                inStmt.getSiteStmts().addAll(consolidateSiteStmts);
            }

            setPrimaryAndPrimaryAddress(inStmt.getSiteStmts());

            // Get primary Site Statement from all Site Statements, existing and initial.
            APSSiteStmt primarySiteStmt = inStmt.getPrimarySiteStmt();

            if (primarySiteStmt != null)
            {
                inStmt.setStatementDate(primarySiteStmt.getStatementDate());
                inStmt.setProcessDate(primarySiteStmt.getProcessDate());
            }

            consolidatedStmts.add(inStmt);
        } // For - Statements

        result.set(consolidatedStmts);
        result.setSuccessful(true);

        return result;
    }


    /**
     * Set Primary site and Primary Address site.
     * 
     * @param siteStmts
     * @return list of updated SiteStatements
     * 
     */
    private void setPrimaryAndPrimaryAddress(final List<APSSiteStmt> siteStmts)
    {
        // Get the latest LastBillPrepDate within all Site Statements
        Date lastBillPrepDate = siteStmts.stream().map(APSSiteStmt::getLastBillPrepDate).max(Date::compareTo).get();

        // Determine multiple AR Address flags
        boolean isMultipleArAddress = (siteStmts.stream().filter(s -> s.isArAddress()).collect(Collectors.toList()).size() > 1);

        // Determine Primary site and Primary Address site
        for (APSSiteStmt siteStmt : siteStmts)
        {
            boolean isPUpdated = false;
            boolean isPAUpdated = false;

            // Set Primary flag
            if (siteStmt.getLastBillPrepDate().compareTo(lastBillPrepDate) < 0)
            {
                isPUpdated = siteStmt.isPrimary();

                siteStmt.setPrimary(false);
            }
            else
            {
                isPUpdated = !siteStmt.isPrimary();

                siteStmt.setPrimary(true);
            } // if-else

            // Set Primary Address flag
            // Single Site Statement
            if (siteStmts.size() == 1)
            {
                siteStmt.setPrimaryAddress(true);
            }
            // Multiple Site Statements
            else
            {
                // AR Address flag exists AND Multiple AR Address flags not exist.
                if (siteStmt.isArAddress() && !isMultipleArAddress)
                {
                    isPAUpdated = !siteStmt.isPrimaryAddress();

                    siteStmt.setPrimaryAddress(true);
                }
                // AR Address flag not exists OR
                // AR Address flag exists AND Multiple AR Address flags exist.
                else
                {
                    if (siteStmt.isPrimary())
                    {
                        isPAUpdated = !siteStmt.isPrimaryAddress();

                        siteStmt.setPrimaryAddress(true);
                    }
                    else
                    {
                        isPAUpdated = siteStmt.isPrimaryAddress();

                        siteStmt.setPrimaryAddress(false);
                    } // if-else - Primary
                } // if-else - AR Address
            } // if-else - Single - Multiple Site Statements
 
            if ( (isPUpdated || isPAUpdated) &&
                 this.statusIdInitial != siteStmt.getStatusId() )
            {
                // Set status to OTHER, so it will be updated later.
                siteStmt.setStatusId(this.statusIdOther);
            }
        } // for
    }
}
