package gov.va.cpss.service;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.CBSSitePatientDAO;
import gov.va.cpss.dao.CBSSiteStmtDAO;
import gov.va.cpss.dao.CBSSiteTransDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;

/**
 * Service class for handling activities relating to consolidating a patient
 * statement.
 * 
 * @author DNS   
 */
@Service
@SuppressWarnings("nls")
public class CbsService {

	private static final Logger logger = Logger.getLogger(CbsService.class.getCanonicalName());

	private static final int MAX_COBOL_FIELD_LENGTH = 9;

	private boolean lateStatementMessage;

	private ProcessStatusDAO processStatusDAO;

	private CBSAccountDAO cbsAccountDAO;

	private CBSStmtDAO cbsStmtDAO;

	private CBSSiteStmtDAO cbsSiteStmtDAO;

	private CBSSitePatientDAO cbsSitePatientDAO;

	private CBSSiteTransDAO cbsSiteTransDAO;

	public CbsService() {
	}

	public boolean getLateStatementMessage() {
		return lateStatementMessage;
	}

	public void setLateStatementMessage(boolean lateStatementMessage) {
		this.lateStatementMessage = lateStatementMessage;
	}

	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	public CBSAccountDAO getCbsAccountDAO() {
		return cbsAccountDAO;
	}

	public void setCbsAccountDAO(CBSAccountDAO cbsAccountDAO) {
		this.cbsAccountDAO = cbsAccountDAO;
	}

	public CBSStmtDAO getCbsStmtDAO() {
		return cbsStmtDAO;
	}

	public void setCbsStmtDAO(CBSStmtDAO cbsStmtDAO) {
		this.cbsStmtDAO = cbsStmtDAO;
	}

	public CBSSitePatientDAO getCbsSitePatientDAO() {
		return cbsSitePatientDAO;
	}

	public void setCbsSitePatientDAO(CBSSitePatientDAO cbsSitePatientDAO) {
		this.cbsSitePatientDAO = cbsSitePatientDAO;
	}

	public CBSSiteStmtDAO getCbsSiteStmtDAO() {
		return cbsSiteStmtDAO;
	}

	public void setCbsSiteStmtDAO(CBSSiteStmtDAO cbsSiteStmtDAO) {
		this.cbsSiteStmtDAO = cbsSiteStmtDAO;
	}

	public CBSSiteTransDAO getCbsSiteTransDAO() {
		return cbsSiteTransDAO;
	}

	public void setCbsSiteTransDAO(CBSSiteTransDAO cbsSiteTransDAO) {
		this.cbsSiteTransDAO = cbsSiteTransDAO;
	}

	/**
	 * Build a consolidated statement for the CBS Account to list of PSPatient
	 * records map.
	 * 
	 * @param batchRunId
	 *            The batch run ID.
	 * @param patientMap
	 *            The map of records to consolidate.
	 * @return A consolidated statement.
	 */
	public CBSStmt buildStatementForPatientMap(final long batchRunId, Entry<Long, List<PSPatient>> patientMap)
			throws RuntimeException {

		// First, create an initial statement for the records in the map.
		CBSStmt statement = createInitialStatement(batchRunId, patientMap.getKey());

		// Then process the list of PSPatient records.
		if (processPatientList(statement, patientMap.getValue()) > 0) {

			// Now check if this should be a late statement or if a new
			// statement already exists for this patient.
			// If an unrecoverable error occurs false will be returned.
			if (!checkForExistingStatement(statement)) {

				// An unrecoverable error has occurred indicating the database
				// is corrupted.
				throw new RuntimeException("Unrecoverable data error checking for existing statement");
			}

		} else {

			// If unexpected data error occurs (patient list is empty or no
			// patients processed) then log an error message and
			// skip the statement.
			logger.error("No patients processed for CBS Account: " + patientMap.getKey());

			// Return null which will skip statement.
			statement = null;
		}

		// Set accntNumDisp after site patients have been processed and
		// statement merged with existing statement
		setAcntNumDisp(statement);

		return statement;
	}

	/**
	 * Check and handle existing statement.
	 * 
	 * @param statement
	 *            The consolidated statement.
	 * @return Flag indicating if successfully checked or if an unrecoverable
	 *         error occurred.
	 */
	public boolean checkForExistingStatement(CBSStmt statement) {

		boolean successful = false;

		// Check if this should be a late statement (a statement for this month
		// already) or if a new statement
		// already exists that needs to be merged.
		final Date statementDate = statement.getSiteStmtL().get(0).getStatementDate();

		// Unrecoverable error if statement date happens to be null.
		if (statementDate != null) {

			Integer statusId = cbsStmtDAO.getStatusIdForExistingStatementByAccountAndMonth(statement.getAccountId(),
					statementDate);

			// If an existing statement exists (!null) then process accordingly.
			if (statusId != null) {

				// Handle existing statement differently depending on its
				// status.
				switch (processStatusDAO.getStatusType(statusId).getStatus()) {

				case NEW:
					// Query for existing statement.
					CBSStmt existingStatement = getExistingStatementForAccount(statement.getAccountId(), statementDate);

					combineWithExistingStatement(statement, existingStatement);
					successful = true;
					break;
				case ACK:
				case ERROR:
				case SENT:
					setAsLateStatement(statement);
					successful = true;
					break;
				case INITIAL:
				case REPLACED:
				case UNDEFINED:
				default:
					logger.error("Existing statement detected of invalid status at time of query");
					break;
				}
			} else {

				// There is no existing statement found.
				successful = true;
			}
		} else {

			logger.error("Unable to check for existing statement because date check was unexpectedly null");
		}

		return successful;
	}

	/**
	 * Create a consolidated statement object for the specified account number.
	 * 
	 * @param accountNumber
	 *            The account number to associate with the consolidated
	 *            statement.
	 * @return The initialized consolidated statement.
	 */
	private CBSStmt createInitialStatement(final long batchRunId, final long accountNumber) {

		// Create a statement for the account.
		CBSStmt statement = new CBSStmt(batchRunId, accountNumber,
				processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL));

		initializeStatement(statement);

		return statement;
	}

	/**
	 * Combine the consolidated statement with the specified existing statement.
	 * 
	 * @param statement
	 *            The consolidated statement.
	 * @param existingStatement
	 *            The existing consolidated statement.
	 */
	public void combineWithExistingStatement(CBSStmt statement, final CBSStmt existingStatement) {

		// Save the replaces id.
		statement.setReplacedStmtId(existingStatement.getId());

		// Initialize the running totals with the values already in the
		// statement.
		double amountDue = statement.getAmountDue().getDouble();
		double previousBalance = statement.getPrevBalance().getDouble();
		double totalCharges = statement.getTotalCharges().getDouble();
		double totalCredits = statement.getTotalCredits().getDouble();
		double newBalance = statement.getNewBalance().getDouble();

		// Add the existing statement values.
		amountDue += existingStatement.getAmountDue().getDouble();
		previousBalance += existingStatement.getPrevBalance().getDouble();
		totalCharges += existingStatement.getTotalCharges().getDouble();
		totalCredits += existingStatement.getTotalCredits().getDouble();
		newBalance += existingStatement.getNewBalance().getDouble();

		// Save the values.
		setAmountDue(statement, amountDue);
		setPrevBalance(statement, previousBalance);
		setTotalCharges(statement, totalCharges);
		setTotalCredits(statement, totalCredits);
		setNewBalance(statement, newBalance);

		// Append the existing list of site statements to the new statement.
		List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();
		siteStatementL.addAll(existingStatement.getSiteStmtL());
		setPrimarySiteFlags(statement, siteStatementL);
		statement.setSiteStmtL(siteStatementL);

		// Take the value of the existing statement late statement message.
		statement.setLateStmtMsg(existingStatement.getLateStmtMsg());
	}

	/**
	 * Set the late message in the statement.
	 * 
	 * @param statement
	 *            The statement that is late.
	 */
	private void setAsLateStatement(CBSStmt statement) {
		statement.setLateStmtMsg(lateStatementMessage);
	}

	/**
	 * Initialize a consolidated statement values.
	 * 
	 * @param statement
	 */
	private void initializeStatement(CBSStmt statement) {

		final Timestamp currentTime = new Timestamp(Calendar.getInstance().getTime().getTime());

		statement.setSiteStmtL(new ArrayList<>());

		setAmountDue(statement, 0);
		setPrevBalance(statement, 0);
		setTotalCharges(statement, 0);
		setTotalCredits(statement, 0);
		setNewBalance(statement, 0);

		statement.setCreatedDate(currentTime);
		statement.setModifiedDate(currentTime);

		statement.setLateStmtMsg(false);
	}

	public void setAmountDue(CBSStmt statement, final double value) {
		statement.setAmountDue(new Money(value, MAX_COBOL_FIELD_LENGTH));
	}

	public void setPrevBalance(CBSStmt statement, final double value) {
		statement.setPrevBalance(new Money(value, MAX_COBOL_FIELD_LENGTH));
	}

	public void setTotalCharges(CBSStmt statement, final double value) {
		statement.setTotalCharges(new Money(value, MAX_COBOL_FIELD_LENGTH));
	}

	public void setTotalCredits(CBSStmt statement, final double value) {
		statement.setTotalCredits(new Money(value, MAX_COBOL_FIELD_LENGTH));
	}

	public void setNewBalance(CBSStmt statement, final double value) {
		statement.setNewBalance(new Money(value, MAX_COBOL_FIELD_LENGTH));
	}

	public void setAcntNumDisp(CBSStmt statement) {
		statement.setAcntNumDisp(CBSStmt.createAcntNumDisp(statement));
	}

	/**
	 * Identify the appropriate site statements for primary patient data and
	 * primary patient address.
	 * 
	 * @param siteStatementL
	 *            The list of site statements for which to calculate primary
	 *            flags.
	 */
	public void setPrimarySiteFlags(CBSStmt statement, List<CBSSiteStmt> siteStatementL) {

		// If there is only one site statement then this is easy.
		if (siteStatementL.size() == 1) {

			siteStatementL.get(0).setIsPrimary(BooleanChar.Y);
			siteStatementL.get(0).setIsPrimaryAddress(BooleanChar.Y);

			// Set the statement dates to the dates of the primary site
			// statement.
			setStatementDates(statement, siteStatementL.get(0));

		} else {

			// If there are multiple site statements then it is more
			// complicated.

			// First, initialize all the sites to not primary.
			// While doing so, count the number of AR flags.
			int arFlagCount = 0;
			for (CBSSiteStmt siteStatement : siteStatementL) {

				siteStatement.setIsPrimary(BooleanChar.N);
				siteStatement.setIsPrimaryAddress(BooleanChar.N);

				// Count how many sites have the AR flag set.
				if (siteStatement.getArAddressFlag().isTrue()) {
					arFlagCount++;
				}
			}

			// Sort statements by most recent lastBillPrepDate to identify
			// isPrimary
			Collections.sort(siteStatementL, new Comparator<CBSSiteStmt>() {
				public int compare(CBSSiteStmt o1, CBSSiteStmt o2) {
					if (o1.getLastBillPrepDate() == null) {
						// If o1 null, then assume older.
						return 1;
					} else if (o2.getLastBillPrepDate() == null) {
						// If o2 is then assume o1 is newer.
						return -1;
					} else {
						// Desire most recent date, so compare o2 to o1.
						return o2.getLastBillPrepDate().compareTo(o1.getLastBillPrepDate());
					}
				}
			});

			// The most recent will be sorted first.
			// It will be used for Patient Details.
			// Set the first in the collection to isPrimary for Patient Details.
			siteStatementL.get(0).setIsPrimary(BooleanChar.Y);

			// Set the statement dates to the dates of the primary site
			// statement.
			setStatementDates(statement, siteStatementL.get(0));

			// Use different logic to identify the primary site statement to use
			// for the Patient
			// Address

			if (arFlagCount == 1) {

				// If exactly one AR flag found then use that site statement as
				// the Primary Address.
				for (CBSSiteStmt siteStatement : siteStatementL) {
					if (siteStatement.getArAddressFlag().isTrue()) {
						siteStatement.setIsPrimaryAddress(BooleanChar.Y);
						break;
					}
				}

			} else {

				// If there is more than one AR flag or if there are no AR flags
				// then use the first in the sorted collection just like how
				// Patient
				// Details is determined.
				siteStatementL.get(0).setIsPrimaryAddress(BooleanChar.Y);
			}
		}
	}

	/**
	 * Set the statement dates using the primary site statement date values.
	 * 
	 * @param statement
	 *            The statement to update.
	 * @param siteStatement
	 *            The site statement as reference.
	 */
	private void setStatementDates(CBSStmt statement, final CBSSiteStmt siteStatement) {
		// Set the statement values to point to the primary dates.
		statement.setStatementDate(siteStatement.getStatementDate());
		statement.setProcessDate(siteStatement.getProcessDate());
	}

	/**
	 * Process the list of raw PSPatient records and roll processed data into
	 * the consolidated statement.
	 * 
	 * @param statement
	 *            The consolidated statement to update.
	 * @param psPatientL
	 *            The list of PSPatient records.
	 * @return Number of patients processed.
	 */
	public int processPatientList(CBSStmt statement, final List<PSPatient> psPatientL) {

		int processedCount = 0;

		if ((psPatientL == null) || psPatientL.isEmpty()) {

			// Unexpected error that should never happen.
			logger.error("Patient list was unexpectedly empty!");

		} else {

			// Get the site statement list.
			List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();

			// Initialize the running totals with the values already in the
			// statement.
			double amountDue = statement.getAmountDue().getDouble();
			double previousBalance = statement.getPrevBalance().getDouble();
			double totalCharges = statement.getTotalCharges().getDouble();
			double totalCredits = statement.getTotalCredits().getDouble();
			double newBalance = statement.getNewBalance().getDouble();

			for (PSPatient psPatient : psPatientL) {
				logger.debug("Processing Patient: " + psPatient.getIcnNumber());

				// Create CBSSiteStmt associated with the PSPatient record.
				CBSSiteStmt siteStatement = getSiteStatementForPatient(psPatient);

				// Calculate Totals for Consolidated Statement
				amountDue += siteStatement.getAmountDue().getDouble();
				previousBalance += siteStatement.getPrevBalance().getDouble();
				totalCharges += siteStatement.getTotalCharges().getDouble();
				totalCredits += siteStatement.getTotalCredits().getDouble();
				newBalance += siteStatement.getNewBalance().getDouble();

				// Append the site statement to the list.
				siteStatementL.add(siteStatement);

				// Increment the number of patients processed.
				processedCount++;
			}

			if (!siteStatementL.isEmpty()) {
				// Identify Primary Site (Patient Details) and Primary Site
				// (Address)
				setPrimarySiteFlags(statement, siteStatementL);

				// Set the site statement list in the Consolidated Statement
				statement.setSiteStmtL(siteStatementL);

				// Set the calculated totals.
				setAmountDue(statement, amountDue);
				setPrevBalance(statement, previousBalance);
				setTotalCharges(statement, totalCharges);
				setTotalCredits(statement, totalCredits);
				setNewBalance(statement, newBalance);
			}
		}

		return processedCount;
	}

	/**
	 * Build a site statement associated with the patient record.
	 * 
	 * @param psPatient
	 *            The patient record.
	 * @return A site statement associated with the specified patient record.
	 */
	private CBSSiteStmt getSiteStatementForPatient(final PSPatient psPatient) {

		final Timestamp currentTime = new Timestamp(Calendar.getInstance().getTime().getTime());

		CBSSiteStmt siteStatement = new CBSSiteStmt();
		siteStatement.setStationNum(psPatient.getPsSite().getFacilityNum());
		siteStatement.setStationPhoneNum(psPatient.getPsSite().getFacilityPhoneNum());
		siteStatement.setProcessDate(psPatient.getPsSite().getProcessDate());
		siteStatement.setStatementDate(psPatient.getPsSite().getStatementDate());
		siteStatement.setAmountDue(psPatient.getAmountDue());
		siteStatement.setPrevBalance(psPatient.getPrevBalance());
		siteStatement.setTotalCharges(psPatient.getTotalCharges());
		siteStatement.setTotalCredits(psPatient.getTotalCredits());
		siteStatement.setNewBalance(psPatient.getNewBalance());
		siteStatement.setSpecialNotes(psPatient.getSpecialNotes());
		siteStatement.setNoParaCdes(psPatient.getRightsObligationParagraphCodes());
		siteStatement.setLargeFontInd(psPatient.getLargeFontInd());
		siteStatement.setArAddressFlag(psPatient.getAddressFlag());
		siteStatement.setLastBillPrepDate(psPatient.getLastBillPrepDate());
		siteStatement.setIsPrimary(BooleanChar.N);
		siteStatement.setIsPrimaryAddress(BooleanChar.N);
		siteStatement.setCreatedDate(currentTime);
		siteStatement.setModifiedDate(currentTime);

		CBSSitePatient sitePatient = new CBSSitePatient();
		sitePatient.setIcn(psPatient.getIcnNumber());
		sitePatient.setDfn(psPatient.getDfnNumber());
		sitePatient.setOldAcntNum(psPatient.getPatientAccount());
		sitePatient.setFirstName(psPatient.getPatientFirstName());
		sitePatient.setLastName(psPatient.getPatientLastName());
		sitePatient.setMiddleName(psPatient.getPatientMiddleName());
		sitePatient.setAddress1(psPatient.getAddress1());
		sitePatient.setAddress2(psPatient.getAddress2());
		sitePatient.setAddress3(psPatient.getAddress3());
		sitePatient.setCity(psPatient.getCity());
		sitePatient.setState(psPatient.getState());
		sitePatient.setZipCode(psPatient.getZipCode());
		sitePatient.setCountry(psPatient.getCountryName());
		sitePatient.setCreatedDate(currentTime);
		sitePatient.setModifiedDate(currentTime);
		siteStatement.setSitePatient(sitePatient);

		List<CBSSiteTrans> siteTransL = new ArrayList<>();
		for (PSDetails details : psPatient.getPsDetailsL()) {

			CBSSiteTrans siteTrans = new CBSSiteTrans();
			siteTrans.setDatePosted(details.getDatePosted());
			siteTrans.setTransDesc(details.getTransDesc());
			siteTrans.setTransAmount(details.getTransAmount());
			siteTrans.setReferenceNum(details.getReferenceNum());
			siteTrans.setOrderNum(details.getSeqNum());
			siteTrans.setCreatedDate(currentTime);
			siteTrans.setModifiedDate(currentTime);
			siteTransL.add(siteTrans);
		}
		siteStatement.setSiteTransL(siteTransL);

		return siteStatement;
	}

	/**
	 * Save an statement and update the database auto generated ID.
	 * 
	 * @param statement
	 *            The consolidated statement.
	 * @return The updated statement populated with the auto generated ID.
	 */
	public void saveStatement(CBSStmt statement) {

		statement.setId(cbsStmtDAO.save(statement));
	}

	/**
	 * Save an site patient and update the database auto generated ID.
	 * 
	 * @param sitePatient
	 *            The site patient.
	 * @return The updated site patient populated with the auto generated ID.
	 */
	public void savePatient(CBSSitePatient sitePatient) {

		sitePatient.setId(cbsSitePatientDAO.save(sitePatient));
	}

	/**
	 * Save an site statement and update the database auto generated ID.
	 * 
	 * @param siteStatement
	 *            The site statement.
	 * @return The updated site statement populated with the auto generated ID.
	 */
	public void saveSiteStatement(CBSSiteStmt siteStatement) {

		siteStatement.setId(cbsSiteStmtDAO.save(siteStatement, true));
	}

	/**
	 * Save the List of CBSStmts. The nested CBSSiteStmts CBSSitePatients and
	 * CBSSiteTrans objects will also be saved (deep save). The auto generated
	 * IDs will NOT be updated in the objects. If the ids are required, then use
	 * the other individual saveXXX() methods.
	 * 
	 * @param statementL
	 *            List of CBSStmts to save
	 */
	public void batchSaveStatements(List<CBSStmt> statementL) {
		cbsStmtDAO.saveBatch(statementL);

		// cbsSiteStmtDAO.saveBatch(statementL);
		for (CBSStmt cbsStmt : statementL) {
			List<CBSSiteStmt> cbsSiteStmtL = cbsStmt.getSiteStmtL();
			
			cbsStmt.setId(cbsStmtDAO.getCBSStmtId(cbsStmt.getBatchRunId(), cbsStmt.getAccountId()));;

			if (cbsSiteStmtL != null) {
				for (CBSSiteStmt cbsSiteStmt : cbsSiteStmtL) {
					cbsSiteStmt.setStmtId(cbsStmt.getId());
					cbsSiteStmtDAO.save(cbsSiteStmt, false);
				}
			}
		}

		cbsSitePatientDAO.saveBatch(statementL);
		cbsSiteTransDAO.saveBatch(statementL);
	}

	/**
	 * Query the database for an existing statement for this account for the
	 * same month as the specified date.
	 * 
	 * @param accountNumber
	 *            The account number for the statement.
	 * @param statementDate
	 *            The date for this month.
	 * @return The existing statement or null if does not exist.
	 */
	private CBSStmt getExistingStatementForAccount(final long accountNumber, final Date statementDate) {

		CBSStmt statement = cbsStmtDAO.getExistingStatementByAccountAndMonth(accountNumber, statementDate);

		if (statement != null) {

			// Then get all CBSSiteStmt from database.
			List<CBSSiteStmt> siteStatementL = cbsSiteStmtDAO.getAllByCBSStmtID(statement.getId(), true);

			// Then get all CBSSiteTrans from database.
			if ((siteStatementL != null) && !siteStatementL.isEmpty()) {

				List<CBSSiteTrans> siteTransL = cbsSiteTransDAO.getAllByCBSSiteStmtID(statement.getId());

				// Sort into a map (CBSSiteStmt.ID to List<CBSSiteTrans>) for
				// the
				// list of CBSSiteTrans.
				Map<Long, List<CBSSiteTrans>> transQueryM = siteTransL.stream()
						.collect(Collectors.groupingBy(w -> w.getSiteStmtId()));

				// Set the appropriate trans list in the CBSSiteStmt.
				for (CBSSiteStmt siteStmt : siteStatementL) {

					logger.debug("SITESTMT.ID: " + siteStmt.getId());

					List<CBSSiteTrans> siteTransMapEntry = transQueryM.get(siteStmt.getId());
					// If no list then an unrecoverable data error.
					if (siteTransMapEntry != null) {

						siteStmt.setSiteTransL(siteTransMapEntry);
					} else {
						final String error = "Unrecoverable data error, site trans map is null for site";
						logger.error(error);
						throw new RuntimeException(error);
					}
				}

				// Add the site statement list to the statement.
				statement.setSiteStmtL(siteStatementL);
			} else {
				final String error = "Unrecoverable data error, site statement list is null or empty";
				logger.error(error);
				throw new RuntimeException(error);
			}
		}

		return statement;
	}

	/**
	 * Publish the INITIAL statements to NEW status.
	 * 
	 * @return The number of statements that were updated.
	 */
	public int publishAllNewStatements() {

		Integer initialStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		Integer newStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);

		if ((initialStatus != null) && (newStatus != null)) {

			return cbsStmtDAO.updateAllStatusFromTo(initialStatus, newStatus);
		} else {

			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.INITIAL + " or "
					+ ProcessStatus.Status.NEW);
		}

		return 0;
	}

	/**
	 * Revert the REPLACED statements to NEW status that are associated with
	 * INITIAL statements.
	 * 
	 * @return The number of statements that were updated.
	 */
	public int revertReplacedStatementsAssociatedWithInitialStatements() {

		int revertedCount = -1;

		Integer initialStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		Integer replacedStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.REPLACED);
		Integer newStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);

		if ((initialStatus != null) && (replacedStatus != null) && (newStatus != null)) {

			revertedCount = cbsStmtDAO.updateStatusFromToForInitial(replacedStatus, newStatus, initialStatus);
		} else {

			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.INITIAL + " or "
					+ ProcessStatus.Status.REPLACED + " or " + ProcessStatus.Status.NEW);
		}

		return revertedCount;
	}

	/**
	 * Remove statements that have INITIAL status.
	 * 
	 * @return The number of statements that were updated.
	 */
	public int removeInitialStatements() {

		int removedCount = -1;

		Integer initialStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);

		if (initialStatus != null) {

			removedCount = cbsStmtDAO.deleteByStatusId(initialStatus);
		} else {

			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.INITIAL);
		}

		return removedCount;
	}

	/**
	 * Query the number of consolidated statements in the NEW state.
	 * 
	 * @return The number of consolidated statements in the NEW state or null if
	 *         error.
	 */
	public Long getCountOfStatementsInState(final ProcessStatus.Status state) {

		Long availableCount = null;

		Integer status = processStatusDAO.getStatusFromEnum(state);

		if (status != null) {

			availableCount = cbsStmtDAO.getStatementCountWithStatus(status);

			if (availableCount == null) {

				logger.error("Invalid count when checking for count of statements");
			}

		} else {

			logger.error("Unable to obtain status mapping for: " + state);
		}

		return availableCount;
	}

	//
	// Fix for Defect 616440
	// Yiping Yao - 01/03/2018
	//
	// Return a list of CBSStmt for a CBS ID within the statement date range and print date.
	//
	public List<CBSStmt> getStatements(long cbsId)
	{
		// Print Delay Days = 3
		// Start Date is 5/1/PreviousYear to End Date 4/30/CurrentYear
		int delayDays = 3;

		//Date startDate = null;
		//Date endDate = null;
		//List<CBSStmt> cbsStmtL = getCbsStmtDAO().getListOfStmntsById(cbsId, delayDays, startDate, endDate);

		int startMonth = 4; // April
		int totalMonths = 12;

		return getCbsStmtDAO().getListOfStmntsById(cbsId, delayDays, startMonth, totalMonths);
	}
}
