package gov.va.cpss.service;

import static gov.va.cpss.ESAPI.EsapiValidationType.LOG_FORGING;
import static gov.va.cpss.ESAPI.EsapiValidator.validateStringInput;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import gov.va.cpss.dao.BatchConfigDAO;
import gov.va.cpss.dao.BatchJobDAO;
import gov.va.cpss.dao.BatchRunDAO;
import gov.va.cpss.dao.BatchSrcTypeDAO;
import gov.va.cpss.dao.BatchStatusDAO;
import gov.va.cpss.dao.BatchTypeDAO;
import gov.va.cpss.model.BatchConfig;
import gov.va.cpss.model.BatchItem;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchSrcType;
import gov.va.cpss.model.BatchStatus;
import gov.va.cpss.model.BatchType;

/**
 * Service class for handling activities relating to running batch jobs.
 * 
 * @author DNS  
 */
@Service
public class BatchService {

	private Logger logger = Logger.getLogger(BatchService.class.getCanonicalName());

	private BatchTypeDAO batchTypeDAO;
	private BatchJobDAO batchJobDAO;
	private BatchRunDAO batchRunDAO;
	private BatchStatusDAO batchStatusDAO;
	private BatchSrcTypeDAO batchSrcTypeDAO;
	private BatchConfigDAO batchConfigDAO;

	public BatchService() {
	}

	public BatchJobDAO getBatchJobDAO() {
		return batchJobDAO;
	}

	public void setBatchJobDAO(BatchJobDAO batchJobDAO) {
		this.batchJobDAO = batchJobDAO;
	}

	public BatchRunDAO getBatchRunDAO() {
		return batchRunDAO;
	}

	public void setBatchRunDAO(BatchRunDAO batchRunDAO) {
		this.batchRunDAO = batchRunDAO;
	}

	public BatchTypeDAO getBatchTypeDAO() {
		return batchTypeDAO;
	}

	public void setBatchTypeDAO(BatchTypeDAO batchTypeDAO) {
		this.batchTypeDAO = batchTypeDAO;
	}

	public BatchStatusDAO getBatchStatusDAO() {
		return batchStatusDAO;
	}

	public void setBatchStatusDAO(BatchStatusDAO batchStatusDAO) {
		this.batchStatusDAO = batchStatusDAO;
	}

	public BatchSrcTypeDAO getBatchSrcTypeDAO() {
		return batchSrcTypeDAO;
	}

	public void setBatchSrcTypeDAO(BatchSrcTypeDAO batchSrcTypeDAO) {
		this.batchSrcTypeDAO = batchSrcTypeDAO;
	}

	public BatchConfigDAO getBatchConfigDAO() {
		return batchConfigDAO;
	}

	public void setBatchConfigDAO(BatchConfigDAO batchConfigDAO) {
		this.batchConfigDAO = batchConfigDAO;
	}

	public List<BatchType> getBatchTypeList() {

		return batchTypeDAO.batchTypeList();
	}

	public BatchItem getLatestBatchItemById(int id) {

		BatchJob bj = batchJobDAO.getBatchJob(id);
		bj.setBatchConfig(batchConfigDAO.getBatchConfig(id));
		
		return getBatchItemForJob(bj);
	}

	public BatchItem getBatchItemForJob(BatchJob bj) {

		if (bj != null) {
			// Set the job type from the db.
			bj.setBatchType(batchTypeDAO.getBatchType(bj.getTypeId()));
			bj.setBatchConfig(batchConfigDAO.getBatchConfig(bj.getId()));
			
		}

		BatchRun run = batchRunDAO.getLastRunByBatchJobId(bj.getId());

		if (run != null) {
			run.setBatchStatus(batchStatusDAO.getStatusType(run.getStatusId()));
		}

		return new BatchItem(bj, run);
	}

	public List<BatchJob> getBatchJobList() {

		List<BatchJob> bjL = batchJobDAO.batchList();

		for (BatchJob bj : bjL) {
			// Set the job type from the db.
			bj.setBatchType(batchTypeDAO.getBatchType(bj.getTypeId()));
			bj.setBatchConfig(batchConfigDAO.getBatchConfig(bj.getId()));
			
		}

		return bjL;
	}

	public List<BatchItem> getCBSSLatestBatchItemsList() {

		List<BatchItem> batchItemL = new ArrayList<>();

		List<BatchJob> jobL = batchJobDAO.cbssBatchList();

		// Get all of the jobs.
		for (BatchJob bj : jobL) {
			batchItemL.add(getBatchItemForJob(bj));
		}

		return batchItemL;
	}
	
	public List<BatchItem> getAPPSLatestBatchItemsList() {

		List<BatchItem> batchItemL = new ArrayList<>();

		List<BatchJob> jobL = batchJobDAO.appsBatchList();

		// Get all of the jobs.
		for (BatchJob bj : jobL) {
			batchItemL.add(getBatchItemForJob(bj));
		}

		return batchItemL;
	}


	public BatchJob getBatchJobByName(String name) {

		// Get the job from the db.
		BatchJob bj = batchJobDAO.getBatchJobByName(name);

		if (bj != null) {

			// Set the job type from the db.
			bj.setBatchType(batchTypeDAO.getBatchType(bj.getTypeId()));
			bj.setBatchConfig(batchConfigDAO.getBatchConfig(bj.getId()));
			
		}

		return bj;
	}
	
	public BatchJob getBatchJobByNameAndSourceType(String name, int sourceType) {

		// Get the job from the db.
		BatchJob bj = batchJobDAO.getBatchJobByNameAndSourceType(name, sourceType);

		if (bj != null) {

			// Set the job type from the db.
			bj.setBatchType(batchTypeDAO.getBatchType(bj.getTypeId()));
			bj.setBatchConfig(batchConfigDAO.getBatchConfig(bj.getId()));
			
		}

		return bj;
	}

	public BatchJob getBatchJobById(int id) {

		// Get the job from the db.
		BatchJob bj = batchJobDAO.getBatchJob(id);

		if (bj != null) {

			// Set the job type from the db.
			bj.setBatchType(batchTypeDAO.getBatchType(bj.getTypeId()));
			bj.setBatchConfig(batchConfigDAO.getBatchConfig(bj.getId()));
			
		}
		return bj;
	}

	public BatchRun startRun(int id) {

		BatchRun bR = null;

		Integer startStatus = batchStatusDAO.getStatusFromEnum(BatchStatus.JobStatus.RUNNING);

		if (startStatus != null) {
			Timestamp startDate = new Timestamp(Calendar.getInstance().getTime().getTime());
			bR = new BatchRun(id, startDate, startStatus);
			int runId = batchRunDAO.save(bR);
			bR.setId(runId);
		} else {
			logger.error("Unable to obtain status mapping for: " + BatchStatus.JobStatus.RUNNING);
		}

		return bR;
	}

	public boolean completeRun(BatchRun bR) {

		boolean success = false;

		Integer endStatus = batchStatusDAO.getStatusFromEnum(BatchStatus.JobStatus.COMPLETE);

		if (endStatus != null) {
			Timestamp endDate = new Timestamp(Calendar.getInstance().getTime().getTime());

			bR.setEndDate(endDate);
			bR.setStatusId(endStatus);
			bR.setBatchStatus(batchStatusDAO.getStatusType(endStatus));

			batchRunDAO.update(bR);

			success = true;

		} else {
			logger.error("Unable to obtain status mapping for: " + BatchStatus.JobStatus.COMPLETE);
		}

		return success;
	}

	public boolean errorRun(BatchRun bR) {

		boolean success = false;

		Integer endStatus = batchStatusDAO.getStatusFromEnum(BatchStatus.JobStatus.ERROR);

		if (endStatus != null) {
			Timestamp endDate = new Timestamp(Calendar.getInstance().getTime().getTime());

			bR.setEndDate(endDate);
			bR.setStatusId(endStatus);
			bR.setBatchStatus(batchStatusDAO.getStatusType(endStatus));

			batchRunDAO.update(bR);

			success = true;

		} else {
			logger.error("Unable to obtain status mapping for: " + BatchStatus.JobStatus.ERROR);
		}

		return success;
	}

	public boolean setNotRunningById(BatchJob batchJob) {

		boolean reset = false;

		Integer runningStatus = batchStatusDAO.getStatusFromEnum(BatchStatus.JobStatus.RUNNING);
		Integer notRunningStatus = batchStatusDAO.getStatusFromEnum(BatchStatus.JobStatus.NOT_RUNNING);

		if ((runningStatus != null) && (notRunningStatus != null)) {

			// Update existing status from Running to NotRunning.
			BatchRun run = batchRunDAO.getLastRunByBatchJobId(batchJob.getId());

			if ((run != null) && (run.getStatusId() == runningStatus)) {
				run.setStatusId(notRunningStatus);
				batchRunDAO.update(run);
				reset = true;
			}

		} else {
			logger.error("Unable to obtain status mapping for: " + BatchStatus.JobStatus.RUNNING + " or "
					+ BatchStatus.JobStatus.NOT_RUNNING + " when attempting to check for running job ("
					+ validateStringInput(batchJob.getName(), LOG_FORGING) + ")");
		}

		return reset;
	}

	public BatchType getBatchTypeForJob(BatchJob batchJob) {
		return batchTypeDAO.getBatchType(batchJob.getTypeId());
	}

	public BatchConfig getBatchConfigForJob(BatchJob batchJob){
		return batchConfigDAO.getBatchConfig(batchJob.getId());
	}
	
	public void updateBatchJob(BatchJob batchJob) {
		batchJobDAO.update(batchJob);
		if(getBatchTypeForJob(batchJob).getJobType().equals(BatchType.JobType.SCHEDULED)){
			batchConfigDAO.updateOrInsert(batchJob);
		}			
	}
	
	public int getBatchSrcTypeId(BatchSrcType.SrcType srcType){
		for(BatchSrcType bst : batchSrcTypeDAO.batchSrcTypeList()) {
			if(bst.getName().equals(srcType.getName())) {
				return bst.getId();
			}
		}
		
		logger.error("No batch source type ID for source type: "+srcType.getName());
		return -1;		
	}
}
