package gov.va.cpss.service;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.model.ps.Constants.LINE_FEED;

import java.time.Year;

import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;

import gov.va.cpss.dao.APSDetailsDAO;
import gov.va.cpss.dao.APSPatientDAO;
import gov.va.cpss.dao.APSPaymentDAO;
import gov.va.cpss.dao.APSReceivedDAO;
import gov.va.cpss.dao.APSReceivedSiteDAO;
import gov.va.cpss.dao.APSSiteDAO;
import gov.va.cpss.dao.APSSitePatientDAO;
import gov.va.cpss.dao.APSSiteStmtDAO;
import gov.va.cpss.dao.APSStmtDAO;
import gov.va.cpss.dao.BatchRunProcessDAO;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.ProcessStatusDAO;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.ProcessStatus.Status;

/**
 * Abstract service class for handling activities relating to process APPS Data.
 * 
 * Copyright DXC / VA
 * May 4, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings({"nls", "static-method"})
public abstract class APPSBaseService<T>
{
    protected Log logger = LogFactory.getLog(getClass());

    //
    // Injected properties
    //
    // Common
    protected ProcessStatusDAO processStatusDAO;
    protected BatchRunProcessDAO batchRunProcessDAO;
    protected CBSAccountDAO cbsAccountDAO;

    // Load APPS
    protected APSReceivedDAO apsReceivedDAO;
    protected APSSiteDAO apsSiteDAO;
    protected APSPatientDAO apsPatientDAO;
    protected APSDetailsDAO apsDetailsDAO;
    protected APSReceivedSiteDAO apsReceivedSiteDAO;

    // Generate APPS
    protected APSStmtDAO apsStmtDAO;
    protected APSSiteStmtDAO apsSiteStmtDAO;
    protected APSSitePatientDAO apsSitePatientDAO;
    protected APSPaymentDAO apsPaymentDAO;


    //
    // Others
    //
    // Date and Time
    protected int previousYear = 0;
    protected int currentYear = 0;
    protected Date currentDate;

    // Status
    protected int statusIdInitial = 0;
    protected int statusIdNew = 0;
    protected int statusIdProcessed = 0;
    protected int statusIdError = 0;
    protected int statusIdOther = 0;

    // Starting time
    protected long startTime = 0L;


    //
    // Constructors
    //
    // Default constructor
    public APPSBaseService()
    {
        // Set years
        this.currentYear = Year.now().getValue();
        this.previousYear = this.currentYear - 1;

        // Set current date time
        this.currentDate = Calendar.getInstance().getTime();
    }


    // Initialize Status
    // This needs to be run after the ProcessStatusDAO is injected.
    private void initStatus()
    {
        Integer processStatus = this.processStatusDAO.getStatusFromEnum(Status.INITIAL);

        if (processStatus != null)
        {
            this.statusIdInitial = processStatus.intValue();
        }
        else
        {
            this.logger.error("Unable to obtain status mapping for: " + Status.INITIAL);
        }

        processStatus = this.processStatusDAO.getStatusFromEnum(Status.NEW);

        if (processStatus != null)
        {
            this.statusIdNew = processStatus.intValue();
        }
        else
        {
            this.logger.error("Unable to obtain status mapping for: " + Status.NEW);
        }

        processStatus = this.processStatusDAO.getStatusFromEnum(Status.PROCESSED);

        if (processStatus != null)
        {
            this.statusIdProcessed = processStatus.intValue();
        }
        else
        {
            this.logger.error("Unable to obtain status mapping for: " + Status.PROCESSED);
        }

        processStatus = this.processStatusDAO.getStatusFromEnum(Status.ERROR);

        if (processStatus != null)
        {
            this.statusIdError = processStatus.intValue();
        }
        else
        {
            this.logger.error("Unable to obtain status mapping for: " + Status.ERROR);
        }

        processStatus = this.processStatusDAO.getStatusFromEnum(Status.OTHER);

        if (processStatus != null)
        {
            this.statusIdOther = processStatus.intValue();
        }
        else
        {
            this.logger.error("Unable to obtain status mapping for: " + Status.OTHER);
        }
    }

    //
    // Access methods
    //
    /**
     * @return
     */
    public ProcessStatusDAO getProcessStatusDAO()
    {
        return this.processStatusDAO;
    }

    /**
     * @param inProcessStatusDAO
     */
    public void setProcessStatusDAO(ProcessStatusDAO inProcessStatusDAO)
    {
        this.processStatusDAO = inProcessStatusDAO;

        initStatus();
    }

    /**
     * @return the batchRunProcessDAO
     */
    public BatchRunProcessDAO getBatchRunProcessDAO()
    {
        return this.batchRunProcessDAO;
    }

    /**
     * @param inBatchRunProcessDAO the batchRunProcessDAO to set
     */
    public void setBatchRunProcessDAO(BatchRunProcessDAO inBatchRunProcessDAO)
    {
        this.batchRunProcessDAO = inBatchRunProcessDAO;
    }

    /**
     * @return the cbsAccountDAO
     */
    public CBSAccountDAO getCbsAccountDAO()
    {
        return this.cbsAccountDAO;
    }

    /**
     * @param inCbsAccountDAO the cbsAccountDAO to set
     */
    public void setCbsAccountDAO(CBSAccountDAO inCbsAccountDAO)
    {
        this.cbsAccountDAO = inCbsAccountDAO;
    }

    /**
     * @return
     */
    public APSReceivedDAO getApsReceivedDAO()
    {
        return this.apsReceivedDAO;
    }

    /**
     * @param inApsReceivedDAO
     */
    public void setApsReceivedDAO(APSReceivedDAO inApsReceivedDAO)
    {
        this.apsReceivedDAO = inApsReceivedDAO;
    }

    /**
     * @return
     */
    public APSSiteDAO getApsSiteDAO()
    {
        return this.apsSiteDAO;
    }

    /**
     * @param inApsSiteDAO
     */
    public void setApsSiteDAO(APSSiteDAO inApsSiteDAO)
    {
        this.apsSiteDAO = inApsSiteDAO;
    }

    /**
     * @return
     */
    public APSPatientDAO getApsPatientDAO()
    {
        return this.apsPatientDAO;
    }

    /**
     * @param inApsPatientDAO
     */
    public void setApsPatientDAO(APSPatientDAO inApsPatientDAO)
    {
        this.apsPatientDAO = inApsPatientDAO;
    }

    /**
     * @return
     */
    public APSDetailsDAO getApsDetailsDAO()
    {
        return this.apsDetailsDAO;
    }

    /**
     * @param inApsDetailsDAO
     */
    public void setApsDetailsDAO(APSDetailsDAO inApsDetailsDAO)
    {
        this.apsDetailsDAO = inApsDetailsDAO;
    }

    /**
     * @return
     */
    public APSReceivedSiteDAO getApsReceivedSiteDAO()
    {
        return this.apsReceivedSiteDAO;
    }

    /**
     * @param inApsReceivedSiteDAO
     */
    public void setApsReceivedSiteDAO(APSReceivedSiteDAO inApsReceivedSiteDAO)
    {
        this.apsReceivedSiteDAO = inApsReceivedSiteDAO;
    }

    /**
     * @return the apsStmtDAO
     */
    public APSStmtDAO getApsStmtDAO()
    {
        return this.apsStmtDAO;
    }

    /**
     * @param inApsStmtDAO the apsStmtDAO to set
     */
    public void setApsStmtDAO(APSStmtDAO inApsStmtDAO)
    {
        this.apsStmtDAO = inApsStmtDAO;
    }

    /**
     * @return the apsSiteStmtDAO
     */
    public APSSiteStmtDAO getApsSiteStmtDAO()
    {
        return this.apsSiteStmtDAO;
    }

    /**
     * @param inApsSiteStmtDAO the apsSiteStmtDAO to set
     */
    public void setApsSiteStmtDAO(APSSiteStmtDAO inApsSiteStmtDAO)
    {
        this.apsSiteStmtDAO = inApsSiteStmtDAO;
    }

    /**
     * @return the apsSitePatientDAO
     */
    public APSSitePatientDAO getApsSitePatientDAO()
    {
        return this.apsSitePatientDAO;
    }

    /**
     * @param inApsSitePatientDAO the apsSitePatientDAO to set
     */
    public void setApsSitePatientDAO(APSSitePatientDAO inApsSitePatientDAO)
    {
        this.apsSitePatientDAO = inApsSitePatientDAO;
    }

    /**
     * @return the apsPaymentDAO
     */
    public APSPaymentDAO getApsPaymentDAO()
    {
        return this.apsPaymentDAO;
    }

    /**
     * @param inApsPaymentDAO the apsPaymentDAO to set
     */
    public void setApsPaymentDAO(APSPaymentDAO inApsPaymentDAO)
    {
        this.apsPaymentDAO = inApsPaymentDAO;
    }

    /**
     * @return the currentDate
     */
    public Date getCurrentDate()
    {
        return this.currentDate;
    }


    //
    // Job methods
    //
    /**
     * Start the job.
     * 
     * @param batchRun
     * @param fileName
     * @return
     */
    public Result<T> startJob(final BatchRun batchRun, final String fileName)
    {
        this.logger.info("Service: Start Job.");

        // Get the starting time
        this.startTime = System.nanoTime();

        return new Result<>(true, null, null);
    }

    /**
     * End the job.
     * 
     * @param execution
     * @param t
     * @return
     */
    public String endJob(JobExecution execution, T t)
    {
        String message = null;
        String tempMessage = null;

        if (execution == null)
        {
            message = "Job ended with null execution.";
            this.logger.error(message);

            tempMessage = processJobError(t);

            if (tempMessage != null)
            {
                message += LINE_FEED + tempMessage;
            }
        }
        else if (!execution.getExitStatus().equals(ExitStatus.COMPLETED))
        {
            if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
            {
                message = "Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY);
                this.logger.error(message);
            }
            else
            {
                message = "Job ended with unknown error: " + execution.getExitStatus();
                this.logger.error(message);
            }

            tempMessage = processJobError(t);

            if (tempMessage != null)
            {
                message += LINE_FEED + tempMessage;
            }
        }
        else
        {
            message = completeJob(execution, t);

            // If unable to successfully complete then treat as an error.
            if (message != null)
            {
                tempMessage = processJobError(t);

                if (tempMessage != null)
                {
                    message += LINE_FEED + tempMessage;
                }
            }
        }

        final long elapsedTime = System.nanoTime() - this.startTime;

        this.logger.info("Service: Job Ended. " + printElapsedTime("job", elapsedTime));

        return message;
    }

    /**
     * Process job error.
     * 
     * @param t
     */
    abstract protected String processJobError(T t);

    /**
     * Complete the job.
     * 
     * @param execution
     * @param t
     * @return
     */
    abstract protected String completeJob(JobExecution execution, T t);


    //
    // Other utility methods
    //
    /**
     * Print out the elapsed time in string format.
     * 
     * @param name
     * @param elapsedTime
     * @return
     */
    public String printElapsedTime(String name, long elapsedTime)
    {
        long seconds = TimeUnit.MINUTES.toSeconds(TimeUnit.NANOSECONDS.toMinutes(elapsedTime));
        long milliseconds = TimeUnit.SECONDS.toMillis(TimeUnit.NANOSECONDS.toSeconds(elapsedTime));

        return String.format("The " + name + " took %d hours, %d minutes, %d seconds, and %d milliseconds to run.", 
                             Long.valueOf(TimeUnit.NANOSECONDS.toHours(elapsedTime)),
                             Long.valueOf(TimeUnit.NANOSECONDS.toMinutes(elapsedTime)),
                             Long.valueOf(TimeUnit.NANOSECONDS.toSeconds(elapsedTime) - seconds),
                             Long.valueOf(TimeUnit.NANOSECONDS.toMillis(elapsedTime) - milliseconds));
    }

    /**
     * Check if it is the previous year.
     * 
     * @param date
     * @return
     */
    public boolean isPreviousYear(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        return (date != null) && (calendar.get(Calendar.YEAR) == this.previousYear);
    }

    /**
     * Return an SQL date from the year number.
     * 
     * @param year
     * @return
     */
    public java.sql.Date getSqlDateFromYear(final int year)
    {
        Calendar calendarDate = Calendar.getInstance();

        // Set to the first day of current year.
        calendarDate.set(Calendar.YEAR, year);
        calendarDate.set(Calendar.MONTH, 0);
        calendarDate.set(Calendar.DAY_OF_MONTH, 1);
        calendarDate.set(Calendar.HOUR_OF_DAY, 0);
        calendarDate.set(Calendar.MINUTE, 0);
        calendarDate.set(Calendar.SECOND, 0);
        calendarDate.set(Calendar.MILLISECOND, 0);

        return new java.sql.Date(calendarDate.getTime().getTime());
    }

    /**
     * Check if it is a valid Facility / Station / Site
     * against the Station Info file.
     * 
     * @param facilityNum
     * @return
     */
    public boolean isValidFacility(String facilityNum)
    {
        return StationInfoUtil.isValidStation(facilityNum);
    }

    /**
     * Check if it is erred.
     * 
     * @param statusId
     * @return
     */
    public boolean isErrorStatus(final int statusId)
    {
        return (statusId == this.statusIdError);
    }
}
