package gov.va.cpss.model.loadbill;

import java.text.ParseException;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;

import gov.va.cpss.job.loadbill.LoadBillProcessingConstants;

public class LoadBillDetailValidator {

	private static final int MAX_ICN_LENGTH = 18;
	private static final int MAX_ACNT_NUM_LENGTH = 19;
	private static final int MAX_DATE_LENGTH = 8;
	private static final int MAX_SUBTOTAL_LENGTH = 10;
	private static final int MAX_DFN_LENGTH = 16;
	
	/**
	 * Return true if there are any validation errors and false if there are none.
	 * @param detail the PD record to validate.
	 */
	public static boolean hasErrors(LoadBillDetail detail) {
		if(StringUtils.isEmpty(detail.getIcnNumber()) ||
				detail.getIcnNumber().length() > MAX_ICN_LENGTH) {
			return true;
		}
		if(StringUtils.isEmpty(detail.getPatientAccountNumber()) ||
				detail.getPatientAccountNumber().length() > MAX_ACNT_NUM_LENGTH) {
			return true;
		}
		if(StringUtils.isEmpty(detail.getPatientDfn()) ||
				detail.getPatientDfn().length() > MAX_DFN_LENGTH ||
				!StringUtils.isNumeric(detail.getPatientDfn())) {
			return true;
		}
		if(StringUtils.isEmpty(detail.getOldestBillDate()) ||
				detail.getOldestBillDate().length() > MAX_DATE_LENGTH) {
			return true;
		}
		// Catch an exception if the format is incorrect.
		try {
			LoadBillProcessingConstants.DATE_FORMAT.parse(detail.getOldestBillDate());
		}
		catch(ParseException e) {
			return true;
		}
		if(StringUtils.isEmpty(detail.getPatientSubtotal()) ||
				detail.getPatientSubtotal().length() > MAX_SUBTOTAL_LENGTH ||
				!isAITCDollarFormat(detail.getPatientSubtotal())) {
			return true;
		}
		return false;
	}
	
	/**
	 * Checks to see if a number is in the form of an AITCDollar number.
	 */
	private static boolean isAITCDollarFormat(String str) {
		char lastChar = str.charAt(str.length() - 1);
		String shortStr = str;
		if(lastChar == '+' || lastChar == '-')
		{
			shortStr = str.substring(0, str.length() - 1);
		}
		//After removing the plus and minus signs, only numbers should be left over.
		return NumberUtils.isDigits(shortStr);
	}
}
