package gov.va.cpss.model.apps;

import static gov.va.cpss.model.ps.Constants.DATE_FORMAT_STRING;

import java.io.Serializable;

import java.text.ParseException;
import java.text.SimpleDateFormat;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import gov.va.cpss.model.AITCDollar;

import gov.va.cpss.model.ps.Patient;
import gov.va.cpss.model.ps.RecordType;

/**
 * Model class for APSPatient.
 * 
 * @author Yiping Yao
 * @version 1.0 - 20161019
 *
 */
@SuppressWarnings("nls")
public class APSPatient extends Patient implements Serializable, APSRecord<RecordType>
{
    /**
     * Generated Serial Version UID
     */
    private static final long serialVersionUID = 597668574567806162L;

    // Constants - Field lengths
    public static final int MAX_TOTAL_AMNT_RCVD_LENGTH = 10;
    public static final int MAX_NUM_OF_PD_LENGTH = 4;

    // Properties
    private APSSite apsSite;
    private AITCDollar totalAmountReceived;
    private String totalAmountReceivedStr;
    private boolean arAddress;
    private String arAddressFlg;
    private Date lastBillPrepDate;
    private String lastBillPrepDateStr;
    private int numOfPD;
    private String numOfPDStr;

    private List<APSDetails> detailsList = new ArrayList<>();

    //
    // Constructors
    //
    public APSPatient()
    {
        super();
    }

    public APSPatient(final Patient patient)
    {
        super(patient);
    }


    //
    // Access Methods
    //
    /**
     * @return the apsSite
     */
    public APSSite getApsSite()
    {
        return this.apsSite;
    }

    /**
     * @param inApsSite the apsSite to set
     */
    public void setApsSite(APSSite inApsSite)
    {
        this.apsSite = inApsSite;
    }

    /**
     * @return the totalAmountReceived
     */
    public AITCDollar getTotalAmountReceived()
    {
        return this.totalAmountReceived;
    }

    /**
     * @param inTotalAmountReceived the totalAmountReceived to set
     */
    public void setTotalAmountReceived(AITCDollar inTotalAmountReceived)
    {
        this.totalAmountReceived = inTotalAmountReceived;
    }

    /**
     * @return the totalAmountReceivedStr
     */
    public String getTotalAmountReceivedStr()
    {
        return this.totalAmountReceivedStr;
    }

    /**
     * @param inTotalAmountReceivedStr the totalAmountReceivedStr to set
     */
    public void setTotalAmountReceivedStr(String inTotalAmountReceivedStr)
    {
        this.totalAmountReceivedStr = inTotalAmountReceivedStr;

        if (this.totalAmountReceivedStr != null && !this.totalAmountReceivedStr.isEmpty())
        {
            this.totalAmountReceived = AITCDollar.fromStringWithoutSign(this.totalAmountReceivedStr);
        }
    }

    /**
     * @return the arAddress
     */
    public boolean isArAddress()
    {
        return this.arAddress;
    }

    /**
     * @param inArAddress the arAddress to set
     */
    public void setArAddress(boolean inArAddress)
    {
        this.arAddress = inArAddress;
    }

    /**
     * @return the arAddressFlg
     */
    public String getArAddressFlg()
    {
        return this.arAddressFlg;
    }

    /**
     * @param inArAddressFlg the arAddressFlg to set
     */
    public void setArAddressFlg(String inArAddressFlg)
    {
        this.arAddressFlg = inArAddressFlg;

        this.arAddress = (this.arAddressFlg.equals("Y")) ? true : false;
    }

    /**
     * @return the lastBillPrepDate
     */
    public Date getLastBillPrepDate()
    {
        return this.lastBillPrepDate;
    }

    /**
     * @param inLastBillPrepDate the lastBillPrepDate to set
     */
    public void setLastBillPrepDate(Date inLastBillPrepDate)
    {
        this.lastBillPrepDate = inLastBillPrepDate;
    }

    /**
     * @return the lastBillPrepDateStr
     */
    public String getLastBillPrepDateStr()
    {
        return this.lastBillPrepDateStr;
    }

    /**
     * @param inLastBillPrepDateStr the lastBillPrepDateStr to set
     */
    public void setLastBillPrepDateStr(String inLastBillPrepDateStr)
    {
        this.lastBillPrepDateStr = inLastBillPrepDateStr;

        this.lastBillPrepDate = parseDate(this.lastBillPrepDateStr);
    }

    /**
     * @return the numOfPD
     */
    public int getNumOfPD()
    {
        return this.numOfPD;
    }

    /**
     * @param inNumOfPD the numOfPD to set
     */
    public void setNumOfPD(int inNumOfPD)
    {
        this.numOfPD = inNumOfPD;
    }

    /**
     * @return the numOfPDStr
     */
    public String getNumOfPDStr()
    {
        return this.numOfPDStr;
    }

    /**
     * @param inNumOfPDStr the numOfPDStr to set
     */
    public void setNumOfPDStr(String inNumOfPDStr)
    {
        this.numOfPDStr = inNumOfPDStr;

        if (this.numOfPDStr != null &&
            StringUtils.isNumeric(this.numOfPDStr))
        {
            this.numOfPD = Integer.parseInt(this.numOfPDStr);
        }
    }

    /**
     * @return the detailsList
     */
    public List<APSDetails> getDetailsList()
    {
        return this.detailsList;
    }

    /**
     * @param inDetailsList the detailsList to set
     */
    public void setDetailsList(List<APSDetails> inDetailsList)
    {
        this.detailsList = inDetailsList;
    }

    /**
     * Return false if there are any validation errors and true if there are none.
     */
    @Override
    public boolean isValid()
    {
        boolean isValid = true;
        StringBuffer strBuffer = new StringBuffer();

        if (StringUtils.isEmpty(this.accountNumber) ||
            this.accountNumber.length() > MAX_ACNT_NUM_LENGTH)
        {
            strBuffer.append("accountNumber is invalid (PH002). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.lastName) ||
            this.lastName.length() > MAX_LAST_NAME_LENGTH)
        {
            strBuffer.append("lastName is invalid (PH003). ");
            isValid = false;
        }

        if(StringUtils.isEmpty(this.firstName) ||
           this.firstName.length() > MAX_FIRST_NAME_LENGTH)
        {
            strBuffer.append("firstName is invalid (PH004). ");
            isValid = false;
        }

        // Optional
        if (!StringUtils.isEmpty(this.middleName) &&
            this.middleName.length() > MAX_MIDDLE_NAME_LENGTH)
        {
            strBuffer.append("middleName is invalid (PH005). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.address1) ||
            this.address1.length() > MAX_ADDRESS_LENGTH)
        {
            strBuffer.append("address1 is invalid (PH006). ");
            isValid = false;
        }

        // Optional
        if (!StringUtils.isEmpty(this.address2) &&
            this.address2.length() > MAX_ADDRESS_LENGTH)
        {
            strBuffer.append("address2 is invalid (PH007). ");
            isValid = false;
        }

        // Optional
        if (!StringUtils.isEmpty(this.address3) &&
            this.address3.length() > MAX_ADDRESS_LENGTH)
        {
            strBuffer.append("address3 is invalid (PH008). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.city) ||
            this.city.length() > MAX_CITY_LENGTH)
        {
            strBuffer.append("city is invalid (PH009). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.state) ||
            this.state.length() > MAX_STATE_LENGTH)
        {
            strBuffer.append("state is invalid (PH010). ");
            isValid = false;
        }

        // If it is foreign state, the ZIP code is not required,
        // otherwise, it is required.
        if (this.state != null && this.state.equalsIgnoreCase(FOREIGN_STATE))
        {
            // Foreign postal code
            if ( !StringUtils.isEmpty(this.zipCode) &&
                 (this.zipCode.length() > MAX_ZIP_CODE_LENGTH ||
                  !StringUtils.isAlphanumeric(this.zipCode)) )
            {
                strBuffer.append("zipCode is invalid (PH011). ");
                isValid = false;
            }
        }
        else
        {
            // U.S. ZIP code
            if ( StringUtils.isEmpty(this.zipCode) ||
                  this.zipCode.length() > MAX_ZIP_CODE_LENGTH ||
                  !StringUtils.isNumeric(this.zipCode) )
            {
                strBuffer.append("zipCode is invalid (PH011). ");
                isValid = false;
            }
        }

        // Optional
        if (!StringUtils.isEmpty(this.country) &&
            this.country.length() > MAX_COUNTRY_LENGTH)
        {
            strBuffer.append("country is invalid (PH012). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.totalAmountReceivedStr) ||
            this.totalAmountReceivedStr.length() > MAX_TOTAL_AMNT_RCVD_LENGTH ||
            !AITCDollar.isAITCDollarFormatWithoutSign(this.totalAmountReceivedStr))
        {
            strBuffer.append("totalAmountReceived is invalid (PH013). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.dfnStr) ||
            this.dfnStr.length() > MAX_DFN_LENGTH ||
            !StringUtils.isNumeric(this.dfnStr))
        {
            strBuffer.append("dfn is invalid (PH014). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.icn) ||
            this.icn.length() > MAX_ICN_LENGTH)
        {
            strBuffer.append("icn is invalid (PH015). ");
            isValid = false;
        }

        if ( StringUtils.isEmpty(this.arAddressFlg) ||
             this.arAddressFlg.length() > MAX_FLAG_LENGTH ||
             (!this.arAddressFlg.equalsIgnoreCase(YES_FLAG) &&
              !this.arAddressFlg.equalsIgnoreCase(NO_FLAG)) )
        {
            strBuffer.append("arAddressFlg is invalid (PH016). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.lastBillPrepDateStr) ||
            this.lastBillPrepDateStr.length() != MAX_DATE_LENGTH)
        {
            strBuffer.append("lastBillPrepDate is invalid (PH017). ");
            isValid = false;
        }

        // Catch an exception if the format is incorrect.
        try
        {
            new SimpleDateFormat(DATE_FORMAT_STRING).parse(this.lastBillPrepDateStr);
        }
        catch(@SuppressWarnings("unused") ParseException e)
        {
            strBuffer.append("lastBillPrepDate is invalid (PH017). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.numOfPDStr) ||
            this.numOfPDStr.length() > MAX_NUM_OF_PD_LENGTH ||
            !StringUtils.isNumeric(this.numOfPDStr))
            {
                strBuffer.append("numOfPD is invalid (PH018). ");
                isValid = false;
            }

        if (!isValid)
        {
            this.message = strBuffer.toString();
        }

        return isValid;
    }

    @Override
    public String toString()
    {
        return String.format("APS Patient (%s) [accountNumber=%s, lastName=%s, firstName=%s, middleName=%s, address1=%s, address2=%s, address3=%s, " + 
                             "city=%s, state=%s, zipCode=%s, country=%s, totalAmountReceivedStr=%s, dfnStr=%s, icn=%s, arAddressFlg=%s, " +
                             "lastBillPrepDateStr=%s, numOfPDStr=%s]" + " Message: %s",
                             this.type, this.accountNumber, this.lastName, this.firstName, this.middleName, this.address1, this.address2, this.address3,
                             this.city, this.state, this.zipCode, this.country, this.totalAmountReceivedStr, this.dfnStr, this.icn, this.arAddressFlg,
                             this.lastBillPrepDateStr, this.numOfPDStr,
                             (this.message == null ? "none" : this.message));
    }

}
