package gov.va.cpss.job.updateicn;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;

import static gov.va.cpss.job.updateicn.UpdateICNConstants.ICN_UPDATE_SUCCESS_KEY;
import static gov.va.cpss.job.updateicn.UpdateICNConstants.TOTAL_ICN_UPDATED_KEY;
import static gov.va.cpss.job.updateicn.UpdateICNConstants.ICN_UPDATE_MESSAGE_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.service.Result;
import gov.va.cpss.service.UpdateICNService;

/**
 * The job processor that invokes UpdateICNService to process ICN updates.
 * 
 * @author Yiping Yao
 * @version 1.0 - 20160810
 *
 */
@SuppressWarnings("nls")
public class UpdateICNProcessor implements ItemProcessor<Object, Object>
{
	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	private JobExecution jobExecution;
	private Long batchRunId;

	private UpdateICNService updateICNService;

	/**
	 * @return the updateICNService
	 */
	public UpdateICNService getUpdateICNService()
	{
		return this.updateICNService;
	}

	/**
	 * @param inUpdateICNService the updateICNService to set
	 */
	public void setUpdateICNService(UpdateICNService inUpdateICNService)
	{
		this.updateICNService = inUpdateICNService;
	}

	/**
	 * Before processing capture the job execution and the batch run id for processing.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution)
	{
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is processed.
		this.jobExecution = stepExecution.getJobExecution();

		// Save the batchRunId at the beginning of the step.
		// It is obtained by the batch prior to the job and passed as a job
		// parameter when the job starts.
		this.batchRunId = this.jobExecution.getJobParameters().getLong(BATCH_RUN_ID_KEY);
	}

	@Override
	public Object process(Object inObject) throws Exception
	{
		this.logger.info("Process begings ...");

		boolean isSuccessful = false;
		String message = null;
		
		Result<Integer[]> result = null;

		if (this.batchRunId != null)
		{
			result = this.updateICNService.updateICN(null);

			if (result != null)
			{
				isSuccessful = result.isSuccessful();

				if (isSuccessful)
				{
					message = "The Update ICN job did run and complete successfully.";

					this.jobExecution.getExecutionContext().put(ICN_UPDATE_SUCCESS_KEY, new Boolean(true));
					this.jobExecution.getExecutionContext().putLong(TOTAL_ICN_UPDATED_KEY, result.get().length);					
					this.jobExecution.getExecutionContext().putString(ICN_UPDATE_MESSAGE_KEY, message);
				}
			}
		}
		
		if (!isSuccessful)
		{
			this.jobExecution.getExecutionContext().put(ICN_UPDATE_SUCCESS_KEY, new Boolean(false));
			this.jobExecution.getExecutionContext().putLong(TOTAL_ICN_UPDATED_KEY, 0);

			message = "The Update ICN job did not run or complete successfully.";

			if (result != null)
			{
				message += "\n" + "Cause: " + result.getMessage();
			}

			this.jobExecution.getExecutionContext().putString(ICN_UPDATE_MESSAGE_KEY, message);
		}

		this.logger.debug(message);

		this.logger.info("Process ends.");

		return null;
	}

}
