package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_MESSAGE_ID_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_OUTPUT_RESOURCE_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_TOTAL_PATIENTS_INCLUDED_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.SBS_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.SBS_OUTPUT_RESOURCE_KEY;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.job.CbssJobBaseTransactionTasklet;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.service.SendCBSService;
import gov.va.cpss.service.SftpService;
import gov.va.cpss.time.TimeUtils;

/**
 * Complete processing by transferring file to sftp server with temporary name,
 * setting appropriate message status and statement status, and renaming
 * temporary file on sftp server with correct name. If any errors occur the
 * database updates are rolled back and the batch job status is set to error
 * condition.
 * 
 * @author DNS   
 */
public class SendCBSProcessFileCompletionTasklet extends CbssJobBaseTransactionTasklet {

	private final String TIMESTAMP_FORMAT = "HH_mm_ss";

	private String tempFilenamePostfix;

	private String serverTargetDirectory;

	private SftpService sftpService;

	private SendCBSService sendCBSService;

	public String getTempFilenamePostfix() {
		return tempFilenamePostfix;
	}

	public void setTempFilenamePostfix(String tempFilenamePostfix) {
		this.tempFilenamePostfix = tempFilenamePostfix;
	}

	public String getServerTargetDirectory() {
		return serverTargetDirectory;
	}

	public void setServerTargetDirectory(String serverTargetDirectory) {
		this.serverTargetDirectory = serverTargetDirectory;
	}

	public SftpService getSftpService() {
		return sftpService;
	}

	public void setSftpService(SftpService sftpService) {
		this.sftpService = sftpService;
	}

	public SendCBSService getSendCBSService() {
		return sendCBSService;
	}

	public void setSendCBSService(SendCBSService sendCBSService) {
		this.sendCBSService = sendCBSService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("File Completion Tasklet Running!");

		// Save the jobExecution for reference.
		JobExecution jobExecution = chunkContext.getStepContext().getStepExecution().getJobExecution();

		// Obtain the temporary local file paths as they will be needed regardless of failure.
		// Consolidated statements file local path
		final String cbsOutputResourceLocalTempFilePath = getOutputResource(jobExecution, CBS_OUTPUT_RESOURCE_KEY);
		if ((cbsOutputResourceLocalTempFilePath == null) || cbsOutputResourceLocalTempFilePath.isEmpty()) {
			// Return now and roll back!
			return false;
		}
		
		// Single statements file local path
		final String sbsOutputResourceLocalTempFilePath = getOutputResource(jobExecution, SBS_OUTPUT_RESOURCE_KEY);
		if ((sbsOutputResourceLocalTempFilePath == null) || sbsOutputResourceLocalTempFilePath.isEmpty()) {
			// Return now and roll back!
			return false;
		}

		// If the job has been successful then transfer files to server and
		// attempt to set appropriate final statement and message statuses.
		if (!jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {

			// Get filename values from the jobExecution or jobParameters.

			// The consolidated statements output filename.
			final String cbsOutputFileName = getJobParameter(jobExecution, CBS_OUTPUT_FILE_NAME_KEY);
			if ((cbsOutputFileName == null) || cbsOutputFileName.isEmpty()) {
				// Flag to do rollback.
				successful = false;				
			}
			// The temporary consolidated statements filename used to transfer to the sftp server.
			final String cbsTargetTemporaryFileName = getTemporaryFilename(cbsOutputFileName);
			if (successful) {
				if (cbsTargetTemporaryFileName != null) {
					taskletLogger.info("cbsTargetTemporaryFileName: " + cbsTargetTemporaryFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Could not create a temporary consolidated statements server filename";
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// The single statements output filename.
			final String sbsOutputFileName = getJobParameter(jobExecution, SBS_OUTPUT_FILE_NAME_KEY);
			if ((sbsOutputFileName == null) || sbsOutputFileName.isEmpty()) {
				// Flag to do rollback.
				successful = false;				
			}
			// The temporary single statements filename used to transfer to the sftp server.
			final String sbsTargetTemporaryFileName = getTemporaryFilename(sbsOutputFileName);
			if (successful) {
				if (sbsTargetTemporaryFileName != null) {
					taskletLogger.info("sbsTargetTemporaryFileName: " + sbsTargetTemporaryFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Could not create a temporary single statements server filename";
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}


			// Get latest Message ID from the jobExecution context.
			final Long msgId = jobExecution.getJobParameters().getLong(CBS_MESSAGE_ID_KEY);
			if (successful) {
				if ((msgId == null) || (msgId < 1)) {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Could not obtain a valid message ID: " + msgId;
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Send consolidated statements file with temporary filename (with timestamp extension).
			// Example:
			// 'CBS-06022016.txt' would be temporarily sent as
			// 'CBS-06022016.txt.<timestamp>.sendcbs'
			if (successful) {
				if (sftpService.ftpFileToServerWithName(cbsOutputResourceLocalTempFilePath, cbsTargetTemporaryFileName, serverTargetDirectory)) {
					taskletLogger.info("Sent file to server sucessfully: " + cbsTargetTemporaryFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful file transfer: " + cbsTargetTemporaryFileName;
					setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Send single statements file with temporary filename (with timestamp extension).
			// Example:
			// 'SBS-06022016.txt' would be temporarily sent as
			// 'SBS-06022016.txt.<timestamp>.sendcbs'
			if (successful) {
				if (sftpService.ftpFileToServerWithName(sbsOutputResourceLocalTempFilePath, sbsTargetTemporaryFileName, serverTargetDirectory)) {
					taskletLogger.info("Sent file to server sucessfully: " + sbsTargetTemporaryFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful file transfer: " + sbsTargetTemporaryFileName;
					setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Mark the CBSStmt statuses to sent. (SendCBSService:L407)
			long totalPatientsIncluded = -1;
			if (successful) {
				totalPatientsIncluded = sendCBSService.updateAllStatementStatusToSentForMessageId(msgId);
				if (totalPatientsIncluded >= 0) {

					taskletLogger.info("Updated all statement status to sent for message ID: " + msgId);

				} else {

					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful attempting to update all statement status to sent for message ID: "
							+ msgId;
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

					// Flag to do rollback.
					successful = false;
				}
			}

			// Update Message status to successful!
			if (successful) {
				if (sendCBSService.setMessageStatus(msgId, ProcessStatus.Status.SUCCESS)) {
					taskletLogger.info("Updated message status to success for message ID: " + msgId);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful attempting to update message status to success for message ID: "
							+ msgId;
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Rename the consolidated statements file on the server (by removing the temporary
			// '.<timestamp>.sendcbs' extension).
			if (successful) {
				if (sftpService.ftpRenameFileInDirectory(cbsTargetTemporaryFileName, cbsOutputFileName, serverTargetDirectory)) {
					taskletLogger.info("Renamed file on server sucessfully: " + cbsOutputFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful file rename: " + cbsOutputFileName;
					setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Rename the single statements file on the server (by removing the temporary
			// '.<timestamp>.sendcbs' extension).
			if (successful) {
				if (sftpService.ftpRenameFileInDirectory(sbsTargetTemporaryFileName, sbsOutputFileName, serverTargetDirectory)) {
					taskletLogger.info("Renamed file on server sucessfully: " + sbsOutputFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful file rename: " + sbsOutputFileName;
					setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Send a done file for consolidated statements to the server.
			final String cbsOutputDoneFileName = getDoneFilename(cbsOutputFileName);
			if (successful) {
				if (sftpService.ftpEmptyFileToServerWithName(cbsOutputDoneFileName, serverTargetDirectory)) {
					taskletLogger.info("Transfer done file to server sucessfully: " + cbsOutputDoneFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful done file transfer: " + cbsOutputDoneFileName;
					setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// Send a done file for consolidated statements to the server.
			final String sbsOutputDoneFileName = getDoneFilename(sbsOutputFileName);
			if (successful) {
				if (sftpService.ftpEmptyFileToServerWithName(sbsOutputDoneFileName, serverTargetDirectory)) {
					taskletLogger.info("Transfer done file to server sucessfully: " + sbsOutputDoneFileName);
				} else {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful done file transfer: " + sbsOutputDoneFileName;
					setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				}
			}

			// If successful then save final statistics.
			if (successful) {
				// Save batch job completion results.
				jobExecution.getExecutionContext().putLong(CBS_TOTAL_PATIENTS_INCLUDED_KEY, totalPatientsIncluded);
			}
		}

		// Regardless of failure or successful batch run, delete local statement files if exist.
		if (sendCBSService.deleteCBSOutputResource(cbsOutputResourceLocalTempFilePath)) {
			taskletLogger.info("Deleted temporary local file sucessfully: " + cbsOutputResourceLocalTempFilePath);
		} else {

			// Don't make this a failure in case transfer had already been
			// successful.
			taskletLogger.warn("Unable to delete the temporary local file: " + cbsOutputResourceLocalTempFilePath);
		}

		if (sendCBSService.deleteCBSOutputResource(sbsOutputResourceLocalTempFilePath)) {
			taskletLogger.info("Deleted temporary local file sucessfully: " + sbsOutputResourceLocalTempFilePath);
		} else {

			// Don't make this a failure in case transfer had already been
			// successful.
			taskletLogger.warn("Unable to delete the temporary local file: " + sbsOutputResourceLocalTempFilePath);
		}

		return successful;
	}

	private String getOutputResource(final JobExecution jobExecution, final String outputResourceKey) {
		
		final String outputResource = jobExecution.getExecutionContext().getString(outputResourceKey);

		// If an error getting the resource then set the failure status and trigger roll back.
		if ((outputResource != null) && !outputResource.isEmpty()) {

			taskletLogger.info(outputResourceKey + ": " + outputResource);

		} else {

			// This is an unexpected and unrecoverable error.
			final String error = "Rollback Triggered - Could not obtain the resource, " + outputResourceKey;
			setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

		}
		
		return outputResource;
	}
	
	private String getJobParameter(final JobExecution jobExecution, final String jobParameterKey) {
		
		final String jobParameter = jobExecution.getJobParameters().getString(jobParameterKey);

		// If an error getting the jobParameter then set the failure status and trigger roll back.
		if ((jobParameter != null) && !jobParameter.isEmpty()) {

			taskletLogger.info(jobParameterKey + ": " + jobParameter);

		} else {

			// This is an unexpected and unrecoverable error.
			final String error = "Rollback Triggered - Could not obtain the job parameter, " + jobParameterKey;
			setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

		}
		
		return jobParameter;
	}
	
	/**
	 * Build a temporary filename for use on the target sftp server. The format
	 * of the temporary filename: Given filename: 'CBS-06022016.txt' would
	 * return temporary filename: 'CBS-06022016.txt.<timestamp>.sendcbs'
	 * 
	 * @param filename
	 *            The base filename.
	 * @return The temporary filename.
	 */
	private String getTemporaryFilename(final String filename) {
		if ((filename != null) && !filename.isEmpty()) {
			StringBuilder tempFilename = new StringBuilder(filename);
			tempFilename.append(".");
			tempFilename.append(TimeUtils.getTimestamp(TIMESTAMP_FORMAT));
			tempFilename.append(".");
			tempFilename.append(tempFilenamePostfix);
			return tempFilename.toString();
		}
		return null;
	}

	/**
	 * Get the done filename based on the output filename. The done filename is
	 * the same base filename but with ".don" extension instead of ".txt"
	 * 
	 * @param filename
	 *            The output filename.
	 * @return The done filename.
	 */
	private String getDoneFilename(final String filename) {

		return filename.substring(0, filename.lastIndexOf(".txt")) + ".don";
	}

}
