package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_MAX_RECORDS_PER_SITE;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemWriter;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PSRecord;
import gov.va.cpss.model.fps.PSSite;

public class SendCBSLoadItemWriter implements ItemWriter<List<CBSStmt>> {

	public static final String PS_SITES_CREATION_ERROR_STATUS = "FAILURE DURING PSSITES CREATION";

	public static final String NO_PRIMARY_SITE_ERROR_STATUS = "NO PRIMARY SITE FOUND";

	private final Logger logger;

	private JobExecution jobExecution;
	
	private SendCBSRuntimeState sendCBSRuntimeState;

	public SendCBSLoadItemWriter() {
		logger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public SendCBSRuntimeState getSendCBSRuntimeState() {
		return sendCBSRuntimeState;
	}

	public void setSendCBSRuntimeState(SendCBSRuntimeState sendCBSRuntimeState) {
		this.sendCBSRuntimeState = sendCBSRuntimeState;
	}

	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		// Save the JobExecution at the beginning
		jobExecution = stepExecution.getJobExecution();

	}

	@Override
	public void write(List<? extends List<CBSStmt>> items) throws Exception {
		logger.info("write");

		if (items != null) {
			for (List<CBSStmt> cbsStmtL : items) {
				if ((cbsStmtL == null) || (cbsStmtL.size() == 0))
					continue;

				generatePSSites(cbsStmtL);
			}
		}
	}

	private void generatePSSites(List<CBSStmt> cbsStmtL) {
		try {
			// Get the PSSites list from the runtime state. PSSite
			// records generated during this chunk will be added to this list.
			List<PSSite> psSitesL = sendCBSRuntimeState.getPSSitesList();

			CBSSiteStmt primarySite = getPrimarySite(cbsStmtL.get(0));
			if (primarySite == null) {
				stopJob(NO_PRIMARY_SITE_ERROR_STATUS, "No primary site found. Unable to continue.");
				return;
			}

			// The CBSStmts will be separated into a list of consolidated
			// statements and single statements
			List<CBSStmt> onlyCbsStmtL = new ArrayList<CBSStmt>();
			List<CBSStmt> onlySbsStmtL = new ArrayList<CBSStmt>();
			separateCbsStmts(cbsStmtL, onlyCbsStmtL, onlySbsStmtL);

			// Save the CBS Lists for this site in the runtime state
			sendCBSRuntimeState.getCBSListsBySite().put(primarySite.getStationNum(), onlyCbsStmtL);
			sendCBSRuntimeState.getSBSListsBySite().put(primarySite.getStationNum(), onlySbsStmtL);

			// Generate PSSite records for each of the statement list types.
			generatePSSitesForStmtList(onlyCbsStmtL, primarySite, psSitesL, PSRecord.DataType.PH);
			generatePSSitesForStmtList(onlySbsStmtL, primarySite, psSitesL, PSRecord.DataType.P1);

		} catch (Exception e) {
			StringBuilder error = new StringBuilder();
			error.append("Unrecoverable error occurred while creating PSSite records, ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());

			stopJob(PS_SITES_CREATION_ERROR_STATUS, error.toString());
		}
	}

	private CBSSiteStmt getPrimarySite(CBSStmt cbsStmt) {
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			if (cbsSiteStmt.getIsPrimary().isTrue()) {
				return cbsSiteStmt;
			}
		}

		return null;
	}

	/**
	 * Splits the list of CBSStmts into a list of consolidated statements and
	 * single statements.
	 * 
	 * @param cbsStmtL
	 *            Original CBSStmt list containing mixed statement types.
	 * @param onlyCbsStmtL
	 *            Consolidated statements will be added to this list.
	 * @param onlySbsStmtL
	 *            Single statements will be added to this list.
	 */
	private void separateCbsStmts(final List<CBSStmt> cbsStmtL, List<CBSStmt> onlyCbsStmtL,
			List<CBSStmt> onlySbsStmtL) {
		if (cbsStmtL == null)
			return;

		for (CBSStmt cbsStmt : cbsStmtL) {
			if (cbsStmt == null)
				continue;

			if ((cbsStmt.getSiteStmtL() != null) && cbsStmt.getSiteStmtL().size() > 1) {
				onlyCbsStmtL.add(cbsStmt);
			} else {
				onlySbsStmtL.add(cbsStmt);
			}
		}
	}

	private void generatePSSitesForStmtList(final List<CBSStmt> cbsStmtL, final CBSSiteStmt primarySite,
			final List<PSSite> psSitesL, final PSRecord.DataType statementType) {
		// Only generate a PS record for this type if there are statements for
		// this type.
		if (cbsStmtL.size() == 0)
			return;

		// PS fields identical for all PS records for this site

		// This effectively adds 1 to quotient if a remainder or leaves
		// quotient as-is if no remainder.
		final int psTotSeqNum = (cbsStmtL.size() + CBS_MAX_RECORDS_PER_SITE - 1) / CBS_MAX_RECORDS_PER_SITE;

		final String psFacilityNum = primarySite.getStationNum();
		final String psFacPhoneNum = primarySite.getStationPhoneNum();
		final Date psStatementDate = primarySite.getStatementDate();
		final Date psProcessDate = primarySite.getProcessDate();

		// PS fields unique to each PS records for this site.
		int psSeqNum = 1;
		int psTotStatement = 0;
		double psStatementVal = 0.0d;

		// Step 1 of 2: Generate a PSSite record for each set of
		// MAX_RECORDS_PER_SITE statements.
		for (CBSStmt cbsStmt : cbsStmtL) {
			psTotStatement++;
			psStatementVal += cbsStmt.getNewBalance().getDouble();
			if (psTotStatement == CBS_MAX_RECORDS_PER_SITE) {
				psSitesL.add(generatePSSite(psSeqNum, psTotSeqNum, psFacilityNum, psFacPhoneNum, psStatementDate,
						psProcessDate, psStatementVal, psTotStatement, statementType));

				psSeqNum++;
				psTotStatement = 0;
				psStatementVal = 0.0d;
			}
		}

		// Step 2 of 2: Handle majority of cases where the cbsStmt record
		// count is not on the exact boundary.
		// Wrap in sequence number check because the above loop will handle
		// case where cbsStmt record counts fall on exact boundary:
		// (psTotStatement % CBS_MAX_RECORDS_PER_SITE) == 0
		if (psSeqNum <= psTotSeqNum) {
			psSitesL.add(generatePSSite(psSeqNum, psTotSeqNum, psFacilityNum, psFacPhoneNum, psStatementDate,
					psProcessDate, psStatementVal, psTotStatement, statementType));
		}

	}

	private PSSite generatePSSite(final int psSeqNum, final int psTotSeqNum, final String psFacilityNum,
			final String psFacPhoneNum, final Date psStatementDate, final Date psProcessDate,
			final double psStatementVal, final int psTotStatement, final PSRecord.DataType statementType) {

		Money psStatementValMoney = new Money(psStatementVal, 11);

		PSSite psSite = new PSSite();
		psSite.setType(PSRecord.DataType.PS);
		psSite.setStatementType(statementType);

		psSite.setSeqNum(psSeqNum);
		psSite.setTotSeqNum(psTotSeqNum);
		psSite.setFacilityNum(psFacilityNum);
		psSite.setFacilityPhoneNum(psFacPhoneNum);
		psSite.setStatementDate(psStatementDate);
		psSite.setProcessDate(psProcessDate);
		psSite.setTotStatement(psTotStatement);
		psSite.setStatementVal(psStatementValMoney);

		logger.debug("Generated PSSite:" + psSite);
		return psSite;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private List<CBSStmt> stopJob(final String status) {
		// Log message.
		logger.error("PSSite creation encountered unrecoverable error and forced stop");
		// Set failure and message.
		setFailureStatus(status);
		// Stop job.
		jobExecution.stop();

		return null;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private List<CBSStmt> stopJob(final String status, final String message) {

		// Set failure.
		stopJob(status);

		// Set failure message.
		setFailureMessage(message);

		return null;
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		logger.error("PSSite creation failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		logger.error("PSSite creation failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}
}
