package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.NEW_STATUS_ID_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.database.JdbcCursorItemReader;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;


/**
 * 
 * 
 * 
 * @author Brad Pickle
 *
 */
public class SendCBSLoadItemReader extends JdbcCursorItemReader<String> implements StepExecutionListener {

    public static final String INITIALIZATION_ERROR_STATUS = "FAILURE INITIALIZING READ";

    private Logger logger;
    
    private ProcessStatusDAO processStatusDAO;
    
    private JobExecution jobExecution;

    public SendCBSLoadItemReader() {
    	logger = Logger.getLogger(this.getClass().getCanonicalName());
    }

	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
        logger.info("Before Step Execution");
        
        jobExecution = stepExecution.getJobExecution();
        
        putNewStatusIdInExecutionContext(stepExecution);
	}

	@Override
	public ExitStatus afterStep(StepExecution arg0) {
		return null;
	}
	
	private void putNewStatusIdInExecutionContext(StepExecution stepExecution) {
		try {
			final Integer newStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);
			if (newStmtStatusId == null) {
	            setFailureStatusAndMessage(INITIALIZATION_ERROR_STATUS,
	            		"Unable to obtain status mapping for CBSStmtStatus " + ProcessStatus.Status.NEW);
			} else {
				stepExecution.getExecutionContext().putLong(NEW_STATUS_ID_KEY, newStmtStatusId);
			}
		} catch (Exception e) {
            StringBuilder error = new StringBuilder();
            error.append("Unable to obtain status mapping for CBSStmtStatus ");
            error.append(ProcessStatus.Status.NEW);
            error.append(" because of ");
            error.append(e.getClass().getSimpleName());
            error.append("\nMessage: ");
            error.append(e.getMessage());
            
            setFailureStatusAndMessage(INITIALIZATION_ERROR_STATUS, error.toString());
		}
	}

    /**
     * Set the failure and message in the job execution context.
     */
    private void setFailureStatusAndMessage(final String status, final String message) {
        // Set job failure.
        setFailureStatus(status);
        
        // Set job failure message.
        setFailureMessage(message);
    }
    
    /**
     * Set the failure in the job execution context.
     */
    private void setFailureStatus(final String status) {
        // Log job failure status.
    	logger.error("Read failed with status: " + status);
        
        // Set job failure.
        jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
    }
    
    /**
     * Set the failure message in the job execution context.
     */
    private void setFailureMessage(final String message) {
        // Log job failure message.
    	logger.error("Read failure message: " + message);
        
        // Set job failure message.
        jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
    }
    
}
