package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;

import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.AfterStep;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.service.SendCBSService;

/**
 * @author Brad Pickle
 *
 */
public class SendCBSLoadItemProcessor implements ItemProcessor<String, List<CBSStmt>> {

	public static final String PROCESSING_ERROR_STATUS = "FAILURE DURING PROCESSING";

	private final Logger processorLogger;

	private JobExecution jobExecution;

	private SendCBSService sendCBSService;

	private long cbsMessageId;

	public SendCBSLoadItemProcessor() {
		processorLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public SendCBSService getSendCBSService() {
		return sendCBSService;
	}

	public void setSendCBSService(SendCBSService sendCBSService) {
		this.sendCBSService = sendCBSService;
	}

	public long getCbsMessageId() {
		return cbsMessageId;
	}

	public void setCbsMessageId(long cbsMessageId) {
		this.cbsMessageId = cbsMessageId;
	}

	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public List<CBSStmt> process(String stationNum) throws Exception {
		processorLogger.info("Loading into memory station number:" + stationNum);

		List<CBSStmt> cbsStmtL = null;

		try {
			cbsStmtL = sendCBSService.getMessageStatementsForSite(cbsMessageId, stationNum, true);
			if ((cbsStmtL == null) || (cbsStmtL.size() == 0)) {
				cbsStmtL = stopJob(PROCESSING_ERROR_STATUS,
						"Unable to obtain consolidated billing statements for station number " + stationNum);
			}
		} catch (Exception e) {
			processorLogger.error(e);
			cbsStmtL = stopJob(PROCESSING_ERROR_STATUS,
					"Unable to obtain consolidated billing statements for station number " + stationNum);
		}
		return cbsStmtL;
	}

	@AfterStep
	public void afterStep(StepExecution stepExecution) {
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private List<CBSStmt> stopJob(final String status) {
		// Log message.
		processorLogger.error("Processor execution encountered unrecoverable error and forced stop");
		// Set failure and message.
		setFailureStatus(status);
		// Stop job.
		jobExecution.stop();

		return null;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private List<CBSStmt> stopJob(final String status, final String message) {

		// Set failure.
		stopJob(status);

		// Set failure message.
		setFailureMessage(message);

		return null;
	}

	/**
	 * Set the failure and message in the job execution context.
	 */
	private void setFailureStatusAndMessage(final String status, final String message) {
		// Set job failure.
		setFailureStatus(status);
		// Set job failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		processorLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		processorLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
