package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;

import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;

/**
 * Set the messageId on NEW CBSStmt records to the messageId of the CBSMessage
 * record for this run.
 * 
 * @author Brad Pickle
 */
public class SendCBSAssociateCBSStmtsTasklet implements Tasklet, StepExecutionListener {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());
	
	private CBSStmtDAO cbsStmtDAO;
	
	private ProcessStatusDAO processStatusDAO;
	
	private long cbsMessageId;
	
	/**
	 * Set to true if no errors, there are CBSStmt records to process and execution should continue.
	 */
	private boolean successful = false;

	public CBSStmtDAO getCbsStmtDAO() {
		return cbsStmtDAO;
	}

	public void setCbsStmtDAO(CBSStmtDAO cbsStmtDAO) {
		this.cbsStmtDAO = cbsStmtDAO;
	}

	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	public long getCbsMessageId() {
		return cbsMessageId;
	}

	public void setCbsMessageId(long cbsMessageId) {
		this.cbsMessageId = cbsMessageId;
	}

	@Override
	public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception {

		logger.info("Begin execute");

		final Integer newStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);
		if (newStmtStatusId == null) {
			final String error = "Unable to obtain status mapping for: " + ProcessStatus.Status.NEW;
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);
		} else {
			int numStatements = cbsStmtDAO.updateMessageIdForStatus(newStmtStatusId, cbsMessageId);
			if (numStatements < 1) {
				final String error = "No consolidated statements found to send.";
				setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
						error);				
			} else {
				successful = true;
			}
		}

		logger.info("End execute");

		return RepeatStatus.FINISHED;
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	protected void setFailureStatus(JobExecution jobExecution, final String status, final String message) {

		// Log job failure status.
		logger.error("Job completion failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);

		// Log job failure message.
		logger.error("Job completion failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
		
		successful = false;
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		successful = false;
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {

		if (successful) {
			return ExitStatus.COMPLETED;
		}
		
		return ExitStatus.FAILED;
	}

}
