package gov.va.cpss.job.sendbill;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.SEND_BILL_ID_KEY;
import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.SEND_BILL_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.SEND_BILL_OUTPUT_RESOURCE_KEY;
//import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_TOTAL_PATIENTS_INCLUDED_KEY;
import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.TOTAL_BILL_PATIENT_DATA_PROCESSED;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.job.CbssJobBaseTransactionTasklet;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.service.SendBillDataService;
import gov.va.cpss.service.SftpService;
import gov.va.cpss.time.TimeUtils;

/**
 * Complete processing by transferring file to sftp server with temporary name,
 * setting appropriate message status and statement status, and renaming
 * temporary file on sftp server with correct name. If any errors occur the
 * database updates are rolled back and the batch job status is set to error
 * condition.
 * 
 * @author DNS   BROWNL
 */
public class SendBillProcessFileCompletionTasklet extends CbssJobBaseTransactionTasklet {

	private final String TIMESTAMP_FORMAT = "HH_mm_ss";

	private String tempFilenamePostfix;

	private String serverTargetDirectory;

	private SftpService sftpService;

	private SendBillDataService sendBillDataService;

	public String getTempFilenamePostfix() {
		return tempFilenamePostfix;
	}

	public void setTempFilenamePostfix(String tempFilenamePostfix) {
		this.tempFilenamePostfix = tempFilenamePostfix;
	}

	public String getServerTargetDirectory() {
		return serverTargetDirectory;
	}

	public void setServerTargetDirectory(String serverTargetDirectory) {
		this.serverTargetDirectory = serverTargetDirectory;
	}

	public SftpService getSftpService() {
		return sftpService;
	}

	public void setSftpService(SftpService sftpService) {
		this.sftpService = sftpService;
	}

	public SendBillDataService getSendBillDataService() {
		return sendBillDataService;
	}

	public void setSendBillDataService(SendBillDataService sendBillDataService) {
		this.sendBillDataService = sendBillDataService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("File Completion Tasklet Running!");

		// Save the jobExecution for reference.
		JobExecution jobExecution = chunkContext.getStepContext().getStepExecution().getJobExecution();

		// Obtain the temporary local file path as it will be needed regardless
		// of failure.
		final String sendBillOutputResourceLocalTempFilePath = jobExecution.getExecutionContext()
				.getString(SEND_BILL_OUTPUT_RESOURCE_KEY);

		// If an error getting the local file path then just return.
		if ((sendBillOutputResourceLocalTempFilePath != null) && !sendBillOutputResourceLocalTempFilePath.isEmpty()) {

			taskletLogger.info("cbsOutputResourceLocalTempFilePath: " + sendBillOutputResourceLocalTempFilePath);

		} else {

			// This is an unexpected and unrecoverable error.
			final String error = "Could not obtain the path to the temporary local output file";
			setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

			// Return now!
			return false;
		}

		// If the job has been successful then transfer file to server and
		// attempt to set appropriate final statement and message statuses.
		if (!jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {

			// Get filename values from the jobExecution or jobParameters.

			// The output filename.
			final String sendBillOutputFileName = jobExecution.getJobParameters()
					.getString(SEND_BILL_OUTPUT_FILE_NAME_KEY);
			if ((sendBillOutputFileName != null) && !sendBillOutputFileName.isEmpty()) {

				taskletLogger.info("cbsOutputFileName: " + sendBillOutputFileName);

			} else {

				// This is an unexpected and unrecoverable error.
				final String error = "Rollback Triggered - Could not obtain the output file name";
				setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

				// Flag to do rollback.
				successful = false;
			}

			// The temporary filename used to transfer to the sftp server.
			final String sendBillTargetTemporaryFileName = getTemporaryFilename(sendBillOutputFileName);
			if (successful && (sendBillTargetTemporaryFileName != null)) {

				taskletLogger.info("cbsTargetTemporaryFileName: " + sendBillTargetTemporaryFileName);
			} else {

				// This is an unexpected and unrecoverable error.
				final String error = "Rollback Triggered - Could not create a temporary server filename";
				setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

				// Flag to do rollback.
				successful = false;
			}

			// Get latest PatAcntBalMsg ID from the jobExecution context.
			if(successful)
			{
				final Long msgId = jobExecution.getJobParameters().getLong(SEND_BILL_ID_KEY);
			
				if ((msgId == null) || (msgId < 1)) {
	
					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Could not obtain a valid message ID: " + msgId;
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);
	
					// Flag to do rollback.
					successful = false;
				} else {
					successful = updateStatusToComplete(successful, jobExecution, msgId);
					//Update PatAcntBalMsgID for SITEBAL -
					sendBillDataService.updateSiteBalPatAcntBalMsgId(msgId);
					//Update total Number of Patients
					taskletLogger.info("Number Of Patients Logged in completion Tasklet: "+jobExecution.getExecutionContext().getLong(TOTAL_BILL_PATIENT_DATA_PROCESSED));
					sendBillDataService.updateTotalPatientProcessed(jobExecution.getExecutionContext().getLong(TOTAL_BILL_PATIENT_DATA_PROCESSED), msgId);
				}
			}
			
			// Send file with temporary filename (with timestamp extension).
			// Example:
			// 'CBS-06022016.txt' would be temporarily sent as
			// 'CBS-06022016.txt.<timestamp>.sendcbs'
			if (successful && sftpService.ftpFileToServerWithName(sendBillOutputResourceLocalTempFilePath,
					sendBillTargetTemporaryFileName, serverTargetDirectory)) {

				taskletLogger.info("Sent file to server sucessfully: " + sendBillTargetTemporaryFileName);
			} else {

				// This is an unexpected and unrecoverable error.
				final String error = "Rollback Triggered - Unsuccessful file transfer: "
						+ sendBillTargetTemporaryFileName;
				setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

				// Flag to do rollback.
				successful = false;
			}

			// Rename the file on the server (by removing the temporary
			// '.<timestamp>.sendbill' extension).
			if (successful && sftpService.ftpRenameFileInDirectory(sendBillTargetTemporaryFileName,
					sendBillOutputFileName, serverTargetDirectory)) {

				taskletLogger.info("Renamed file on server sucessfully: " + sendBillOutputFileName);

			} else {

				// This is an unexpected and unrecoverable error.
				final String error = "Rollback Triggered - Unsuccessful file rename: " + sendBillOutputFileName;
				setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

				// Flag to do rollback.
				successful = false;
			}

			// Send a done file to the server.
			final String sendBillOutputDoneFileName = getDoneFilename(sendBillOutputFileName);
			if (successful
					&& sftpService.ftpEmptyFileToServerWithName(sendBillOutputDoneFileName, serverTargetDirectory)) {

				taskletLogger.info("Transfer done file to server sucessfully: " + sendBillOutputDoneFileName);

			} else {

				// This is an unexpected and unrecoverable error.
				final String error = "Rollback Triggered - Unsuccessful done file transfer: "
						+ sendBillOutputDoneFileName;
				setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

				// Flag to do rollback.
				successful = false;
			}

			// If successful then save final statistics.
			if (successful) {
				taskletLogger.info("Successful SFTP!");
				// TODO?
				// // Save batch job completion results.
				
			}
		}

		// Regardless of failure or successful batch run, delete local CBS file
		// if exists.
		if (sendBillDataService.deleteBillDataOutputResource(sendBillOutputResourceLocalTempFilePath)) {

			taskletLogger.info("Deleted temporary local file sucessfully: " + sendBillOutputResourceLocalTempFilePath);
		} else {

			// Don't make this a failure in case transfer had already been
			// successful.
			taskletLogger.warn("Unable to delete the temporary local file: " + sendBillOutputResourceLocalTempFilePath);
		}

		return successful;
	}

	private boolean updateStatusToComplete(boolean successful, JobExecution jobExecution, final Long msgId) {
		if(sendBillDataService.setMessageStatus(msgId, ProcessStatus.Status.SUCCESS))
		{
			taskletLogger.info("Updated message status to success for message ID: " + msgId);
		}
		else
		{
			// This is an unexpected and unrecoverable error.
			final String error = "Rollback Triggered - Unsuccessful attempting to update message status to success for message ID: "
					+ msgId;
			setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

			// Flag to do rollback.
			successful = false;
		}
		return successful;
	}

	/**
	 * Build a temporary filename for use on the target sftp server. The format
	 * of the temporary filename: Given filename: 'CBS-06022016.txt' would
	 * return temporary filename: 'CBS-06022016.txt.<timestamp>.sendcbs'
	 * 
	 * @param filename
	 *            The base filename.
	 * @return The temporary filename.
	 */
	private String getTemporaryFilename(final String filename) {
		if ((filename != null) && !filename.isEmpty()) {
			StringBuilder tempFilename = new StringBuilder(filename);
			tempFilename.append(".");
			tempFilename.append(TimeUtils.getTimestamp(TIMESTAMP_FORMAT));
			tempFilename.append(".");
			tempFilename.append(tempFilenamePostfix);
			return tempFilename.toString();
		}
		return null;
	}

	/**
	 * Get the done filename based on the output filename. The done filename is
	 * the same base filename but with ".don" extension instead of ".txt"
	 * 
	 * @param filename
	 *            The output filename.
	 * @return The done filename.
	 */
	private String getDoneFilename(final String filename) {

		return filename.substring(0, filename.lastIndexOf(".txt")) + ".don";
	}

}
