package gov.va.cpss.job.sendbill;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;
import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.TOTAL_BILL_PATIENT_DATA_PROCESSED;
import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.SEND_BILL_TOTAL_RECORD_COUNT;
import java.util.ArrayList;
import java.util.List;

import static gov.va.cpss.job.sendbill.SendBillDataProcessingConstants.SEND_BILL_ID_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ItemProcessListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.AfterStep;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.model.bal.SiteBalance;
import gov.va.cpss.service.SendBillDataService;

/**
 * Custom ItemProcessor used by the Process FPS Data batch job to register the
 * ICN.
 * 
 * @author DNS  
 */
public class SendBillDataItemProcessor implements ItemProcessor<SiteBalance, List<SendBillStringDTO>>,
		ItemProcessListener<SiteBalance, List<SendBillStringDTO>> {

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt processing.
	 */
	private boolean forceStop = false;

	private final Logger processorLogger;

	private JobExecution jobExecution;

	private int batchRunId;

	private SendBillDataService sendBillDataService;

	/**
	 * @return the sendBillDataService
	 */
	public SendBillDataService getSendBillDataService() {
		return sendBillDataService;
	}

	/**
	 * @param sendBillDataService
	 *            the sendBillDataService to set
	 */
	public void setSendBillDataService(SendBillDataService sendBillDataService) {
		this.sendBillDataService = sendBillDataService;
	}

	private SendBillStringDTO combinedDTO;

	/**
	 * @return the combinedDTO
	 */
	public SendBillStringDTO getCombinedDTO() {
		return combinedDTO;
	}

	/**
	 * @param combinedDTO
	 *            the combinedDTO to set
	 */
	public void setCombinedDTO(SendBillStringDTO combinedDTO) {
		this.combinedDTO = combinedDTO;
	}

	public SendBillDataItemProcessor() {
		processorLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	@Override
	public List<SendBillStringDTO> process(SiteBalance pojo) throws Exception {
		
		processorLogger.debug("Begin Process Currently in Item Processor");
		processorLogger.debug("Total Patient Count has Been Decreased: " + jobExecution.getExecutionContext().getInt(SEND_BILL_TOTAL_RECORD_COUNT));
		// Do not attempt to process if forced stop.
		if (forceStop) {
			processorLogger.debug("FORCE STOP");
			return null;
		}
		
		List<SendBillStringDTO> writeDTO = new ArrayList<SendBillStringDTO>();
		
			try {
				
				long currentAccountId = pojo.getCbsAccount().getId();
				
				if (getCombinedDTO() != null) {
					if (currentAccountId == getCombinedDTO().getCbssId()) {
						setCombinedDTO(getSendBillDataService().processExistingBillRecord(pojo, getCombinedDTO()));
						accrueProcessingCount(TOTAL_BILL_PATIENT_DATA_PROCESSED, 1);
						processorLogger.debug("Number of Patients Processed is: " + jobExecution.getExecutionContext().getLong(TOTAL_BILL_PATIENT_DATA_PROCESSED));
						
						if(decreaseTotalPatienCount(SEND_BILL_TOTAL_RECORD_COUNT, 1))
						{
							writeDTO.add(getCombinedDTO());
						}
						
					} else {
						//jobExecution.getExecutionContext().
						// Existing is ready to write.
						processorLogger.debug("Number of Patients Processed is: " + jobExecution.getExecutionContext().getLong(TOTAL_BILL_PATIENT_DATA_PROCESSED));
						processorLogger.debug("--------------------The Data is being Written---------------");
						writeDTO.add(getCombinedDTO());
						processorLogger.debug("Write Complete Recieving New Record");
						// Received a new record. True
						setCombinedDTO(getSendBillDataService().processNewBillRecord(pojo));
						accrueProcessingCount(TOTAL_BILL_PATIENT_DATA_PROCESSED, 1);
						
						if(decreaseTotalPatienCount(SEND_BILL_TOTAL_RECORD_COUNT, 1))
						{
							writeDTO.add(getCombinedDTO());
						}
						
						
					}
				} else {
					// Create new record True
						setCombinedDTO(getSendBillDataService().processNewBillRecord(pojo));
						accrueProcessingCount(TOTAL_BILL_PATIENT_DATA_PROCESSED, 1);
						processorLogger.debug("Number of Patients Processed is: " + jobExecution.getExecutionContext().getLong(TOTAL_BILL_PATIENT_DATA_PROCESSED));
						if(decreaseTotalPatienCount(SEND_BILL_TOTAL_RECORD_COUNT, 1))
						{
							writeDTO.add(getCombinedDTO());
						}
						
				}

			} catch (Exception e) {

				// Unrecoverable error so stop the job.
				stopJob(PROCESSING_FAILURE_STATUS, "Unrecoverable processing error: " + e.getMessage());
			}

	    
		processorLogger.debug("End Process");
		if(writeDTO.isEmpty())
		{
			return null;
		}
		return writeDTO;
		
	}

	/**
	 * Before processing capture the job execution and the received id for data
	 * processing.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		processorLogger.info("Before Step Execution");

		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is
		// processed.
		jobExecution = stepExecution.getJobExecution();
		// Sets The Default Patients Processed Value
		jobExecution.getExecutionContext().putLong(TOTAL_BILL_PATIENT_DATA_PROCESSED, 0);
		// Save the batchRunId at the beginning of the step.

	}

	/**
	 * After processing check for unexpected conditions that suggest an error.
	 * 
	 * @param stepExecution
	 */
	@AfterStep
	public void afterStep(StepExecution stepExecution) {

		processorLogger.info("After Step Execution");
	}

	@Override
	public void beforeProcess(SiteBalance item) {
		processorLogger.debug("Before Process");
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			processorLogger.error("System failure detected.");
			forceStop = true;
		}
		// Get the id of the BatchRun record for this job.
		setBatchRunId(jobExecution.getJobParameters().getLong(SEND_BILL_ID_KEY).intValue());
	}

	@Override
	public void afterProcess(SiteBalance item, List<SendBillStringDTO> result) {
		// Do not need to do anything after process.
	}

	@Override
	public void onProcessError(SiteBalance item, Exception e) {
		processorLogger.error("Processor encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to process item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		error.append("\nCause: ");
		error.append(e.getCause().getMessage().trim());

		// Set failure and message.
		stopJob(PROCESSING_FAILURE_STATUS, "Unrecoverable processor error");
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		processorLogger.error("Processor execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		processorLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		processorLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

	public int getBatchRunId() {
		return batchRunId;
	}

	public void setBatchRunId(int batchRunId) {
		this.batchRunId = batchRunId;
	}

	/**
	 * Accrue the given processing count in the job execution context.
	 * 
	 * @param key
	 *            Name of the processing count.
	 * @param numOfBills
	 *            Amount to add to the processing count.
	 */
	private void accrueProcessingCount(final String key, final int numOfBills) {
		long totalStmts = jobExecution.getExecutionContext().getLong(key);
		totalStmts += numOfBills;
		jobExecution.getExecutionContext().putLong(key, totalStmts);
	}

	private boolean decreaseTotalPatienCount(final String key, final Integer numOfPatients) {
		int totalStmts = jobExecution.getExecutionContext().getInt(key);
		processorLogger.debug("Total Patient Count is: " + totalStmts);
		if (totalStmts > 0) {
			totalStmts = totalStmts - numOfPatients;
			processorLogger.debug("Total Patient Count has Been Decreased: " + totalStmts);
			jobExecution.getExecutionContext().putInt(key, totalStmts);
			if (totalStmts == 0) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}

}
