package gov.va.cpss.job.sendapps;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.job.CbssJobBaseTransactionTasklet;
import gov.va.cpss.job.sendapps.SendAPPSProcessingConstants;
import gov.va.cpss.model.sendapps.SendAppsPSRecord;
import gov.va.cpss.model.ps.RecordType;
import gov.va.cpss.service.SendAPPSService;

/**
 * Splits the APPS file into a sequence of smaller files to send to AITC-CCPC.
 * 
 * @author Andrew Vance
 *
 */
@SuppressWarnings("nls")
public final class SendAPPSFileSplitterTasklet extends CbssJobBaseTransactionTasklet {

	private static final String STATEMENT_RECORD_PREFIX = RecordType.PH.getName();

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	private int maxStatementPerFile; 

	private String appsFilePath;

	private int currentNumberOfStatements;

	private int outputFileIndex;

	private BufferedWriter writer;

	private SendAPPSService sendAPPSService;

	public int getMaxStatementPerFile() {
		return this.maxStatementPerFile;
	}

	public void setMaxStatementPerFile(int inMaxStatementPerFile) {
		this.maxStatementPerFile = inMaxStatementPerFile;
	}

	public SendAPPSService getSendAPPSService() {
		return this.sendAPPSService;
	}

	public void setSendAPPSService(SendAPPSService inSendAPPSService) {
		this.sendAPPSService = inSendAPPSService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {
		this.logger.info("Splitting APPS file into chunks of " + this.maxStatementPerFile + " statements per file");

		this.appsFilePath = this.sendAPPSService.getStagingDirectory() + File.separator
				+ getJobParameter(chunkContext.getStepContext().getStepExecution().getJobExecution(),
						SendAPPSProcessingConstants.APPS_OUTPUT_FILE_NAME_KEY);

		try (final BufferedReader reader = Files.newBufferedReader(Paths.get(this.appsFilePath))) {
			if(reader == null) {
				this.logger.error("Could not initialize reader with filepath: " + Paths.get(this.appsFilePath));
				return false;
			}
 
			initializeFirstWriter();

			if(this.writer == null) {
				this.logger.error("Could not initialize writer");
				return false;
			}
			
			// Write PS record
			final SendAppsPSRecord psRecord = new SendAppsPSRecord();
			psRecord.setType(RecordType.PS);
			psRecord.setTotPatient( ((Long) chunkContext.getStepContext().getJobExecutionContext().getOrDefault(SendAPPSProcessingConstants.APPS_TOTAL_PATIENTS_INCLUDED_KEY, Integer.valueOf(0))).longValue() ); 
			psRecord.setStatementYear(this.sendAPPSService.getPreviousYear());
			this.writer.write(psRecord.toFileOutputString());
			this.writer.newLine();

			while(reader.ready()) {

				final String currentLine = reader.readLine();

				if(currentLine.startsWith(STATEMENT_RECORD_PREFIX)) {
					this.currentNumberOfStatements++;

					if(this.currentNumberOfStatements > this.maxStatementPerFile) {
						initializeNextWriter();
						if(this.writer == null) {
							this.logger.error("Could not open writer for next file");
							return false;
						}
					}
				}

				this.writer.write(currentLine);
				this.writer.newLine();
			}			

			this.writer.close();			
			reader.close();

		} catch (IOException e) {
			this.logger.error("File Reader exception - cannot open or read file: " + e.getMessage());
			//e.printStackTrace();
			return false;
		}

		if(deleteAppsFile()) {
			this.logger.debug("Deleted file " + this.appsFilePath);
		} else {
			this.logger.warn("Could not delete file " + this.appsFilePath);
			return false;
		}

		return true;
	}

	/**
	 * Creates and opens the first output file in the sequence and sets outputFileIndex and
	 * currentNumberOfStatements to their initial state.
	 */
	private void initializeFirstWriter() {
		this.outputFileIndex = 1;
		this.currentNumberOfStatements = 0;	

		try {
			this.writer = Files.newBufferedWriter(Paths.get(getSplitFilePath()));
		} catch (IOException e) {
			e.printStackTrace();
			this.writer = null;
		}
	}
	
	/**
	 * Opens the next output file in sequence after closing the current file and updating the parameters.
	 */
	private void initializeNextWriter() {
		this.outputFileIndex++;
		this.currentNumberOfStatements=1;

		try {
			this.writer.close();
			this.writer = Files.newBufferedWriter(Paths.get(getSplitFilePath()));
		} catch (IOException e) {
			this.writer = null;
			e.printStackTrace();
		}	
	}

	/**
	 * Deletes the master APPS file containing all records.
	 * @return true if the file was deleted
	 */
	private boolean deleteAppsFile() {
		return new File(this.appsFilePath).delete();
	}

	/**
	 * Builds a file path for the current file in sequence.
	 * @return the complete file path
	 */
	private String getSplitFilePath() {
		return this.appsFilePath.substring(0, this.appsFilePath.lastIndexOf(".txt"))
				+ String.format("_%d.txt", Integer.valueOf(this.outputFileIndex));
	}

	private String getJobParameter(final JobExecution jobExecution, final String jobParameterKey) {

		final String jobParameter = jobExecution.getJobParameters().getString(jobParameterKey);

		// If an error getting the jobParameter then set the failure status and trigger roll back.
		if ((jobParameter != null) && !jobParameter.isEmpty()) {

			taskletLogger.info(jobParameterKey + ": " + jobParameter);

		} else {

			// This is an unexpected and unrecoverable error.
			final String error = "Rollback Triggered - Could not obtain the job parameter, " + jobParameterKey;
			setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

		}

		return jobParameter;
	}
}
