package gov.va.cpss.job.loadbill;

import static gov.va.cpss.job.CbssJobProcessingConstants.READ_FAILURE_STATUS;
import static gov.va.cpss.job.loadbill.LoadBillProcessingConstants.PU_HEADER_COMPLETE_KEY;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;

import org.springframework.batch.item.ExecutionContext;
import org.springframework.batch.item.ParseException;
import org.springframework.batch.item.UnexpectedInputException;

import gov.va.cpss.job.CbssSftpItemReader;
import gov.va.cpss.job.CbssStreamToQueueThread;
import gov.va.cpss.model.loadbill.LoadBillDetail;
import gov.va.cpss.model.loadbill.LoadBillDetailValidator;
import gov.va.cpss.model.loadbill.LoadBillHeader;
import gov.va.cpss.model.loadbill.LoadBillHeaderValidator;
import gov.va.cpss.model.loadbill.LoadBillRecord;

public class LoadBillSftpLineItemReader<T> extends CbssSftpItemReader<Map.Entry<LoadBillHeader, List<LoadBillDetail>>> {
	
	private int headerCount = 1;
	private Set<String> facilitySet = new HashSet<String>();
	
	@Override
	protected CbssStreamToQueueThread getQueueBuilderThread(long expectedByteCount, InputStream in,
			BlockingQueue<String> outputQueue) {
		try {
			return new CbssStreamToQueueThreadLoadBillImpl(expectedByteCount, in, outputQueue);
		} catch (UnsupportedEncodingException e) {
			readerLogger.error("Error creating data stream processing thread: " + e.getMessage());
		}
		return null;
	}

	@Override
	public Map.Entry<LoadBillHeader, List<LoadBillDetail>> read()
			throws Exception, UnexpectedInputException, ParseException {
		
//		readerLogger.info("Attempting Read...");
		
		// Flag to indicate if data error was detected.
		boolean dataError = false;
		ExecutionContext context = jobExecution.getExecutionContext();

		// This will return a map entry of header to list of details.
		LoadBillHeader header = null;
		List<LoadBillDetail> detailL = new ArrayList<>();

		// Read the next line from the queue.
		// The returned line should return a mapped object built from the tokenizer.
		Object line = super.read();
		
		// If read is null then just return null.
		if (line == null) {
			return null;
		}

		long expectedDetailCount = -1;
		long detailCounter = 0;
		StringBuffer strBuff = new StringBuffer();
		while (line != null) {
			
//			readerLogger.info("Read record: " + ((LoadBillRecord)line).toString());

			if (line instanceof LoadBillHeader) {
				
//				readerLogger.info("   Read header Q: |" + ((LoadBillHeader)line).toString() + "|");
				
				if (header == null) {
					header = (LoadBillHeader) line;
					// Validate the header to ensure it has proper information
					if(LoadBillHeaderValidator.hasErrors(header)) {
						// If there is an invalid header, consider the whole file invalid.
						dataError = true;
						setFailureStatusAndMessage(READ_FAILURE_STATUS, "Invalid header found: " + header);
						break;
					}
					if(headerCount != Integer.parseInt(header.getMessageNum())) {
						dataError = true;
						setFailureStatusAndMessage(READ_FAILURE_STATUS,
								"Invalid header found (message number is incorrect): " + header);
						break;
					}
					int totalMessageNum = Integer.parseInt(header.getTotMessageNum());
					if(totalMessageNum == 0) {
						headerCount++;
						// Set the complete header flag to false in the context.
						// Files should always end up complete so if this flag is not set then the file was invalid.
						jobExecution.getExecutionContext().putString(PU_HEADER_COMPLETE_KEY, Boolean.FALSE.toString());
					}
					else if(headerCount == totalMessageNum) {
						headerCount = 1;
						// Set the complete header flag to true in the context.
						// Files should always end up complete so if this flag is not set then the file was invalid.
						jobExecution.getExecutionContext().putString(PU_HEADER_COMPLETE_KEY, Boolean.TRUE.toString());
					}
					else {
						dataError = true;
						setFailureStatusAndMessage(READ_FAILURE_STATUS,
								"Invalid header found (total number is not equal to current message number or 0): " + header);
						break;
					}
					// Only increment the count if the facility was not already in the set
					if(facilitySet.add(header.getStationNum())) {
						int previousFacilityTotal = 0;
						if(context.containsKey(LoadBillProcessingConstants.TOTAL_FACILITY_COUNT_KEY)) {
							previousFacilityTotal = context.getInt(LoadBillProcessingConstants.TOTAL_FACILITY_COUNT_KEY);
						}
						context.putInt(LoadBillProcessingConstants.TOTAL_FACILITY_COUNT_KEY, previousFacilityTotal + 1);
					}
					expectedDetailCount = Integer.parseInt(header.getTotNumPatientRecords());
					if (expectedDetailCount <= 0) {
						// Expected detail count is invalid.
						dataError = true;
						setFailureStatusAndMessage(READ_FAILURE_STATUS, "Detail count in header was invalid: " + expectedDetailCount);
						break;
					}
				} else {
					// Read a header unexpectedly.
					dataError = true;
					setFailureStatusAndMessage(READ_FAILURE_STATUS, "Read a header in middle of header processing");
					break;
				}

			} else if (line instanceof LoadBillDetail) {
				
//				readerLogger.info("   Read detail Q: |" + ((LoadBillDetail)line).toString() + "|");

				LoadBillDetail detail = (LoadBillDetail) line;
				if(LoadBillDetailValidator.hasErrors(detail)) {
					++detailCounter;
					strBuff.append(String.format("%s: %s\n", header.getStationNum(), detail.getIcnNumber()));
					if(detailCounter >= expectedDetailCount) {
						// If the specified amount of PD records are read, break out of the loop to go to the next
						// PU that is expected.
						break;
					}
					else {
						// Advance to the next line after getting an error.
						line = (LoadBillRecord) super.read();
						continue;
					}
				}
				if (header != null) {

					detailL.add(detail);
					++detailCounter;
					if (detailCounter == expectedDetailCount) {
						// Processing is successful.
						break;
					} else if (detailCounter > expectedDetailCount) {
						// Invalid detail ID.
						dataError = true;
						setFailureStatusAndMessage(READ_FAILURE_STATUS, "Detail value is too large or invalid: " + detail.getPatientAccountNumber());
						break;
					}

				} else {
					// Header should not be null at this point.
					dataError = true;
					setFailureStatusAndMessage(READ_FAILURE_STATUS, "Read a detail record but it has no header associated with it");
					break;
				}

			} else {

				// Unrecognized record.
				dataError = true;
				setFailureStatusAndMessage(READ_FAILURE_STATUS, "Unrecognized record type");
				break;
			}

			line = (LoadBillRecord) super.read();
		}
		
		// If there was a data error then just return null.
		// This assumes when the data error was detected the failure was set in the jobExecution context.
		if (dataError) {
			return null;
		}

		// Do final sanity check if detail count is correct.
		if (detailL.size() <= expectedDetailCount) {
			if(!context.containsKey(LoadBillProcessingConstants.DETECTED_DETAIL_ERRORS_KEY)){
				context.putString(LoadBillProcessingConstants.DETECTED_DETAIL_ERRORS_KEY, strBuff.toString());
			}
			else{
				context.putString(LoadBillProcessingConstants.DETECTED_DETAIL_ERRORS_KEY, 
						context.getString(LoadBillProcessingConstants.DETECTED_DETAIL_ERRORS_KEY) + strBuff.toString());
			}
			Map.Entry<LoadBillHeader, List<LoadBillDetail>> entry = new AbstractMap.SimpleEntry<>(header, detailL);
			return entry;

		} else {

			// Error condition.
			setFailureStatusAndMessage(READ_FAILURE_STATUS, "Detail list did not match expected size.");
		}
		
		return null;
	}

}
