package gov.va.cpss.job.loadbill;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ExecutionContext;
import org.springframework.batch.item.ItemWriter;

import gov.va.cpss.model.bal.SiteBalance;
import gov.va.cpss.service.LoadBillService;

/**
 * Implementation of ItemWriter used to handle writing of site balances when
 * processing.
 * 
 * @author DNS   
 */
public class LoadBillDelegateMultiItemWriter
		implements ItemWriter<List<SiteBalance>>, ItemWriteListener<List<SiteBalance>>, StepExecutionListener {

	private final Logger writerLogger = Logger.getLogger(LoadBillDelegateMultiItemWriter.class.getCanonicalName());

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt writes.
	 */
	private boolean forceStop = false;

	private JobExecution jobExecution;
	
	private LoadBillService loadBillService;

	/*
	 * Delegate writer used to take advantage of batch writing site transaction
	 * records. Only writes new Site Balance records.
	 */
	private ItemWriter<SiteBalance> newSiteBalDatabaseItemWriter;
	
	/*
	 * Delegate writer used to take advantage of batch writing site transaction
	 * records. Only updates existing Site Balance recods.
	 */
	private ItemWriter<SiteBalance> existingSiteBalDatabaseItemWriter;

	public LoadBillService getLoadBillService() {
		return this.loadBillService;
	}
	
	public void setLoadBillService(LoadBillService loadBillService) {
		this.loadBillService = loadBillService;
	}
	
	public ItemWriter<SiteBalance> getNewSiteBalDatabaseItemWriter() {
		return newSiteBalDatabaseItemWriter;
	}

	public void setNewSiteBalDatabaseItemWriter(ItemWriter<SiteBalance> newSiteBalDatabaseItemWriter) {
		this.newSiteBalDatabaseItemWriter = newSiteBalDatabaseItemWriter;
	}
	
	public ItemWriter<SiteBalance> getExistingSiteBalDatabaseItemWriter() {
		return existingSiteBalDatabaseItemWriter;
	}

	public void setExistingSiteBalDatabaseItemWriter(ItemWriter<SiteBalance> existingSiteBalDatabaseItemWriter) {
		this.existingSiteBalDatabaseItemWriter = existingSiteBalDatabaseItemWriter;
	}

	@Override
	public void write(List<? extends List<SiteBalance>> items) throws Exception {

		writerLogger.debug("Begin Write");

		// Do not attempt to write if forced stop.
		if (forceStop) {
			return;
		}

		try {
			// Combined list of site balance lists.
			List<SiteBalance> newSiteBalL = new ArrayList<>();
			List<SiteBalance> existingSiteBalL = new ArrayList<>();
			// Loop through all of the statements to write.
			
			for (List<SiteBalance> sbL : items) {
				for(SiteBalance siteBal: sbL) {
					// If a record with the same station and account numbers is present, update that record with new information.
					if(loadBillService.getSiteBalanceByAcntNumAndStationNum(siteBal.getPatientDfn(), siteBal.getStationNum()) == null) {
						newSiteBalL.add(siteBal);
					}
					else {
						existingSiteBalL.add(siteBal);
					}
				}
			}

			// We can use batch to save the combined list.
			// Ensure that at least one record is being updated or added.
			if (!newSiteBalL.isEmpty() || !existingSiteBalL.isEmpty()) {
				writerLogger.debug("Writing new Site Balances");
				newSiteBalDatabaseItemWriter.write(newSiteBalL);
				writerLogger.debug("Updating existing Site Balances");
				existingSiteBalDatabaseItemWriter.write(existingSiteBalL);
				// Keep a running total of the amount of facilities and patients
				ExecutionContext execContext = jobExecution.getExecutionContext();
				int previousPatientTotal = 0;
				if(execContext.containsKey(LoadBillProcessingConstants.TOTAL_PATIENT_COUNT_KEY)) {
					previousPatientTotal = execContext.getInt(LoadBillProcessingConstants.TOTAL_PATIENT_COUNT_KEY);
				}
				execContext.putInt(LoadBillProcessingConstants.TOTAL_PATIENT_COUNT_KEY, 
						newSiteBalL.size() + existingSiteBalL.size() + previousPatientTotal);
			}
		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Error during write because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}

		writerLogger.debug("End Write");
	}

	@Override
	public void beforeWrite(List<? extends List<SiteBalance>> items) {
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			writerLogger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void afterWrite(List<? extends List<SiteBalance>> items) {
	}

	@Override
	public void onWriteError(Exception e, List<? extends List<SiteBalance>> items) {

		writerLogger.error("Writer encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to write item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());
		}

		// Set failure and message.
		stopJob(WRITE_FAILURE_STATUS, "Unrecoverable writer error");
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		writerLogger.info("Before Step Execution");
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is
		// processed.
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		writerLogger.info("After Step Execution");
		// Do not need to do anything after step.
		return null;
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		writerLogger.error("Writer execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		writerLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		writerLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
