package gov.va.cpss.job.loadbill;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.RECEIVED_ID_KEY;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ItemProcessListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.bal.SiteBalRec;
import gov.va.cpss.model.bal.SiteBalance;
import gov.va.cpss.model.cbs.CBSAccount;
import gov.va.cpss.model.icn.VistaAccount;
import gov.va.cpss.model.loadbill.LoadBillDetail;
import gov.va.cpss.model.loadbill.LoadBillHeader;
import gov.va.cpss.service.LoadBillService;
import gov.va.cpss.service.VistaAccountService;
import gov.va.cpss.util.CpssUtils;

public class LoadBillDataProcessor
		implements ItemProcessor<Entry<LoadBillHeader, List<LoadBillDetail>>, List<SiteBalance>>,
		ItemProcessListener<Entry<LoadBillHeader, List<LoadBillDetail>>, List<SiteBalance>> {

	private static final Logger processorLogger = Logger.getLogger(LoadBillDataProcessor.class.getCanonicalName());

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt processing.
	 */
	private boolean forceStop = false;
	
	private JobExecution jobExecution;
	
	private Long siteBalRecId;
	
	private LoadBillService loadBillService;
	
	private VistaAccountService vistaAccountService;

	public LoadBillService getLoadBillService() {
		return loadBillService;
	}

	public void setLoadBillService(LoadBillService loadBillService) {
		this.loadBillService = loadBillService;
	}

	public VistaAccountService getVistaAccountService() {
		return vistaAccountService;
	}

	public void setVistaAccountService(VistaAccountService vistaAccountService) {
		this.vistaAccountService = vistaAccountService;
	}

	@Override
	public List<SiteBalance> process(Entry<LoadBillHeader, List<LoadBillDetail>> entry) throws Exception {

		processorLogger.debug("Processing header: " + entry.getKey().toString());

		// For each PU, use the accompanying PD's to sum and build a list to
		// convert to SiteBal objects
		// Map based on the ICN number
		HashMap<String, SiteBalance> pdMap = new HashMap<String, SiteBalance>();
		SimpleDateFormat format = LoadBillProcessingConstants.DATE_FORMAT;
		
		for (LoadBillDetail detail : entry.getValue()) {
			
			// Update the Load Bill Detail from the VistaAccount table.
			// Upon successful return the detail record should have the most up to date ICN from VistaAccount/MVI Registration.
			if (updateLoadBillDetailAccountRegistration(entry.getKey(), detail)) {	
				String icn = detail.getIcnNumber();
				if (pdMap.get(icn) == null) {
					// Create new SiteBalance
					SiteBalance siteBalance = new SiteBalance(0);
					CBSAccount cbsAccount = new CBSAccount();
					cbsAccount.setId(detail.getCbssAcntId());
					siteBalance.setCbsAccount(cbsAccount);
					siteBalance.setStationNum(entry.getKey().getStationNum());
					siteBalance.setOldAcctNum(detail.getPatientAccountNumber());
					siteBalance.setPatientDfn(Long.parseLong(detail.getPatientDfn()));
					siteBalance.setBalance(AITCDollar.fromString(detail.getPatientSubtotal()).getDouble());
					siteBalance.setOldestBillDate(format.parse(detail.getOldestBillDate()));
					siteBalance.setProcessDate(format.parse(entry.getKey().getProcessDate()));
					siteBalance.setSiteBalRec(new SiteBalRec(siteBalRecId));
					siteBalance.setPatientAcctBalMsg(null);
					pdMap.put(icn, siteBalance);
				} else {
					SiteBalance siteBalance = pdMap.get(icn);
					siteBalance.setBalance(siteBalance.getBalance() + AITCDollar.fromString(detail.getPatientSubtotal()).getDouble());
					// Use the oldest date between the existing bills
					Date oldestBillDate = format.parse(detail.getOldestBillDate());
					if (oldestBillDate.compareTo(siteBalance.getOldestBillDate()) < 0) {
						siteBalance.setOldestBillDate(oldestBillDate);
					}
				}
			} else {
				
				// This is an unrecoverable error so throw exception to stop processing.
				throw new RuntimeException("Problem registering Site/DFN for detail record: " + entry.getKey().getStationNum() + "/" + detail.getPatientDfn());
			}
		}
		return new ArrayList<SiteBalance>(pdMap.values());
	}
	
	/**
	 * 
	 */
	private boolean updateLoadBillDetailAccountRegistration(LoadBillHeader h, LoadBillDetail p) {
		
		boolean successful = false;
		
		final long dfn = Long.parseLong(p.getPatientDfn());

		// dfn is the long version, it has to be trimmed to the short version.
		// because short version dfn is the one saved in the vistaAccount table.
		long shortDfn = CpssUtils.getDfnNumber(dfn);
		VistaAccount vista = vistaAccountService.getAccountRecord(shortDfn, h.getStationNum());

		if (vista != null) {
			p.setIcnNumber(vista.getIcn());
			p.setCbssAcntId(vista.getCbssAcntId());
			successful = true;
		} else {
			vista = new VistaAccount(dfn, h.getStationNum(), p.getIcnNumber(),
					vistaAccountService.getAccountNumberForICN(p.getIcnNumber()));
			p.setCbssAcntId(vista.getCbssAcntId());
			successful = vistaAccountService.registerAccount(vista);
		}
		
		return successful;
	}
	
	/**
	 * Before processing capture the job execution and the received id for data
	 * processing.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		processorLogger.debug("Before Step Execution");

		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is
		// processed.
		jobExecution = stepExecution.getJobExecution();

		// Save the batchRunId at the beginning of the step.
		// It is obtained by the batch prior to the job and passed as a job
		// parameter when the job starts.
		siteBalRecId = jobExecution.getJobParameters().getLong(RECEIVED_ID_KEY);
	}

	@Override
	public void afterProcess(Entry<LoadBillHeader, List<LoadBillDetail>> arg0, List<SiteBalance> arg1) {
		processorLogger.debug("After Process");
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			processorLogger.error("System failure detected.");
			forceStop = true;
		}
		
	}

	@Override
	public void beforeProcess(Entry<LoadBillHeader, List<LoadBillDetail>> arg0) {
	}

	@Override
	public void onProcessError(Entry<LoadBillHeader, List<LoadBillDetail>> arg0, Exception arg1) {
		processorLogger.error("Exception during Load Bill Data processing: ", arg1);
	}
}
