package gov.va.cpss.job.fps;

import static gov.va.cpss.job.CbssJobProcessingConstants.EMPTY_FILE_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.INCOMPLETE_FILE_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;
import static gov.va.cpss.job.fps.FpsProcessingConstants.RECEIVED_ID_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.TOTAL_SITE_COUNT_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.TOTAL_STATEMENT_COUNT_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.AfterStep;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSRecord;
import gov.va.cpss.model.fps.PSSite;

/**
 * Custom ItemProcessor used by the Process FPS Data batch job to process
 * records parsed from the input flat file prior to writing the records to the
 * database.
 * 
 * @author DNS   */
public class FpsItemProcessor implements ItemProcessor<PSRecord, PSRecord> {

	private static final Logger processorLogger = Logger.getLogger(FpsItemProcessor.class.getCanonicalName());

	private JobExecution jobExecution;

	private Long receivedId;

	// PS contains a sequence number.
	// These are used to verify the sequence and number of PS is correct during
	// processing.
	private int nextExpectedSeqNum = 1;
	private int savedSeqNum;
	private int savedTotSeqNum;

	// PS contains number of PH.
	// These are used to verify the number of PH.
	private int nextExpectedStatementCount = 1;
	private int savedStatementCount;
	private int savedTotStatementCount;

	private String savedFacilityNum;

	// PH contains number of PD.
	// These are used to verify the number of PD.
	private int nextExpectedDetailsCount = 1;
	private int savedDetailsCount;
	private int savedTotDetailsCount;

	@Override
	public PSRecord process(PSRecord pojo) throws Exception {

		// If job has already failed then just return null.
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			return null;
		}

		// The data is processed as:
		// PS, then subsequent 1-* PH.
		// PH, then subsequent 1-* PD.

		if (pojo instanceof PSSite) {

			// Verify the previous PS was properly processed.
			if (!lastPatientCompleted()) {
				return stopJob(PROCESSING_FAILURE_STATUS);
			}

			// If a PSSite row then reset the child keys to null to prevent
			// erroneous foreign key references.
			resetSite();
			savedFacilityNum = (((PSSite) pojo).getFacilityNum() == null) ? "" : ((PSSite) pojo).getFacilityNum().trim();

			// At this point receivedId FK should not be null.
			if (receivedId != null) {

				if (!processSiteRecord((PSSite) pojo)) {
					return stopJob(PROCESSING_FAILURE_STATUS);
				}

			} else {

				// Unrecoverable error so stop the job.
				return stopJob(PROCESSING_FAILURE_STATUS, "Attempted to process PS row with a null Received FK");
			}

		} else if (pojo instanceof PSPatient) {

			// Verify the previous PH was properly processed.
			if (!lastDetailsCompleted()) {
				return stopJob(PROCESSING_FAILURE_STATUS);
			}

			// If a PSPatient row then reset the child keys to null to prevent
			// erroneous foreign key references.
			resetPatient();

			if (!processPatientRecord((PSPatient) pojo)) {
				return stopJob(PROCESSING_FAILURE_STATUS);
			}

		} else if (pojo instanceof PSDetails) {

			if (!processDetailsRecord((PSDetails) pojo)) {
				return stopJob(PROCESSING_FAILURE_STATUS);
			}

		} else {

			// Unrecoverable error so stop the job.
			return stopJob(PROCESSING_FAILURE_STATUS, "Attempted to process unknown row");
		}

		return pojo;
	}

	/**
	 * Before processing capture the job execution and the received id for data
	 * processing.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		processorLogger.info("Before Step Execution");

		// Save the job execution at the beginning of the step.
		// The execution context will be used to set foreign key values as rows
		// are processed.
		jobExecution = stepExecution.getJobExecution();

		// Save the receivedId at the beginning of the step.
		// It is obtained by the batch prior to the job and passed as a job
		// parameter when the job starts.
		receivedId = jobExecution.getJobParameters().getLong(RECEIVED_ID_KEY);

		// Initialize the processing counts.
		jobExecution.getExecutionContext().putLong(TOTAL_SITE_COUNT_KEY, 0);
		jobExecution.getExecutionContext().putLong(TOTAL_STATEMENT_COUNT_KEY, 0);

		// Initialize the processing verification data.
		// PS processing
		nextExpectedSeqNum = 1;
		savedSeqNum = 0;
		savedTotSeqNum = 0;
		// PH processing
		nextExpectedStatementCount = 1;
		savedStatementCount = 0;
		savedTotStatementCount = 0;
		// PD processing
		nextExpectedDetailsCount = 1;
		savedDetailsCount = 0;
		savedTotDetailsCount = 0;
	}

	/**
	 * After processing check for unexpected conditions that suggest an error.
	 * 
	 * @param stepExecution
	 */
	@AfterStep
	public void afterStep(StepExecution stepExecution) {

		// If no other error detected then check for other possible error
		// conditions.
		if (!jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {

			// If read count is zero then report a job failure.
			if (stepExecution.getReadCount() == 0) {
				setFailureStatusAndMessage(EMPTY_FILE_ERROR_STATUS, "Input file is empty");
			} else if (!lastDetailsCompleted() || !lastPatientCompleted() || !lastSiteCompleted()) {
				setFailureStatus(INCOMPLETE_FILE_ERROR_STATUS);
			}

		}

		processorLogger.info("After Step Execution");
	}

	/**
	 * Verify the last site processing was completed.
	 * 
	 * @return Boolean indicating if processing completed.
	 */
	private boolean lastSiteCompleted() {

		boolean successful = true;

		if (savedTotSeqNum != savedSeqNum) {
			successful = false;
			setFailureMessage(
					"PS count (" + savedSeqNum + ") of (" + savedTotSeqNum + ") indicates incomplete sequence");
		} else {
			successful = lastPatientCompleted();
		}

		return successful;
	}

	/**
	 * Verify the last patient processing was completed.
	 * 
	 * @return Boolean indicating if processing completed.
	 */
	private boolean lastPatientCompleted() {

		boolean successful = true;

		if (savedStatementCount != savedTotStatementCount) {
			successful = false;
			setFailureMessage("PH count (" + savedStatementCount + ") of (" + savedTotStatementCount
					+ ") indicates incomplete statement");
		} else {
			successful = lastDetailsCompleted();
		}

		return successful;
	}

	/**
	 * Verify the last details processing was completed.
	 * 
	 * @return Boolean indicating if processing completed.
	 */
	private boolean lastDetailsCompleted() {

		boolean successful = true;

		if (savedDetailsCount != savedTotDetailsCount) {
			successful = false;
			setFailureMessage("PD count (" + savedDetailsCount + ") of (" + savedTotDetailsCount
					+ ") indicates incomplete details");
		}

		return successful;
	}

	private boolean validPatientAccount(String patientAccount) {
		final String trimmedPatientAccount = (patientAccount == null) ? "" : patientAccount.trim();
		if (trimmedPatientAccount.length() < 11) {
			// Minimum length (1 (fac-num) + 9 (ssn) + 1 (last name) == 11)
			setFailureMessage("Invalid PatientAccount: Does not meet minimum length requirement.");
			return false;
		}

		if ((savedFacilityNum == null) || (savedFacilityNum.length() < 1)
				|| !trimmedPatientAccount.startsWith(savedFacilityNum)) {
			setFailureMessage("Invalid PatientAccount: Facility number does not match PS record.");
			return false;
		}

		return true;
	}

	/**
	 * Process site record.
	 */
	private boolean processSiteRecord(PSSite site) {
		boolean successful = true;

		site.setPsReceivedId(receivedId);

		// Verify the sequence number is the expected value.
		if (site.getSeqNum() == nextExpectedSeqNum) {

			// If the sequence number is 1 then save the expected total sequence
			// number or verify the total sequence number.
			if (site.getSeqNum() == 1) {
				savedTotSeqNum = site.getTotSeqNum();

			} else if (site.getTotSeqNum() != savedTotSeqNum) {
				successful = false;
				setFailureMessage("Attempted to process PS with invalid total seq num (" + site.getTotSeqNum()
						+ ") but expected (" + savedTotSeqNum + ")");
			}

			// Verify the sequence number is valid.
			if (site.getSeqNum() <= site.getTotSeqNum()) {

				// Set the next expected sequence number.
				if (site.getSeqNum() < site.getTotSeqNum()) {
					++nextExpectedSeqNum;
				} else {
					nextExpectedSeqNum = 1;
				}

				// Finally, save data.
				// The sequence number.
				savedSeqNum = site.getSeqNum(); // This PS sequence
				savedTotStatementCount = site.getTotStatement(); // The number
																	// of
																	// expected
																	// PH
				if (savedTotStatementCount == 0) {
					successful = false;
					setFailureMessage("Attempted to process PS with invalid total statement count (0)");
				}
				
			} else {
				successful = false;
				setFailureMessage("Attempted to process PS with invalid seq num (" + site.getSeqNum() + ") of ("
						+ site.getTotSeqNum() + ")");
			}

		} else {
			successful = false;
			setFailureMessage("Attempted to process PS with invalid seq num (" + site.getSeqNum() + ") but expected ("
					+ nextExpectedSeqNum + ")");
		}

		return successful;
	}

	/**
	 * Process patient record.
	 */
	private boolean processPatientRecord(PSPatient patient) {

		boolean successful = true;

		// Increment the count of patient.
		++savedStatementCount;

		// Verify the statement count is the expected value.
		if (savedStatementCount == nextExpectedStatementCount) {

			// Verify the statement count is valid.
			if (savedStatementCount <= savedTotStatementCount) {

				// Set the next expected statement count.
				if (savedStatementCount < savedTotStatementCount) {
					++nextExpectedStatementCount;
				} else {
					nextExpectedStatementCount = 1;
				}

				// Finally, save data.
				savedTotDetailsCount = patient.getNumOfLines(); // The number of
																// expected PD

				if (savedTotDetailsCount == 0) {
					successful = false;
					setFailureMessage("Attempted to process PH with invalid total details count (0)");
				} else {
					successful = validPatientAccount(patient.getPatientAccount());
				}
			} else {
				successful = false;
				setFailureMessage("Attempted to process PH with invalid count (" + savedStatementCount + ") of ("
						+ savedTotStatementCount + ")");
			}

		} else {
			successful = false;
			setFailureMessage("Attempted to process PH with unexpected count (" + savedStatementCount
					+ ") but expected (" + nextExpectedStatementCount + ")");
		}

		return successful;
	}

	/**
	 * Process details record.
	 */
	private boolean processDetailsRecord(PSDetails details) {

		boolean successful = true;

		// Increment the count of details.
		++savedDetailsCount;

		// Verify the details count is the expected value.
		if (savedDetailsCount == nextExpectedDetailsCount) {

			// Verify the details count is valid.
			if (savedDetailsCount <= savedTotDetailsCount) {

				// Set the next expected details count.
				if (savedDetailsCount < savedTotDetailsCount) {
					++nextExpectedDetailsCount;
				} else {
					nextExpectedDetailsCount = 1;
				}

			} else {
				successful = false;
				setFailureMessage("Attempted to process PD with invalid count (" + savedDetailsCount + ") of ("
						+ savedTotDetailsCount + ")");
			}

		} else {
			successful = false;
			setFailureMessage("Attempted to process PD with unexpected count (" + savedDetailsCount + ") but expected ("
					+ nextExpectedDetailsCount + ")");
		}

		return successful;
	}

	/**
	 * Reset the site values from the job execution to prepare to process next
	 * site.
	 */
	private void resetSite() {

		// Reset PH counts.
		nextExpectedStatementCount = 1;
		savedStatementCount = 0;
		savedTotStatementCount = 0;

		resetPatient();
	}

	/**
	 * Reset the patient values from the job execution to prepare to process
	 * next patient.
	 */
	private void resetPatient() {

		// Reset PD counts.
		nextExpectedDetailsCount = 1;
		savedDetailsCount = 0;
		savedTotDetailsCount = 0;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private PSRecord stopJob(final String status) {
		// Log message.
		processorLogger.error("Processor execution encountered unrecoverable error and forced stop");
		// Set failure and message.
		setFailureStatus(status);
		// Stop job.
		jobExecution.stop();

		return null;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private PSRecord stopJob(final String status, final String message) {

		// Set failure.
		stopJob(status);

		// Set failure message.
		setFailureMessage(message);

		return null;
	}

	/**
	 * Set the failure and message in the job execution context.
	 */
	private void setFailureStatusAndMessage(final String status, final String message) {
		// Set job failure.
		setFailureStatus(status);
		// Set job failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		processorLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		processorLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
