package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_DETECTED_PATIENTS_ERROR_KEY;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;

import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSSite;
import gov.va.cpss.model.icn.VistaAccount;
import gov.va.cpss.service.FpsService;
import gov.va.cpss.service.VistaAccountService;
import gov.va.cpss.util.CpssUtils;

/**
 * Query for all PSPatient records associated with PSSite records associated
 * with PSReceived records in the NEW state.
 * 
 * @author DNS  
 */
public class CbsQueryNewPSPatientTasklet implements Tasklet {

	private final Logger taskletLogger;

	private FpsService fpsService;

	private VistaAccountService vistaAccountService;

	private CbsRuntimeState cbsRuntimeState;

	public CbsQueryNewPSPatientTasklet() {
		taskletLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public FpsService getFpsService() {
		return fpsService;
	}

	public void setFpsService(FpsService fpsService) {
		this.fpsService = fpsService;
	}

	public VistaAccountService getVistaAccountService() {
		return vistaAccountService;
	}

	public void setVistaAccountService(VistaAccountService vistaAccountService) {
		this.vistaAccountService = vistaAccountService;
	}

	public CbsRuntimeState getCbsRuntimeState() {
		return cbsRuntimeState;
	}

	public void setCbsRuntimeState(CbsRuntimeState cbsRuntimeState) {
		this.cbsRuntimeState = cbsRuntimeState;
	}

	@Override
	public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception {

		taskletLogger.info("Begin execute");
		
		// Read in all the PSSite records for NEW PSReceived into a Map by id.
		// Added this optimization to read the sites in one query rather than one at a time during the following loop.
		final Map<Long, PSSite> newPsSiteMap = loadNewPsSites();

		// Loop through all of the NEW PSSite.
		while (!cbsRuntimeState.isDataError() && (cbsRuntimeState.pollCurrentPSSiteID() != null)) {

			
			final long currentPSSiteID = cbsRuntimeState.getCurrentPSSiteID();
			final PSSite currentPSSite = newPsSiteMap.get(currentPSSiteID);
			
			// Get all patient for this PSSite.
			List<PSPatient> psPatientL = null;
			if (currentPSSite == null) {
				psPatientL = fpsService.getPSPatientListForPSSiteID(currentPSSiteID);
			} else {
				psPatientL = fpsService.getPSPatientListForPSSite(currentPSSite);
			}

			// Set error flag if empty. This flag will be checked by the
			// subsequent reader.
			if ((psPatientL == null) || psPatientL.isEmpty()) {

				taskletLogger
						.error("Error obtaining patient list for site ID(" + cbsRuntimeState.getCurrentPSSiteID() + ")");
				cbsRuntimeState.setDataError(true);
			} else {
				
				// Save list of patient errors.
				List<String> patientErrorL = new ArrayList<>();

				// Loop through each PSPatient in the list to create a
				// Map<CBSAccount, List<PSPatient>> for this site.
				// 1. Get 'dfn/site' pair from the PSPatient object.
				// 2. Check if in VistaAccount table:
				// a. if exists:
				// i. Update the PSPatient object (in memory, leave db record
				// alone) with ICN from VistaAccount table.
				// b. if does not exist:
				// i. Use ICN from PSPatient object to get CBS Account Number
				// from CBSAccount table.
				// ii. Create VistaAccount record using CBS Account Number and
				// ICN from PSPatient object.
				// 3. Save this Site Map<CBSAccount, List<PSPatient>> into the
				// consolidated CbsRuntimeState Map<CBSAccount,
				// List<PSPatient>>.
				Map<Long, List<PSPatient>> psPatientM = new HashMap<>();

				for (PSPatient p : psPatientL) {
					
					try {
						boolean successful = false;
						final long dfn = p.getDfnNumber();
						final String site = p.getPsSite().getFacilityNum();
	
						// dfn is the long version, it has to be trimmed to the short version.
						// because short version dfn is the one saved in the vistaAccount table.
						long shortDfn = CpssUtils.getDfnNumber(dfn);
						VistaAccount vista = vistaAccountService.getAccountRecord(shortDfn, site);
	
						if (vista != null) {
							p.setIcnNumber(vista.getIcn());
							successful = true;
						} else {
							vista = new VistaAccount(dfn, site, p.getIcnNumber(),
									vistaAccountService.getAccountNumberForICN(p.getIcnNumber()));
							successful = vistaAccountService.registerAccount(vista);
						}
	
						if (successful) {
							if (psPatientM.containsKey(vista.getCbssAcntId())) {
								psPatientM.get(vista.getCbssAcntId()).add(p);
							} else {
								List<PSPatient> pL = new ArrayList<>();
								pL.add(p);
								psPatientM.put(vista.getCbssAcntId(), pL);
							}
						} else {
							appendToErrorList(p, patientErrorL);
						}
						
					} catch (RuntimeException e) {
						
						appendToErrorList(p, patientErrorL);
					} catch (Exception e) {
						
						taskletLogger.error(e.getMessage());
						appendToErrorList(p, patientErrorL);
					}
				}

				// Save all of the PSPatients for this site into the patient
				// map.
				cbsRuntimeState.savePSPatientMap(psPatientM);
				
				// Save the detected patients with error to the context for later reporting.
				saveDetectedPatientDataErrors(patientErrorL, chunkContext.getStepContext().getStepExecution().getJobExecution());
			}
		}

		taskletLogger.info("End execute");

		return RepeatStatus.FINISHED;
	}
	
	/**
	 * Returns a Map of all PSSite records for NEW PSReceived records.
	 * 
	 * @return	Map<Long, PSSite> keyed by PSSite.id
	 */
	private Map<Long, PSSite> loadNewPsSites() {
		
		Map<Long, PSSite> newPsSiteMap = new HashMap<>();
		if (!cbsRuntimeState.isDataError()) {
			List<PSSite> psSiteL = fpsService.getPSSitesForNewPSReceived();
			
			if ((psSiteL == null) || psSiteL.isEmpty()) {
				taskletLogger.error("Error obtaining site list for new received statement files");
				cbsRuntimeState.setDataError(true);
			} else {
				for (PSSite psSite : psSiteL) {
					newPsSiteMap.put(psSite.getId(), psSite);
				}
			}
		}

		return newPsSiteMap;
	}
	
	/**
	 * Append the error to the list.
	 * 
	 * @param psPatient
	 *            The patient record.
	 * @param patientErrorL
	 *            The error list.
	 */
	private void appendToErrorList(final PSPatient psPatient, List<String> patientErrorL) {
		StringBuffer strBuff = new StringBuffer();
		strBuff.append(psPatient.getIcnNumber());
		strBuff.append(" ");
		strBuff.append(psPatient.getPsSite().getFacilityNum());
		strBuff.append(" ");
		strBuff.append(psPatient.getPatientLastName());
		patientErrorL.add(strBuff.toString());
	}
	
	/**
	 * Save detected patient data errors to job execution context.
	 * @param patientErrorL This list of error patients.
	 */
	private void saveDetectedPatientDataErrors(final List<String> patientErrorL, JobExecution jobExecution) {

		// If patient data had errors capture in the runtime data.
		if (!patientErrorL.isEmpty()) {

			taskletLogger.error("There were some unprocessed patient records");

			String existingPatientErrors = null;
			StringBuffer strBuff = new StringBuffer();

			// Add (append if necessary) to the job execution.
			if (jobExecution.getExecutionContext().containsKey(CBS_DETECTED_PATIENTS_ERROR_KEY)) {
				existingPatientErrors = jobExecution.getExecutionContext().getString(CBS_DETECTED_PATIENTS_ERROR_KEY);
			}

			if (existingPatientErrors == null) {
				strBuff.append("Unprocessed Records:");
			} else {
				strBuff.append(existingPatientErrors);
			}

			// Append to the error message.
			for (String p : patientErrorL) {
				strBuff.append("\n");
				strBuff.append(p);
			}

			// Save the message to the job execution context.
			jobExecution.getExecutionContext().putString(CBS_DETECTED_PATIENTS_ERROR_KEY, strBuff.toString());
		}
	}

}
