package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.service.CbsService;

/**
 * Recover previous partial processing by reverting any REPLACED statements to
 * NEW state and removing the processed consolidated statements in INITIAL
 * state.
 * 
 * @author DNS 
 */
public class CbsProcessStmtSystemFailureRecoveryTasklet extends CbsJobBaseCompletionTasklet {

	private CbsService cbsService;

	public CbsService getCbsService() {
		return cbsService;
	}

	public void setCbsService(CbsService cbsService) {
		this.cbsService = cbsService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("Stmt System Failure Recovery Tasklet Running!");

		// Revert REPLACED statements to NEW that are associated with INITIAL
		// Statements
		final int replacedCount = cbsService.revertReplacedStatementsAssociatedWithInitialStatements();

		if (successful && (replacedCount >= 0)) {
			taskletLogger.info("Count of statements reverted to NEW (" + replacedCount + ")");
		} else {
			// unrecoverable error.
			final String error = "Rollback Triggered - Error occurred reverting REPLACED status rows for CBSStmt: "
					+ replacedCount;
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);

			// Flag to do rollback.
			successful = false;
		}

		// Delete all INITIAL statements.
		final int deletedCount = cbsService.removeInitialStatements();

		if (successful && (deletedCount >= 0)) {
			taskletLogger.info("Count of initial statements deleted (" + deletedCount + ")");
		} else {
			// unrecoverable error.
			final String error = "Rollback Triggered - Error occurred deleting INITIAL status rows for CBSStmt: "
					+ deletedCount;
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);

			// Flag to do rollback.
			successful = false;
		}

		// Set exit status of this tasklet to FAILED if not successful.
		if (successful) {

			taskletLogger.info("Stmt System Failure Recovery Tasklet was Successful");

		} else {

			// Set the completion statistics to zero since the job failed.
			updateStatistics(chunkContext.getStepContext().getStepExecution().getJobExecution(), 0);

			taskletLogger.error("Stmt System Failure Recovery Tasklet Detected an Unrecoverable Failure");
			chunkContext.getStepContext().getStepExecution().setExitStatus(ExitStatus.FAILED);
		}

		return successful;
	}

}
