package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_ERROR_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_AVAILABLE_STATISTICS_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_AVAILABLE_STATISTICS_LABEL;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobExecutionListener;

import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.service.CbsService;

/**
 * Custom JobExecutionListener used to process custom exit status to handle
 * errors during Generate CBS batch job execution.
 * 
 * @author DNS 
 */
public class CbsJobExecutionListener implements JobExecutionListener {

	private final Logger listenerLogger;

	private CbsService cbsService;

	public CbsJobExecutionListener() {
		listenerLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public CbsService getCbsService() {
		return cbsService;
	}

	public void setCbsService(CbsService cbsService) {
		this.cbsService = cbsService;
	}

	@Override
	public void beforeJob(JobExecution jobExecution) {
		listenerLogger.info("Beginning Job Execution");

		// Initialize written running totals.
		jobExecution.getExecutionContext().putLong(CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY, 0);
		jobExecution.getExecutionContext().putLong(CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY, 0);
	}

	@Override
	public void afterJob(JobExecution jobExecution) {

		listenerLogger.info("After Job");

		// System failure takes precedence over data error.
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {

			final String exitStatus = jobExecution.getExecutionContext().getString(JOB_FAILURE_KEY);

			if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY)) {

				final String exitMessage = jobExecution.getExecutionContext().getString(JOB_FAILURE_MESSAGE_KEY);

				listenerLogger.error("JobFailure detected, setting exit status (" + exitStatus + "): " + exitMessage);

				jobExecution.setExitStatus(new ExitStatus(exitStatus, exitMessage));
			} else {

				listenerLogger.error("JobFailure detected, setting exit status (" + exitStatus + ")");

				jobExecution.setExitStatus(new ExitStatus(exitStatus));
			}

		} else if (jobExecution.getExecutionContext().containsKey(JOB_ERROR_KEY)) {

			final String exitStatus = jobExecution.getExecutionContext().getString(JOB_ERROR_KEY);

			listenerLogger.error("JobError detected, setting exit status (" + exitStatus + ")");

			jobExecution.setExitStatus(new ExitStatus(exitStatus));
		}

		// Set the completion statistics.
		setCompletionStatistics(jobExecution);

		listenerLogger.info("Ending Job Execution");
	}

	/**
	 * Set the completion statistics.
	 * 
	 * @param jobExecution
	 *            The job execution to append statisitics to.
	 */
	private void setCompletionStatistics(JobExecution jobExecution) {

		final Long availableCount = cbsService.getCountOfStatementsInState(ProcessStatus.Status.NEW);

		if (availableCount != null) {
			jobExecution.getExecutionContext().putString(CBS_AVAILABLE_STATISTICS_KEY,
					CBS_AVAILABLE_STATISTICS_LABEL + availableCount);
		} else {

			jobExecution.getExecutionContext().putString(CBS_AVAILABLE_STATISTICS_KEY,
					CBS_AVAILABLE_STATISTICS_LABEL + "Statistics Unavailable");
		}
	}

}
