package gov.va.cpss.job.caps;

import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;

import static gov.va.cpss.model.ps.Constants.BATCH_PROCESS_ID_KEY;
import static gov.va.cpss.model.ps.Constants.RECEIVED_IDS_KEY;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.springframework.batch.core.StepExecution;

import gov.va.cpss.job.CBSSPagingItemProcessor;

import gov.va.cpss.model.apps.APSDetails;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSSite;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSStmt;

import gov.va.cpss.service.GenerateAPPSService;
import gov.va.cpss.service.Result;

/**
 * 
 * Custom ItemProcessor used by the Generate APPS batch job to process
 * records parsed from the database prior to writing the records to the
 * database.
 * 
 * Copyright HPE / VA
 * January 25, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class CAPSPagingItemProcessor extends CBSSPagingItemProcessor<APSPatient, List<Entry<APSStmt, List<APSSiteStmt>>>>
{
    private Long batchProcessId;
    private StepExecution stepExecution;

    // Injected property
    private GenerateAPPSService generateAPPSService;

    public GenerateAPPSService getGenerateAPPSService()
    {
        return this.generateAPPSService;
    }

    public void setGenerateAPPSService(GenerateAPPSService inGenerateAPPSService)
    {
        this.generateAPPSService = inGenerateAPPSService;
    }


    @Override
    public void beforeStep(StepExecution inStepExecution)
    {
        super.beforeStep(inStepExecution);

        // Save the batchProcessId at the beginning of the step.
        // It is obtained by the batch prior to the job and passed as a job
        // parameter when the job starts.
        this.batchProcessId = this.jobExecution.getJobParameters().getLong(BATCH_PROCESS_ID_KEY);

        this.stepExecution = inStepExecution;

        // Using Execution Context to store APSReceived ID's for later use.
        this.stepExecution.getExecutionContext().put(RECEIVED_IDS_KEY, new ArrayList<Long>());
    }
    
    @Override
    protected List<Entry<APSStmt, List<APSSiteStmt>>> doProcess(List<APSPatient> inPatients) throws Exception
    {
        this.logger.debug("Start Processing data.");

        // Get the starting time
        final long startTime = System.nanoTime();

        // Read APSSite data associated with the NEW APSPatient
        Result<List<APSSite>> resultSite = this.generateAPPSService.getSitesForPatients(inPatients);

        List<APSSite> sites = resultSite.get();

        if (!resultSite.isSuccessful() || sites == null || sites.isEmpty())
        {
            stopJob(PROCESSING_FAILURE_STATUS,
                    "Failed to get associated sites for patients: " + resultSite.getMessage());

            return null;
        }

        // Read APSDetails data associated with the NEW APSPatient
        Result<List<APSDetails>> resultDetails = this.generateAPPSService.getDetailsForPatients(inPatients);

        List<APSDetails> detailsList = resultDetails.get();

        if (!resultDetails.isSuccessful() || detailsList == null || detailsList.isEmpty())
        {
            stopJob(PROCESSING_FAILURE_STATUS,
                    "Failed to get associated details for patients: " + resultDetails.getMessage());

            return null;
        }
        
        // Put associated details to patient.
        for (APSPatient patient : inPatients)
        {
            List<APSDetails> patientDetailsList = detailsList.stream().filter(d -> d.getApsPatientId() == patient.getId()).collect(Collectors.toList());
            patient.setDetailsList(patientDetailsList);
        }

        // Put associated patients to site.
        List<Entry<APSSite, List<APSPatient>>> sitesList = new ArrayList<>();

        // Extract APSReceived (id) from APSSite for later use.
        @SuppressWarnings("unchecked")
        List<Long> apsReceivedIds = (List<Long>) this.stepExecution.getExecutionContext().get(RECEIVED_IDS_KEY);

        for (APSSite site : sites)
        {
            // Put associated patients to site.
            List<APSPatient> patients = inPatients.stream().filter(p -> p.getApsSite().getId() == site.getId()).collect(Collectors.toList());
            Entry<APSSite, List<APSPatient>> siteEntry = new AbstractMap.SimpleEntry<>(site, patients);
            sitesList.add(siteEntry);

            // Extract APSReceived (id) from APSSite for later use.
            Long apsReceivedId = Long.valueOf(site.getApsReceivedId());

            if (!apsReceivedIds.contains(apsReceivedId))
            {
                apsReceivedIds.add(apsReceivedId);
            }
        }


        // Create Site Statement
        Result<List<APSSiteStmt>> resultSiteStmt = this.generateAPPSService.createSiteStatements(sitesList);

        List<APSSiteStmt> siteStmts = resultSiteStmt.get();

        if (!resultSiteStmt.isSuccessful() || siteStmts == null || siteStmts.isEmpty())
        {
            stopJob(PROCESSING_FAILURE_STATUS,
                    "Failed to create site statements: " + resultSiteStmt.getMessage());

            return null;
        }


        List<APSStmt> initialStmts;
        List<APSStmt> consolidatedStmts;

        // Create initial Statements
        Result<List<APSStmt>> resultStmt = this.generateAPPSService.createInitialStatements(this.batchProcessId.longValue(), siteStmts);
        initialStmts = resultStmt.get(); 

        if (!resultStmt.isSuccessful() || initialStmts == null || initialStmts.isEmpty())
        {
            stopJob(PROCESSING_FAILURE_STATUS,
                    "Failed to create initial statements: " + resultStmt.getMessage());

            return null;
        }


        // Consolidate Statements
        resultStmt = this.generateAPPSService.consolidateStatements(initialStmts);
        consolidatedStmts = resultStmt.get();

        if (!resultStmt.isSuccessful() || consolidatedStmts == null || consolidatedStmts.isEmpty())
        {
            stopJob(PROCESSING_FAILURE_STATUS,
                    "Failed to consolidate statements: " + resultStmt.getMessage());

            return null;
        }

        List<Entry<APSStmt, List<APSSiteStmt>>> stmtEntriesList = new ArrayList<>();

        for (APSStmt stmt : consolidatedStmts)
        {
            Entry<APSStmt, List<APSSiteStmt>> stmtEntry = new AbstractMap.SimpleEntry<>(stmt, stmt.getSiteStmts());
            stmtEntriesList.add(stmtEntry);
        }

        final long elapsedTime = System.nanoTime() - startTime;

        this.logger.debug("End Processing data. " + this.generateAPPSService.printElapsedTime("method", elapsedTime));

        return stmtEntriesList;
    }
}
