package gov.va.cpss.job.appsprintack;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.TOTAL_APPS_RECEIVED_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.TOTAL_APPS_UPDATED_KEY;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemWriter;

import gov.va.cpss.job.UpdateAPPSPrintAckJob;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;
import gov.va.cpss.model.appsprintack.APPSPAFileRecord;
import gov.va.cpss.service.UpdateAPPSPrintAckService;

/**
 * Custom ItemWriter used by the UpdateAppsPrintAck job to process the print
 * acknowledgements and update the consolidated statement statuses accordingly.
 */
public class UpdateAppsPrintAckItemWriter implements ItemWriter<APPSPAFileRecord>, ItemWriteListener<APPSPAFileRecord> {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt writes.
	 */
	private boolean forceStop = false;

	private JobExecution jobExecution;

	private UpdateAPPSPrintAckService updateAppsPrintAckService;

	public UpdateAPPSPrintAckService getUpdateAPPSPrintAckService() {
		return updateAppsPrintAckService;
	}

	public void setUpdateAPPSPrintAckService(UpdateAPPSPrintAckService updatePrintAckService) {
		this.updateAppsPrintAckService = updatePrintAckService;
	}

	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		logger.info("Before Step Execution");

		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public void write(List<? extends APPSPAFileRecord> recordL) throws Exception {
		// Do not attempt to write if forced stop.
		if(forceStop) {
			return;
		}
		
		try {
			List<Long> batchRunProcessIdList = UpdateAPPSPrintAckJob.getCurrentFileAppsBatchRunProcessIdList(jobExecution);
			List<Long> currentBatchRunProcessIdList = new ArrayList<Long>();
			
			for(APPSPAFileRecord record : recordL) {
				logger.debug("Write: " + record);
				
				final BatchRunProcess batchRunProcess = record.getBatchRunProcess();
				
				currentBatchRunProcessIdList.add(updateAppsPrintAckService.initializeBatchRunProcess(batchRunProcess));
				
				logger.debug("StatementYear:" + record.getStatementYear());
				logger.debug("PrintDate:" + record.getPrintDate());
				
				final List<APPSADFileRecord> missingPatientAccts = new ArrayList<APPSADFileRecord>();

				final int numStmtsUpdated = updateAppsPrintAckService.processPrintAckDetails(batchRunProcess, record.getPrintAckDetailL(),
						record.getStatementYear(), record.getPrintDate(), missingPatientAccts);
				logger.debug("numStmtsUpdated: " + numStmtsUpdated);
				
				if(stopJobIfMissingStatements(missingPatientAccts, record)) {
					break;
				}
				
				updateAppsPrintAckService.setPrintAckSuccess(batchRunProcess);
				
				accrueProcessingCount(TOTAL_APPS_RECEIVED_KEY, numStmtsUpdated);
				accrueProcessingCount(TOTAL_APPS_UPDATED_KEY, numStmtsUpdated);
				
				batchRunProcessIdList.addAll(currentBatchRunProcessIdList);
				UpdateAPPSPrintAckJob.putCurrentFileAppsBatchRunProcessIdList(jobExecution, batchRunProcessIdList);
			}
		} catch (Exception e) {
			
			StringBuilder error = new StringBuilder();
			error.append("Error during write because of ");
			error.append(e.getClass().getSimpleName());
			
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}
	}

	/**
	 * Accrue the given processing count in the job execution context.
	 * 
	 * @param key
	 *            Name of the processing count.
	 * @param numStmts
	 *            Amount to add to the processing count.
	 */
	private void accrueProcessingCount(final String key, final int numStmts) {
		long totalStmts = jobExecution.getExecutionContext().getLong(key);
		totalStmts += numStmts;
		jobExecution.getExecutionContext().putLong(key, totalStmts);
	}

	/**
	 * Check the given List of APPSADFileRecords representing AD records where no
	 * corresponding patient statement exists for the statement date. If the
	 * List is not empty, then stop the job.
	 * 
	 * @param missingPatientAccts
	 *            List of APPSADFileRecords for patients where no corresponding
	 *            patient statement exists.
	 * @param paFileRecord
	 *            The APPSPAFileRecord with the missing patient accounts
	 * @return true if there are missing patient statements and the job should
	 *         stop.
	 */
	private boolean stopJobIfMissingStatements(List<APPSADFileRecord> missingPatientAccts, APPSPAFileRecord paFileRecord) {
		final boolean isMissingStatements = (missingPatientAccts.size() > 0);
		
		if (isMissingStatements) {
			UpdateAPPSPrintAckJob.putCurrentFileMissingPatientAcctList(jobExecution, missingPatientAccts);
			
			StringBuilder error = new StringBuilder();
			error.append("Invalid patient accounts encountered. ");
			error.append("No patient statement found for the following patient accounts: ");

			String delim = "";
			for (APPSADFileRecord adFileRecord : missingPatientAccts) {
				error.append(delim);
				error.append(adFileRecord.getPatientAccount());
				delim = ", ";
			}

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}
		
		return isMissingStatements;
	}

	@Override
	public void beforeWrite(List<? extends APPSPAFileRecord> items) {
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			logger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void afterWrite(List<? extends APPSPAFileRecord> items) {
	}

	@Override
	public void onWriteError(Exception e, List<? extends APPSPAFileRecord> items) {

		logger.error("Writer encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to write item because of ");
		if(e.getClass().getSimpleName().equalsIgnoreCase("NullPointerException")){
			error.append("No Data");
		}else{
			error.append(e.getClass().getSimpleName());
		}
		error.append("\nMessage: ");
		error.append(e.getMessage());
		if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());
		}

		// Set failure and message.
		stopJob(WRITE_FAILURE_STATUS, error.toString());
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		logger.error("Writer execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		logger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		logger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
