package gov.va.cpss.job.appsprintack;

import static gov.va.cpss.job.CbssJobProcessingConstants.EMPTY_FILE_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.DATE_RECEIVED_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.TOTAL_APPS_RECEIVED_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.TOTAL_APPS_UPDATED_KEY;

import java.sql.Timestamp;
import java.util.ArrayList;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.AfterStep;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.job.UpdateAPPSPrintAckJob;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;
import gov.va.cpss.model.appsprintack.APPSPAFileRecord;

/**
 * Custom ItemProcessor used by the UpdateAPPSPrintAck job to validate the
 * APPSPAFileRecord instantiated from each line of the input file and build a
 * BatchRunProcess from it.
 * 
 */
public class UpdateAppsPrintAckItemProcessor implements ItemProcessor<APPSPAFileRecord, APPSPAFileRecord> {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());;

	private JobExecution jobExecution;

	private int batchRunId;
	private String fileName;
	private Timestamp dateReceived;

	private long lineNumber = 0;
	private APPSPAFileRecord previousRecord = null;

	/**
	 * Before processing capture the job execution and job parameters for the
	 * job and initialize processing counts.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		logger.info("Before Step Execution");

		// Save the job execution at the beginning of the step.
		jobExecution = stepExecution.getJobExecution();

		// Get the id of the BatchRun record for this job.
		batchRunId = jobExecution.getJobParameters().getLong(BATCH_RUN_ID_KEY).intValue();

		// Get the name of the file being processed.
		fileName = jobExecution.getJobParameters().getString(INPUT_RESOURCE_KEY);

		// Get the date the file being processed was received.
		dateReceived = new Timestamp(jobExecution.getJobParameters().getLong(DATE_RECEIVED_KEY));

		// Initialize the processing counts.
		jobExecution.getExecutionContext().putLong(TOTAL_APPS_RECEIVED_KEY, 0);
		jobExecution.getExecutionContext().putLong(TOTAL_APPS_UPDATED_KEY, 0);
		UpdateAPPSPrintAckJob.putCurrentFileAppsBatchRunProcessIdList(jobExecution, new ArrayList<Long>());
		UpdateAPPSPrintAckJob.putCurrentFileMissingPatientAcctList(jobExecution, new ArrayList<APPSADFileRecord>());

		lineNumber = 0;
		previousRecord = null;
	}

	@Override
	public APPSPAFileRecord process(APPSPAFileRecord record) throws Exception {
		boolean success = true;
		lineNumber++;
		BatchRunProcess batchRunProcess = null;
		APPSPAFileRecord returnRecord = null;

		success = validateAPPSPAFileRecord(record);

		previousRecord = record;

		if (success) {

			batchRunProcess = new BatchRunProcess();

			batchRunProcess.setBatchRunId(batchRunId);
			batchRunProcess.setFileName(fileName);
			batchRunProcess.setProcessDate(dateReceived);

			returnRecord = record;
			record.setBatchRunProcess(batchRunProcess);
		}

		return returnRecord;
	}

	/**
	 * After processing check for unexpected conditions that suggest an error.
	 * 
	 * @param stepExecution
	 */
	@AfterStep
	public void afterStep(StepExecution stepExecution) {

		// If no other error detected then check for other possible error
		// conditions.
		if (!jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {

			// If read count is zero then report a job failure.
			if (stepExecution.getReadCount() == 0) {
				setFailureStatusAndMessage(EMPTY_FILE_ERROR_STATUS, "Input file is empty");
			} /*
				 * TODO: find out what the equivalent to lastSiteCompleted would
				 * be else if (!lastSiteCompleted()) {
				 * setFailureStatus(INCOMPLETE_FILE_ERROR_STATUS); }
				 */

		}

		logger.info("After Step Execution");
	}

	/**
	 * Validate the APPSPAFileRecord instantiated from the current line of the
	 * input file. Job will be stopped if invalid.
	 * 
	 * @param record
	 *            APPSPAFileRecord to validate.
	 * @return True if record is valid.
	 */
	private boolean validateAPPSPAFileRecord(APPSPAFileRecord record) {

		if (record == null) {
			stopJob(PROCESSING_FAILURE_STATUS, "Attempted to process unknown row at line " + lineNumber + ".");
			return false;
		}

		logger.debug("Validating: " + record);

		if (!validateType(record)) {
			return false;
		}

		if (!validateSeqNum(record)) {
			return false;
		}

		if (!validateTotSeqNum(record)) {
			return false;
		}

		if (!validateTotStatement(record)) {
			return false;
		}

		return true;
	}

	/**
	 * Validate the type property of the given APPSPAFileRecord. Job will be
	 * stopped if invalid.
	 * 
	 * @param record
	 *            APPSPAFileRecord to validate.
	 * @return True if record is valid.
	 */
	private boolean validateType(APPSPAFileRecord record) {

		if (record.getType() != APPSPAFileRecord.DataType.PA) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Unknown record type at line " + lineNumber + ". Expected " + APPSPAFileRecord.DataType.PA + ".");
			return false;
		}

		return true;
	}

	/**
	 * Validate the seqNum property of the given APPSPAFileRecord. Job will be
	 * stopped if invalid.
	 * 
	 * @param record
	 *            APPSPAFileRecord to validate.
	 * @return boolean indicating if APPSPAFileRecord seqNum is valid.
	 */
	private boolean validateSeqNum(APPSPAFileRecord record) {

		if (record.getSeqNum() < 1) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Invalid SEQ-NUM at line " + lineNumber + ". SEQ-NUM=" + record.getSeqNum());
			return false;
		}

		if (previousRecord != null) {
			if (record.getSeqNum() != (previousRecord.getSeqNum() + 1)) {
				stopJob(PROCESSING_FAILURE_STATUS, "Invalid SEQ-NUM at line " + lineNumber + ". SEQ-NUM="
						+ record.getSeqNum() + "  Expected " + (previousRecord.getSeqNum() + 1) + ".");
				return false;
			}
		} else {
			if (record.getSeqNum() != 1) {
				stopJob(PROCESSING_FAILURE_STATUS,
						"Invalid SEQ-NUM at line " + lineNumber + ". SEQ-NUM=" + record.getSeqNum() + "  Expected 1.");
				return false;
			}
		}

		return true;
	}

	/**
	 * Validate the totStatement property of the given APPSPAFileRecord. Job
	 * will be stopped if invalid.
	 * 
	 * @param record
	 *            APPSPAFileRecord to validate.
	 * @return boolean indicating if APPSPAFileRecord totSeqNum is valid.
	 */
	private boolean validateTotSeqNum(APPSPAFileRecord record) {
		long totSeqNum;

		// TOT-SEQ-NUM is expected to be blank in the situation that the record
		// is not the last PA in a sequence.
		if (record.getTotSeqNum().equals("")) {
			return true;
		} else {
			try {
				totSeqNum = Long.parseLong(record.getTotSeqNum());
			} catch (NumberFormatException e) {
				stopJob(PROCESSING_FAILURE_STATUS,
						"Invalid TOT-SEQ-NUM at line " + lineNumber + ". TOT-SEQ-NUM=" + record.getTotSeqNum());
				return false;
			}
		}

		if (totSeqNum < 0) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Invalid TOT-SEQ-NUM at line " + lineNumber + ". TOT-SEQ-NUM=" + record.getTotSeqNum());
			return false;
		}

		if ((totSeqNum != 0) && (totSeqNum < record.getSeqNum())) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Invalid TOT-SEQ-NUM or SEQ-NUM at line " + lineNumber + ". TOT-SEQ-NUM=" + record.getTotSeqNum()
							+ ", SEQ-NUM=" + record.getSeqNum()
							+ ".  TOT-SEQ-NUM must be greater than or equal to SEQ-NUM.");
			return false;
		}

		return true;
	}

	/**
	 * Validate the totStatement property of the given APPSPAFileRecord. Job
	 * will be stopped if invalid.
	 * 
	 * @param record
	 *            APPSPAFileRecord to validate.
	 * @return boolean indicating if APPSPAFileRecord totStatement is valid.
	 */
	private boolean validateTotStatement(APPSPAFileRecord record) {

		if (record.getTotStatement() != record.getPrintAckDetailL().size()) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Number of AD records does not match TOT-STATEMENT at line " + lineNumber + ".  AD records found:"
							+ record.getPrintAckDetailL().size() + "  TOT-STATEMENT=" + record.getTotStatement());
			return false;
		}

		return true;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private void stopJob(final String status) {
		// Log message.
		logger.error("Processor execution encountered unrecoverable error and forced stop");
		// Set failure and message.
		setFailureStatus(status);
		// Stop job.
		jobExecution.stop();
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private void stopJob(final String status, final String message) {
		// Set failure.
		stopJob(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure and message in the job execution context.
	 */
	private void setFailureStatusAndMessage(final String status, final String message) {
		// Set job failure.
		setFailureStatus(status);
		// Set job failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		logger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		logger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
