package gov.va.cpss.job.apps;

import static gov.va.cpss.job.CbssJobProcessingConstants.EMPTY_FILE_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.INCOMPLETE_FILE_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.READ_FAILURE_STATUS;

import static gov.va.cpss.model.ps.Constants.ERROR_SITES_KEY;
import static gov.va.cpss.model.ps.Constants.FILE_COMPLETE_KEY;
import static gov.va.cpss.model.ps.Constants.INITIAL_SITE_COUNT;
import static gov.va.cpss.model.ps.Constants.TOTAL_SITE_COUNT_KEY;
import static gov.va.cpss.model.ps.Constants.TOTAL_PATIENT_COUNT_KEY;
import static gov.va.cpss.model.ps.Constants.LINE_FEED;
import static gov.va.cpss.model.ps.Constants.EMPTY_STRING;
import static gov.va.cpss.model.ps.Constants.DOUBLE_ZERO;
import static gov.va.cpss.model.ps.Constants.LONG_ZERO;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;

import java.math.BigDecimal;
import java.math.RoundingMode;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;

import org.springframework.batch.item.ExecutionContext;
import org.springframework.batch.item.ParseException;
import org.springframework.batch.item.UnexpectedInputException;

import gov.va.cpss.job.CbssSftpItemReader;
import gov.va.cpss.job.CbssStreamToQueueThread;

import gov.va.cpss.model.apps.APPSCompositeData;
import gov.va.cpss.model.apps.APSDetails;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSSite;
import gov.va.cpss.model.apps.APSRecord;
import gov.va.cpss.model.apps.APSReceivedSite;

import gov.va.cpss.model.ps.RecordType;

import gov.va.cpss.service.LoadAPPSService;

/**
 * 
 * Implementation of ItemReader used to handle reading of raw records when
 * loading and processing APPS data from FTP server.
 * 
 * Copyright DXC / VA
 * April 3, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 *
 * Updates and changes due to changes of requirements and validation rules.
 * 
 * Copyright HPE / VA
 * March 10, 2017
 * 
 * @author Yiping Yao
 * @version 2.0.0
 *
 * 
 * Rewrite, updates and changes due to significant changes of requirements
 * and validation rules.
 * 
 * Copyright DXE / VA
 * April 13, 2017
 * 
 * @author Yiping Yao
 * @version 3.0.0
 *
 */
@SuppressWarnings({"nls", "unchecked"})
public class APPSLineItemReader extends CbssSftpItemReader<APPSCompositeData>
{
    // Constants
    public final static String INVALID_DATA = "Invalid data";
    public final static String KEY_SEPARATOR = " - ";

    // Step Execution Context
    private ExecutionContext stepExecutionContext;

    // Job Execution Context
    private ExecutionContext jobExecutionContext;

    // Message Builder
    private StringBuilder sysMessages;

    // Site messages
    private List<String> siteMessages;

    // All sites messages
    private List<String> allSitesMessages;

    // Site line object
    private static final String SITE_LINE_OBJECT = "SiteLineObject";

    // Facility / Station / Site Number
    private String facilityNum;

    // Patient Number
    private String patientNumber;

    // Received Amount from PD
    private double receivedAmount = DOUBLE_ZERO;

    // Total Received Amount from PH
    private double totalReceivedAmount = DOUBLE_ZERO;

    // Flag for site successful or erred.
    private boolean isSiteSuccessful = true;

    // Flag for site valid or not.
    private boolean isSiteValid = true;

    // Flag for first patient in a site.
    private boolean isFirstPatient = true;

    // Counts for a single Site PS
    private long siteCount = INITIAL_SITE_COUNT;
    private long siteTotal = LONG_ZERO;
    private long siteReceivedCount = LONG_ZERO;
    private long previousSiteTotal = LONG_ZERO;
    private long siteCountOffset = LONG_ZERO;

    // Counts for all Sites in a file
    private long allSitesCount = INITIAL_SITE_COUNT;
    private long allSitesTotal = LONG_ZERO;

    // Counts for Patient PH
    private int patientCount = 0;
    private int patientTotal = 0;

    // Counts for Details PD
    private int detailsCount = 0;
    private int detailsTotal = 0;

    // Received Data Site
    private APSReceivedSite receivedSite;

    //
    // Injected properties
    //
    private LoadAPPSService loadAPPSService;

    public LoadAPPSService getLoadAPPSService()
    {
        return this.loadAPPSService;
    }

    public void setLoadAPPSService(LoadAPPSService inLoadAPPSService)
    {
        this.loadAPPSService = inLoadAPPSService;
    }

    private int rejectionPeriod;

    /**
     * @return the rejectionPeriod
     */
    public int getRejectionPeriod()
    {
        return this.rejectionPeriod;
    }

    /**
     * @param inRejectionPeriod the rejectionPeriod to set
     */
    public void setRejectionPeriod(int inRejectionPeriod)
    {
        this.rejectionPeriod = inRejectionPeriod;
    }


    @Override
    public void beforeStep(StepExecution inStepExecution)
    {
        readerLogger.debug("APPSLineItemReader2: Before Step Execution");

        super.beforeStep(inStepExecution);

        if (this.jobExecution == null)
        {
            this.jobExecution = inStepExecution.getJobExecution();
        }

        this.jobExecutionContext = this.jobExecution.getExecutionContext();
        this.stepExecutionContext = inStepExecution.getExecutionContext();

        // Initialize the error sites list
        this.jobExecutionContext.put(ERROR_SITES_KEY, new ArrayList<String>());

        // Initialize the total processing counts for all Sites and all Patients.
        this.jobExecutionContext.putLong(TOTAL_SITE_COUNT_KEY, LONG_ZERO);
        this.jobExecutionContext.putLong(TOTAL_PATIENT_COUNT_KEY, LONG_ZERO);

        // Initialize messages builder.
        this.sysMessages = new StringBuilder();

        // Initialize all sites messages.
        this.allSitesMessages = new ArrayList<>();
    }

    @Override
    public ExitStatus afterStep(StepExecution inStepExecution)
    {
        readerLogger.debug("APPSLineItemReader2: After Step Execution");

        // If no other error detected then check for other possible error conditions.
        if (!this.jobExecutionContext.containsKey(JOB_FAILURE_KEY))
        {
            // If read count is zero then report a job failure.
            if (inStepExecution.getReadCount() == 0)
            {
                setFailureStatusAndMessage(EMPTY_FILE_ERROR_STATUS, "Input file is empty.");
            }
            // NEW:
            // Since one Site data could be in multiple files,
            // we can no longer check if a Site is completed or not.
            //else if (!isLastDetailsCompleted() || !isLastPatientCompleted() || !isLastSiteCompleted())
            else if (!isLastDetailsCompleted() || !isLastPatientCompleted())
            {
                setFailureStatusAndMessage(INCOMPLETE_FILE_ERROR_STATUS, "File is not completed.");
            }

            if (!this.sysMessages.toString().isEmpty())
            {
                setFailureStatusAndMessage(READ_FAILURE_STATUS, this.sysMessages.toString());
            }

            // NEW:
            // Catching errors in multiple sites cases.
            List<String> erredSites = (List<String>) this.jobExecutionContext.get(ERROR_SITES_KEY);

            if (!erredSites.isEmpty())
            {
                StringBuilder errorSitesBuilder = new StringBuilder();

                errorSitesBuilder.append(LINE_FEED);
                errorSitesBuilder.append("Error Site(s): ");

                for (int i = 0; i < erredSites.size(); i++)
                {
                    if (i > 0)
                    {
                        errorSitesBuilder.append(", ");
                    }

                    errorSitesBuilder.append(erredSites.get(i));
                }

                errorSitesBuilder.append(".");
                errorSitesBuilder.append(LINE_FEED);

                String message = "."; 

                if (!this.allSitesMessages.isEmpty())
                {
                    message = ": " + this.allSitesMessages.get(0);
                }

                setFailureStatusAndMessage(READ_FAILURE_STATUS, errorSitesBuilder.toString() +
                                                                "Read error(s) encountered" + message);
            }
        }

        return super.afterStep(inStepExecution);
    }

    @Override
    protected CbssStreamToQueueThread getQueueBuilderThread(long expectedByteCount,
                                                            InputStream in,
                                                            BlockingQueue<String> outputQueue)
    {
        try
        {
            return new APPSStreamToQueueThread(expectedByteCount, in, outputQueue);
        }
        catch (UnsupportedEncodingException e)
        {
            String message = "Error creating data stream processing thread: " + e.getMessage();

            this.sysMessages.append(LINE_FEED);
            this.sysMessages.append(message);

            readerLogger.error(message);
        }

        return null;
    }

    @Override
    public APPSCompositeData read() throws Exception, UnexpectedInputException, ParseException
    {
        String message = EMPTY_STRING;

        // If there is already error, stop read.
        if (this.jobExecutionContext.containsKey(JOB_FAILURE_KEY))
        {
            message = "APPSLineItemReader2.read() - Job already erred before reading. - Not reading.";

            this.sysMessages.append(LINE_FEED);
            this.sysMessages.append(message);
            this.sysMessages.append(LINE_FEED);
            this.sysMessages.append(this.jobExecution.getExecutionContext().getString(JOB_FAILURE_MESSAGE_KEY));

            readerLogger.debug(message);

            return null;
        }

        readerLogger.debug("APPSLineItemReader2.read() - Starting Read ...");

        // Get the starting time
        final long startTime = System.nanoTime();

        // This will return a map entry of site that has a list of patients.
        APSSite site = null;
        List<APSPatient> patients = null;

        // Temporary variables for reading
        APSPatient patient = null;
        APSDetails details = null;

        // Read the line from Job Execution Context (previous read),
        // or from from the queue (initial read).
        // The returned line should return a mapped object built from the tokenizer configured in XML.
        Object line = this.stepExecutionContext.get(SITE_LINE_OBJECT);

        // If no previous read, then read from the queue.
        if (line == null)
        {
            line = super.read();

            // If read is still null, which means, either the end of file, then just stop and return null;
            // or there is fatal error, then log the error and stop.
            if (line == null)
            {
                if (this.jobExecutionContext.containsKey(JOB_FAILURE_KEY))
                {
                    // NEW:
                    // Log and process error messages before job stops.
                    // And send a data wrapper to processor so it can catch and log job failure message.
                    readerLogger.debug("APPSLineItemReader2.read() - Line Null: read error.");
                    return new APPSCompositeData();
                }

                readerLogger.debug("APPSLineItemReader2.read() - Line Null: end of file.");
                return null;
            }
        }

        // If read is unknown type, then just stop and return null.
        if ( !(line instanceof APSRecord<?>) )
        {
            message = "APPSLineItemReader2.read() - Line read is unknown type or error. - Stop reading.";

            this.sysMessages.append(LINE_FEED);
            this.sysMessages.append(message);

            setFailureStatusAndMessage(READ_FAILURE_STATUS, message);

            readerLogger.debug(message);
            return null;
        }

        //boolean isSuccessful = true;
        APPSCompositeData compositeData = new APPSCompositeData();

        // Will only read one line / one PS record (containing PH's and PD's as well)
        // at a time, which corresponds to one single Entry, that is, one Item.
        while (line instanceof APSRecord<?>)
        {
            message = EMPTY_STRING;

            APSRecord<RecordType> apsRecord = (APSRecord<RecordType>) line;

            // Read each type of data
            switch (apsRecord.getType())
            {
                // Site
                case PS:
                {
                    readerLogger.debug("   Read Site Q: |" + apsRecord + "|");

                    this.siteMessages = new ArrayList<>();

                    site = (APSSite) apsRecord;
                    this.isSiteSuccessful = validateSite(site) && this.isSiteSuccessful;

                    // NEW:
                    // Continue to process even there is error.
                    //if (isSuccessful)
                    {
                        patients = new ArrayList<>();

                        // Reset counts for PH, PD.
                        resetPatient();
                        resetDetails();
                    }

                    break;
                }
                // Patient
                case PH:
                {
                    //readerLogger.debug("   Read Patient Q: |" + apsRecord + "|");

                    // NEW:
                    // Continue to process even there is error.
                    //if (isSuccessful)
                    {
                        patient = (APSPatient) apsRecord;
                        this.isSiteSuccessful = validatePatient(patient) && this.isSiteSuccessful;

                        //if (isSuccessful && patients != null)
                        if (patients != null)
                        {
                            patients.add(patient);

                            // Reset counts for PD
                            resetDetails();
                        }
                    }

                    break;
                }
                // Details
                case PD:
                {
                    //readerLogger.debug("   Read Details Q: |" + apsRecord + "|");

                    details = (APSDetails) apsRecord;
                    this.isSiteSuccessful = validateDetails(details) && this.isSiteSuccessful;

                    // NEW:
                    // Continue to process even there is error.
                    //if (isSuccessful && patient != null)
                    if (patient != null)
                    {
                        patient.getDetailsList().add(details);
                    }

                    break;
                }
                case UNDEFINED:
                default:
                {
                    // Unrecoverable error or unknown type, so stop the job.
                    // NEW:
                    // Will log error and continue.
                    this.isSiteSuccessful = false;
                    message = "Unrecognized record type: " + apsRecord.getType() + ".";
                    this.siteMessages.add(message);
                    //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
                    break;
                }
            } // Switch

            // NEW:
            // Will continue whether it's erred or not.
            //if (isSuccessful)
            {
                line = super.read();

                //readerLogger.debug("APPSLineItemReader.read() - Read record at the end of while loop: " + line);

                // Check to see if the end of file or the end of line / PS record has reached,
                // if so, check to see if previous read has completed.
                if ( line == null || ((APSRecord<?>) line).getType() == RecordType.PS )
                {
                    // Check Patient and Details are completed for the Site.
                    this.isSiteSuccessful = isLastPatientCompleted() && this.isSiteSuccessful;
                    this.isSiteSuccessful = isLastDetailsCompleted() && this.isSiteSuccessful;

                    // NEW:
                    // Since one Site data could be in multiple files,
                    // we can no longer check if a Site is completed or not.
                    // And we need to set file completion to true.
                    if (line == null)
                    {
                    //    isSuccessful = isLastSiteCompleted() && isSuccessful;
                        this.isSiteSuccessful = processEndOfSite() && this.isSiteSuccessful;
                        this.jobExecutionContext.putString(FILE_COMPLETE_KEY, Boolean.TRUE.toString());
                    }

                    // Put already read site line object into Job Execution Context,
                    // so the next read can read and process it.
                    this.stepExecutionContext.put(SITE_LINE_OBJECT, line);

                    // NEW:
                    // Set composite data for either data or messages.
                    if (this.isSiteSuccessful)
                    {
                        // No error, save to data.
                        compositeData.setSiteEntry(new AbstractMap.SimpleEntry<>(site, patients));
                        compositeData.setValid(true);
                    }
                    else if (!this.siteMessages.isEmpty())
                    {
                        // Error, save to message.
                        compositeData.setMessageEntry(new AbstractMap.SimpleEntry<>(site, this.siteMessages));
                        compositeData.setValid(false);

                        this.allSitesMessages.addAll(this.siteMessages);

                        if ( !((List<String>) this.jobExecutionContext.get(ERROR_SITES_KEY)).contains(this.facilityNum) )
                        {
                            ((List<String>) this.jobExecutionContext.get(ERROR_SITES_KEY)).add(this.facilityNum);
                        }
                    }

                    break;
                } // if
            }
            // NEW:
            // No need to break or stop, if there is error.
            //else
            //{
                // If any error or finished reading, stop reading
            //    break;
            //} // if-else
        } // While

        final long elapsedTime = System.nanoTime() - startTime;

        readerLogger.debug("APPSLineItemReader2.read() - End Read. " + this.loadAPPSService.printElapsedTime("method", elapsedTime));

        return compositeData;
    }


    // Check to see if the site read has completed
    private boolean isLastSiteCompleted()
    {
        if ( (this.siteCount == 0 && this.siteTotal > 0) ||
             (this.siteCount > 0 && this.siteCount != this.siteTotal) )
        {
            // NEW:
            // Since one Site data could be in multiple files,
            // Incomplete Site is no longer an error.
            //String message = "Site: " + this.facilityNum + ". - PS count (" + this.siteCount + ") of (" + this.siteTotal + " or more) indicates incomplete site (PS002 / PS003).";
            //this.siteMessages.add(message);
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            return false;
        }

        return true;
    }

    // Check to see if the last / previous patient read has completed
    private boolean isLastPatientCompleted()
    {
        if ( (this.patientCount == 0 && this.patientTotal > 0) || 
             (this.patientCount > 0 && this.patientCount != this.patientTotal) )
        {
            String message = "Site: " + this.facilityNum + ". - PH count (" + this.patientCount + ") of (" + this.patientTotal + ") indicates incomplete patient (PS006).";
            this.siteMessages.add(message);
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            return false;
        }

        return true;
    }

    // Check to see if the last / previous details read has completed
    private boolean isLastDetailsCompleted()
    {
        if ( (this.detailsCount == 0 && this.detailsTotal > 0) ||
             (this.detailsCount > 0 && this.detailsCount != this.detailsTotal) )
        {
            String message = "Site: " + this.facilityNum + ". - Patient: " + this.patientNumber + ". - PD count (" + this.detailsCount + ") of (" + this.detailsTotal + ") indicates incomplete details (PH018).";
            this.siteMessages.add(message);
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            return false;
        }

        return true;
    }

    // Validate Site (PS) record
    private boolean validateSite(APSSite site)
    {
        boolean isValid = true;
        String message = EMPTY_STRING;

        // Count for all sites read in a file
        this.allSitesCount++;

        // NEW:
        // Since we may have multiple sites in one file,
        // we need to check if it's the same site or the site has changed or not.
        isValid = checkSiteChange(site);

        this.siteCount++;

        // NEW:
        // Will always take in the first SeqNum, since a single Site data
        // may be in multiple files, the starting SeqNum will not always
        // be one.
        if (this.siteCount == 1)
        {
            this.siteCountOffset = site.getSeqNum() - 1;
        }

        // Check if the record is valid.
        if (!site.isValid())
        {
            message = "Site: " + this.facilityNum + ". - " + INVALID_DATA + KEY_SEPARATOR + site;
            this.siteMessages.add(message);
            isValid = false;
        }

        // Check if SeqNum is out of sequence
        if ( (this.siteCount + this.siteCountOffset) != site.getSeqNum() )
        {
            message = "Site: " + this.facilityNum + ". - Invalid site found - seq number " + site.getSeqNum() + " is incorrect, expected " + (this.siteCount + this.siteCountOffset) + " (PS002).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        // Check if the TotSeqNum is the same for the same site in the same file.
        if (this.siteTotal > 0 && this.siteTotal != site.getTotSeqNum())
        {
            message = "Site: " + this.facilityNum + ". - Invalid site found - tot seq number " + site.getTotSeqNum() + " is incorrect, expected " + this.siteTotal + " (PS003).";
            this.siteMessages.add(message);
            isValid = false;
        }

        this.siteTotal = site.getTotSeqNum();

        // NEW:
        // Check if the TotSeqNum is the same for the same site from previously received same site.
        if (this.previousSiteTotal > 0 && this.siteTotal != this.previousSiteTotal)
        {
            message = "Site: " + this.facilityNum + ". - Invalid site found - tot seq number " + this.siteTotal + " is incorrect, expected " + this.previousSiteTotal + " from previously received Site data (PS003).";
            this.siteMessages.add(message);
            isValid = false;
        }

        // NEW:
        // For the site data lines: the total site is the total number of one site, and
        // with one site data in multiple file, it could be less than the site total,
        // so, only check larger-than case.
        if ( (this.siteCount + this.siteCountOffset) > this.siteTotal )
        {
            message = "Site: " + this.facilityNum + ". - Invalid site found - seq number " + (this.siteCount + this.siteCountOffset) + " is larger than the tot seq number " + this.siteTotal + " (PS002)." +
                                                                    " Or, tot seq number " + this.siteTotal + " is smaller than the last seq number " + (this.siteCount + this.siteCountOffset) + " (PS003).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        // NEW:
        // For the site data lines: the total site is the total number of one site,
        // and with one site data in multiple file, it still should be no 
        // larger than the site total.
        if ( (this.siteCount + this.siteReceivedCount) > this.siteTotal )
        {
            message = "Site: " + this.facilityNum + ". - Invalid site found - the number of messages " + (this.siteCount + this.siteReceivedCount) + " is larger than the tot seq number " + this.siteTotal + " (PS002).";
            this.siteMessages.add(message);
            isValid = false;
        }

        // For all data lines: the total all sites count is the total number of all sites.
        if (this.allSitesCount < this.allSitesTotal)
        {
            // Set the complete file flag to false in the context.
            // Files should always end up complete so if this flag is not set then the file was invalid.
            this.jobExecutionContext.putString(FILE_COMPLETE_KEY, Boolean.FALSE.toString());
        }
        // The very last data line: it is the total sites.
        else if (this.allSitesCount == this.allSitesTotal)
        {
            // Set the complete file flag to true in the context.
            // Files should always end up complete so if this flag is not set then the file was invalid.
            this.jobExecutionContext.putString(FILE_COMPLETE_KEY, Boolean.TRUE.toString());
        }
        // Otherwise, it is invalid.
        else
        {
            message = "Site: " + this.facilityNum + ". - Invalid file / site found - total all sites count " + this.allSitesTotal + " is not equal to the last count for all sites " + this.allSitesCount + ".";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        readerLogger.debug("Site: " + this.facilityNum + ". - Site Count: " + this.siteCount + ". Total site: " + this.siteTotal + ". Previously received site total: " + this.previousSiteTotal);


        // Check Facility Number
        if (!this.isSiteValid)
        {
            // The Facility Number is not valid.
            message = "Site: " + this.facilityNum + ". - PS facility number was invalid (not in Station Info list): " + site.getFacilityNum() + " (PS004).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        // Check Patient count
        this.patientTotal = Integer.parseInt(site.getTotalPatientStr());

        if (this.patientTotal <= 0)
        {
            // Expected patient count is invalid.
            message = "Site: " + this.facilityNum + ". - PS patient count was invalid: " + this.patientTotal + " (PS006).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        // Check Statement Year
        if (!this.loadAPPSService.isPreviousYear(site.getStatementDate()))
        {
            // The Site is not for the previous year.
            message = "Site: " + this.facilityNum + ". - PS statement year was invalid (not previous year): " + site.getStatementDate() + " (PS007).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }


        //
        // Add total sites and total patients (for all sites).
        //

        // Add all sites total
        //long previousSiteTotal = this.jobExecutionContext.getLong(TOTAL_SITE_COUNT_KEY);
        //this.jobExecutionContext.putLong(TOTAL_SITE_COUNT_KEY, previousSiteTotal + 1);
        this.jobExecutionContext.putLong(TOTAL_SITE_COUNT_KEY, this.allSitesCount);

        // Add all patients total
        long previousPatientTotal = this.jobExecutionContext.getLong(TOTAL_PATIENT_COUNT_KEY);

        this.jobExecutionContext.putLong(TOTAL_PATIENT_COUNT_KEY, previousPatientTotal + this.patientTotal);

        readerLogger.debug("Total sites: " + this.jobExecutionContext.getLong(TOTAL_SITE_COUNT_KEY) + " | " +
                           "Total patients: " + this.jobExecutionContext.getLong(TOTAL_PATIENT_COUNT_KEY));

        return isValid;
    }

    // Validate Patient (PH) record
    private boolean validatePatient(APSPatient patient)
    {
        boolean isValid = true;
        String message = EMPTY_STRING;

        this.patientCount++;

        //readerLogger.debug("Patient count: " + this.patientCount + ". Total patient: " + this.totalPatient);

        // Check if patient has changed.
        isValid = checkPatientChange(patient);

        // Check if the record is valid.
        if (!patient.isValid())
        {
            message = "Site: " + this.facilityNum + ". - " + INVALID_DATA + KEY_SEPARATOR + patient;
            this.siteMessages.add(message);
            isValid = false;
        }

        this.patientNumber = patient.getAccountNumber();

        if (this.patientCount > this.patientTotal)
        {
            // Invalid patient count.
            message = "Site: " + this.facilityNum + ". - PH count is too large or invalid: " + this.patientCount + ", expected: " + this.patientTotal + " (PS006).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        this.detailsTotal = Integer.parseInt(patient.getNumOfPDStr());

        //readerLogger.debug("Total details: " + this.totalDetails);

        if (this.detailsTotal <= 0)
        {
            // Expected details count is invalid.
            message = "Site: " + this.facilityNum + ". - Patient: " + this.patientNumber + ". - PH total details in patient is invalid: " + this.detailsTotal + " (PH018).";
            this.siteMessages.add(message);
            isValid = false;
            //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
            //return false;
        }

        return isValid;
    }

    // Validate Details (PD) record
    private boolean validateDetails(APSDetails details)
    {
        boolean isValid = true;
        String message = EMPTY_STRING;

        this.detailsCount++;

        //readerLogger.debug("Details count: " + this.detailsCount + ". Total details: " + this.totalDetails);

        // Check if the record is valid.
        if (!details.isValid())
        {
            message = "Site: " + this.facilityNum + ". - Patient: " + this.patientNumber + ". - " + INVALID_DATA + KEY_SEPARATOR + details;
            this.siteMessages.add(message);
            isValid = false;
        }

       if (this.detailsCount > this.detailsTotal)
       {
           // Invalid details count.
           message = "Site: " + this.facilityNum + ". - Patient: " + this.patientNumber + ". - PD count is too large or invalid: " + this.detailsCount + ", expected: " + this.detailsTotal + " (PH018).";
           this.siteMessages.add(message);
           isValid = false;
           //setFailureStatusAndMessage(READ_FAILURE_STATUS, message);
           //return false;
       }

       // Transaction amount is required, but no need to log error here,
       // since it will be caught in the details validation check. 
       if (details.getTransactionAmount() != null)
       {
           this.receivedAmount += details.getTransactionAmount().getDouble();
       }

       return isValid;
    }

    // NEW:
    // Check if site has changed, since a single file may contain multiple sites.
    // Return if there is any error or not - valid data.
    private boolean checkSiteChange(APSSite site)
    {
        boolean isSuccessful = true;

        // NEW:
        // For new or changed site,
        // - Record and process previous Site information, if any;
        // - Retrieve previous site-information for current Site, if any;
        // - Save current Site number in the step execution context for later use.
        if (this.facilityNum == null || this.facilityNum.isEmpty() ||
            !this.facilityNum.equalsIgnoreCase(site.getFacilityNum().trim()))
        {
            // For previous Site
            isSuccessful = processEndOfSite();

            this.facilityNum = site.getFacilityNum().trim();

            try
            {
                // Get received site for current Site.
                this.receivedSite = this.loadAPPSService.getReceivedSite(this.facilityNum);
            }
            catch (Exception ex)
            {
                this.siteMessages.add(ex.getMessage());
                isSuccessful = false;
            }

            // Check if the site is valid or not.
            this.isSiteValid = this.loadAPPSService.isValidFacility(this.facilityNum);

            // Add this site total count to the all sites total count
            this.allSitesTotal += site.getTotSeqNum();

            // Reset Site
            resetSite();
        }

        return isSuccessful;
    }

    // NEW:
    // Check if patient has changed, since we need the patient to log error,
    // and also check its associated details.
    // Return if there is any error or not - valid data.
    private boolean checkPatientChange(APSPatient patient)
    {
        boolean isSuccessful = true;

        // For new or changed patient, save patient account number in the step execution context for later use.
        if (this.patientNumber == null || this.patientNumber.isEmpty() ||
            !this.patientNumber.equalsIgnoreCase(patient.getAccountNumber().trim()))
        {
            // Check last / previous reads completed or not.
            // Not checking only for the initial site read, since it was already checked.
            if (!this.isFirstPatient)
            {
                if (isLastDetailsCompleted())
                {
                    double amount = BigDecimal.valueOf(this.receivedAmount).setScale(2, RoundingMode.HALF_UP).doubleValue();

                    if (amount != this.totalReceivedAmount)
                    {
                        String message = "Site: " + this.facilityNum + ". - Patient: " + this.patientNumber + ". - PH total received amount in patient is invalid. Should be: " + amount + " (PH013).";
                        this.siteMessages.add(message);
                        isSuccessful = false;
                    }
                }
                else
                {
                    isSuccessful = false;
                }
            }
            else
            {
                this.isFirstPatient = false;
            }

            this.patientNumber = patient.getAccountNumber().trim();

            // Total amount received is required, but no need to log error here,
            // since it will be caught in patient validation check. 
            if (patient.getTotalAmountReceived() != null)
            {
                this.totalReceivedAmount = patient.getTotalAmountReceived().getDouble();
            }
            else
            {
                this.totalReceivedAmount = DOUBLE_ZERO;
            }
        }

        return isSuccessful;
    }

    // NEW:
    // Reset PS Site
    private void resetSite()
    {
        // Reset site processing counts, and
        // Reset site error flag.
        if (this.receivedSite != null)
        {
            if (this.loadAPPSService.isErrorStatus(this.receivedSite.getStatusId()))
            {
                // Check if it is inside of error site rejection period or not.
                if (this.loadAPPSService.isSiteStillErred(this.receivedSite, this.rejectionPeriod))
                {
                    this.siteReceivedCount = this.receivedSite.getNumOfMessages();
                    this.previousSiteTotal = this.receivedSite.getTotMsgNum();
                    this.isSiteSuccessful = false; 
                }
                else
                {
                    // Outside of rejection period, reset to start from fresh.
                    this.siteReceivedCount = LONG_ZERO;
                    this.previousSiteTotal = LONG_ZERO;
                    this.isSiteSuccessful = true;

                    this.loadAPPSService.deleteReceivedSite(this.receivedSite);
                    this.receivedSite = null;
                }
            }
            else
            {
                this.siteReceivedCount = this.receivedSite.getNumOfMessages();
                this.previousSiteTotal = this.receivedSite.getTotMsgNum();
                this.isSiteSuccessful = true; 
            }
        }
        else
        {
            this.siteReceivedCount = LONG_ZERO;
            this.previousSiteTotal = LONG_ZERO;
            this.isSiteSuccessful = true;
        }

        this.siteCount = INITIAL_SITE_COUNT;
        this.siteTotal = LONG_ZERO;
    }

    // Reset PH Patient
    private void resetPatient()
    {
        this.patientCount = 0;
        this.isFirstPatient = true;
    }

    // Reset PD Details
    private void resetDetails()
    {
        this.detailsCount = 0;
        this.receivedAmount = DOUBLE_ZERO;
    }

    // NEW:
    // Process the end of site
    private boolean processEndOfSite()
    {
        boolean isSuccessful = true;

        try
        {
            // If there is a previous received Site, update it.
            if (this.receivedSite != null)
            {
                // Add the number of messages received so far.
                this.siteCount += this.receivedSite.getNumOfMessages();
                this.receivedSite.setNumOfMessages(this.siteCount);

                this.loadAPPSService.updateReceivedSite(this.receivedSite, this.isSiteSuccessful);
            }

            // If not first read or initial Site,
            // Get previously received Site.
            if (this.facilityNum != null && !this.facilityNum.isEmpty())
            {
                // No previous received Site and current Site is completed.
                if (this.receivedSite == null)
                {
                    // If the Site is completed and successful, no need to insert a new
                    // APSReceivedSite, otherwise, insert a new row for later use.
                    if (!this.isSiteSuccessful || !isLastSiteCompleted())
                    {
                        this.receivedSite = new APSReceivedSite();
                        this.receivedSite.setFacilityNum(this.facilityNum);
                        this.receivedSite.setNumOfMessages(this.siteCount);
                        this.receivedSite.setTotMsgNum(this.siteTotal);
                        this.receivedSite.setReceivedDate(this.loadAPPSService.getCurrentDate());

                        this.loadAPPSService.saveReceivedSite(this.receivedSite, this.isSiteSuccessful);
                    }
                }
                else
                {
                    // If the Site is completed and successful, delete the existing
                    // APSReceivedSite, since there is no need for it.
                    if (this.isSiteSuccessful && isLastSiteCompleted())
                    {
                        this.loadAPPSService.deleteReceivedSite(this.receivedSite);
                    }
                }
            } // if-else
        } // try
        catch (Exception ex)
        {
            this.siteMessages.add(ex.getMessage());
            isSuccessful = false;
        }

        return isSuccessful;
    }
}
