package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.DATE_RECEIVED_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.INPUT_DIRECTORY_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.MISSING_PATIENT_ACCT_LIST_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.PRINT_ACK_REC_ID_LIST_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.TOTAL_CBS_RECEIVED_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.TOTAL_CBS_UPDATED_KEY;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.printack.ADFileRecord;

public class UpdatePrintAckJob extends AbstractProcessFileListJob {

	@Autowired
	private FlowJob UpdatePrintAckJobBatch;

	private long totalCBSReceived;

	private long totalCBSUpdated;

	private List<Long> printAckRecIdList;

	private List<ADFileRecord> missingPatientAcctList;

	@Autowired
	private String updatePrintAckServerTargetDirectory;

	@Autowired
	private String updatePrintAckServerErrorTargetDirectory;

	@Autowired
	private String updatePrintAckServerArchiveTargetDirectory;

	public long getTotalCBSReceived() {
		return totalCBSReceived;
	}

	public long getTotalCBSUpdated() {
		return totalCBSUpdated;
	}

	public List<Long> getPrintAckRecIdList() {
		return printAckRecIdList;
	}

	public List<ADFileRecord> getMissingPatientAcctList() {
		return missingPatientAcctList;
	}

	@Override
	protected FlowJob getJob() {
		return UpdatePrintAckJobBatch;
	}

	@Override
	protected boolean runJob(BatchRun bR, FlowJob job) {
		totalCBSReceived = 0L;
		totalCBSUpdated = 0L;

		printAckRecIdList = new ArrayList<Long>();

		missingPatientAcctList = new ArrayList<ADFileRecord>();

		final boolean success = super.runJob(bR, job);

		captureStatistics();

		return success;
	}

	@Override
	protected boolean processFile(final String file, final int batchRunId, final FlowJob job) {
		boolean success = true;

		Timestamp dateReceived = new Timestamp(Calendar.getInstance().getTime().getTime());

		JobParameters parameters = getParameters(batchRunId, dateReceived);

		JobExecution execution = null;
		try {
			execution = executeJob(job, parameters);

			if (execution == null) {
				success = false;
				final String errorMsg = "Processing ended with null execution";
				logExecuteJobException(errorMsg);
				appendErrorMessage(errorMsg);
			} else if (execution.getExitStatus() != ExitStatus.COMPLETED) {
				success = false;

				String errorMsg;
				if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
					errorMsg = "No acknowledgements will be recorded for this file as processing ended with failure: "
							+ execution.getExecutionContext().getString(JOB_FAILURE_KEY);
				} else {
					errorMsg = "No acknowledgements will be recorded for this file as processing ended with unknown error: "
							+ execution.getExitStatus();
				}
				appendErrorMessage(errorMsg);
			}

		} catch (Exception e) {
			appendErrorMessage(e.getMessage());
			success = false;
		}

		// Capture PrintAckRecIds and MissingPatientAccts regardless of success or failure
		capturePrintAckRecIds(execution);
		captureMissingPatientAccts(execution);
		if (success) {
			// Accrue processing counts for this file, if successful.
			totalCBSReceived += execution.getExecutionContext().getLong(TOTAL_CBS_RECEIVED_KEY);
			totalCBSUpdated += execution.getExecutionContext().getLong(TOTAL_CBS_UPDATED_KEY);
		}

		return success;
	}

	/**
	 * Build parameters for the batch job based.
	 * 
	 * @param batchRunId
	 *            Id of the BatchRun record associated with this job.
	 * @param dateReceived
	 *            Date the file being processed was received.
	 * @return JobParameters object for this batch job run.
	 */
	private JobParameters getParameters(final int batchRunId, final Timestamp dateReceived) {

		// Build job parameters for the input filename, batchRunId and
		// receivedDate
		return new JobParametersBuilder().addString(INPUT_RESOURCE_KEY, filename)
				.addString(INPUT_DIRECTORY_KEY, getDataDirectory()).addLong(BATCH_RUN_ID_KEY, new Long(batchRunId))
				.addLong(DATE_RECEIVED_KEY, dateReceived.getTime()).toJobParameters();
	}

	/**
	 * Gets the list of PrintAckRec ids that have been created for the last
	 * processed file and accrues them in printAckRecIdList.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 */
	private void capturePrintAckRecIds(JobExecution jobExecution) {
		List<Long> currentFilePrintAckRecIdList = UpdatePrintAckJob.getCurrentFilePrintAckRecIdList(jobExecution);
		printAckRecIdList.addAll(currentFilePrintAckRecIdList);
	}

	/**
	 * Gets the missing patient accounts created for the last processed file and
	 * accrues them in missingPatientAcctList.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 */
	private void captureMissingPatientAccts(JobExecution jobExecution) {
		List<ADFileRecord> currentFileMissingPatientAcctList = UpdatePrintAckJob.getCurrentFileMissingPatientAcctList(jobExecution);
		jobLogger.debug("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXcurrentFileMissingPatientAcctList.size():" + currentFileMissingPatientAcctList.size());
		missingPatientAcctList.addAll(currentFileMissingPatientAcctList);
	}

	/**
	 * Gets the List of PrintAckRec ids that have been created by the job for
	 * the current file being processed from the job execution context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @return List<Long> of PrintAckRec ids created by the job for the current
	 *         file.
	 */
	public static List<Long> getCurrentFilePrintAckRecIdList(JobExecution jobExecution) {
		@SuppressWarnings("unchecked")
		List<Long> currentFilePrintAckRecIdList = (List<Long>) jobExecution.getExecutionContext()
				.get(PRINT_ACK_REC_ID_LIST_KEY);
		if (currentFilePrintAckRecIdList == null) {
			currentFilePrintAckRecIdList = new ArrayList<Long>();
			UpdatePrintAckJob.putCurrentFilePrintAckRecIdList(jobExecution, currentFilePrintAckRecIdList);
		}

		return currentFilePrintAckRecIdList;
	}

	/**
	 * Puts the PrintAckRec ids that have been created by the job for the
	 * current file being processed in the job execution context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @param currentFilePrintAckRecIdList
	 *            List<Long> of PrintAckRec ids created by the job for the
	 *            current file.
	 */
	public static void putCurrentFilePrintAckRecIdList(JobExecution jobExecution,
			List<Long> currentFilePrintAckRecIdList) {
		jobExecution.getExecutionContext().put(PRINT_ACK_REC_ID_LIST_KEY, currentFilePrintAckRecIdList);
	}

	/**
	 * Gets the List of ADFileRecord objects representing patient accounts for
	 * the current file that are missing statements from the job execution
	 * context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @return List<ADFileRecord> of missing patient accounts for the current
	 *         file.
	 */
	public static List<ADFileRecord> getCurrentFileMissingPatientAcctList(JobExecution jobExecution) {
		@SuppressWarnings("unchecked")
		List<ADFileRecord> missingPatientAccts = (List<ADFileRecord>) jobExecution.getExecutionContext()
				.get(MISSING_PATIENT_ACCT_LIST_KEY);
		if (missingPatientAccts == null) {
			missingPatientAccts = new ArrayList<ADFileRecord>();
			UpdatePrintAckJob.putCurrentFileMissingPatientAcctList(jobExecution, missingPatientAccts);
		}

		return missingPatientAccts;
	}

	/**
	 * Puts the List of ADFileRecord objects representing patient accounts for
	 * the current file that are missing statements in the job execution
	 * context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @param missingPatientAccts
	 *            List<Long> of PrintAckRec ids created by the job for the
	 *            current file.
	 */
	public static void putCurrentFileMissingPatientAcctList(JobExecution jobExecution,
			List<ADFileRecord> missingPatientAccts) {
		jobExecution.getExecutionContext().put(MISSING_PATIENT_ACCT_LIST_KEY, missingPatientAccts);
	}

	/**
	 * Update the info message for the batch job with the processing statistics.
	 */
	private void captureStatistics() {
		StringBuffer strBuff = new StringBuffer();

		strBuff.append("\nTotal CBS Received: ").append(totalCBSReceived);
		strBuff.append("\nTotal CBS Updated: ").append(totalCBSUpdated);

		appendInfoMessage(strBuff.toString());
	}

	// Getters Below are used for Setting the Directory Tree for CBSS to modify
	// the directory tree see DirectoryTree.prop file
	@Override
	public String getArchiveSubDirectory() {
		return updatePrintAckServerArchiveTargetDirectory;
	}

	@Override
	public String getErrorSubDirectory() {
		return updatePrintAckServerErrorTargetDirectory;
	}

	@Override
	public String getDataDirectory() {
		return updatePrintAckServerTargetDirectory;
	}

	@Override
	public String getProcessListErrorMessage() {
		return "Unable to connect to the SFTP server to retrieve the print acknowledgement files.\n";
	}

}