package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.BATCH_RUN_PROCESS_ID_LIST_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.DATE_RECEIVED_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.INPUT_DIRECTORY_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.MISSING_PATIENT_ACCT_LIST_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.TOTAL_APPS_RECEIVED_KEY;
import static gov.va.cpss.job.appsprintack.AppsUpaProcessingConstants.TOTAL_APPS_UPDATED_KEY;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;

//TODO: javadoc
/**
 * 
 * @author Andrew Vance
 */
public class UpdateAPPSPrintAckJob extends AbstractProcessFileListJob {
	
	@Autowired
	private FlowJob UpdateAppsPrintAckJobBatch;

	private long totalAPSReceived;

	private long totalAPSUpdated;

	private List<Long> batchRunProcessIdList;

	private List<APPSADFileRecord> missingPatientAcctList;

	@Autowired
	private String updatePrintAckAPPSServerTargetDirectory;

	@Autowired
	private String updatePrintAckAPPSServerErrorTargetDirectory;

	@Autowired
	private String updatePrintAckAPPSServerArchiveTargetDirectory;

	public long getTotalAPSReceived() {
		return totalAPSReceived;
	}

	public long getTotalAPSUpdated() {
		return totalAPSUpdated;
	}

	public List<Long> getBatchRunProcessIdList() {
		return batchRunProcessIdList;
	}

	public List<APPSADFileRecord> getMissingPatientAcctList() {
		return missingPatientAcctList;
	}

	@Override
	protected FlowJob getJob() {
		return UpdateAppsPrintAckJobBatch;
	}

	@Override
	protected boolean runJob(BatchRun bR, FlowJob job) {
		totalAPSReceived = 0L;
		totalAPSUpdated = 0L;

		batchRunProcessIdList = new ArrayList<Long>();

		missingPatientAcctList = new ArrayList<APPSADFileRecord>();

		final boolean success = super.runJob(bR, job);

		captureStatistics();

		return success;
	}

	@Override
	protected boolean processFile(final String file, final int batchRunId, final FlowJob job) {
		boolean success = true;

		Timestamp dateReceived = new Timestamp(Calendar.getInstance().getTime().getTime());

		JobParameters parameters = getParameters(batchRunId, dateReceived);

		JobExecution execution = null;
		try {
			execution = executeJob(job, parameters);

			if (execution == null) {
				success = false;
				final String errorMsg = "Processing ended with null execution";
				logExecuteJobException(errorMsg);
				appendErrorMessage(errorMsg);
			} else if (execution.getExitStatus() != ExitStatus.COMPLETED) {
				success = false;

				String errorMsg;
				if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
					errorMsg = "No acknowledgements will be recorded for this file as processing ended with failure: "
							+ execution.getExecutionContext().getString(JOB_FAILURE_KEY);
				} else {
					errorMsg = "No acknowledgements will be recorded for this file as processing ended with unknown error: "
							+ execution.getExitStatus();
				}
				appendErrorMessage(errorMsg);
			}

		} catch (Exception e) {
			appendErrorMessage(e.getMessage());
			success = false;
		}

		// Capture PrintAckRecIds and MissingPatientAccts regardless of success or failure
		captureAppsPrintAckRecIds(execution);
		captureMissingPatientAccts(execution);
		if (success) {
			// Accrue processing counts for this file, if successful.
			totalAPSReceived += execution.getExecutionContext().getLong(TOTAL_APPS_RECEIVED_KEY);
			totalAPSUpdated += execution.getExecutionContext().getLong(TOTAL_APPS_UPDATED_KEY);
		}

		return success;
	}

	/**
	 * Build parameters for the batch job based.
	 * 
	 * @param batchRunId
	 *            Id of the BatchRun record associated with this job.
	 * @param dateReceived
	 *            Date the file being processed was received.
	 * @return JobParameters object for this batch job run.
	 */
	private JobParameters getParameters(final int batchRunId, final Timestamp dateReceived) {

		// Build job parameters for the input filename, batchRunId and
		// receivedDate
		return new JobParametersBuilder().addString(INPUT_RESOURCE_KEY, filename)
				.addString(INPUT_DIRECTORY_KEY, getDataDirectory()).addLong(BATCH_RUN_ID_KEY, new Long(batchRunId))
				.addLong(DATE_RECEIVED_KEY, dateReceived.getTime()).toJobParameters();
	}

	/**
	 * Gets the list of PrintAckRec ids that have been created for the last
	 * processed file and accrues them in printAckRecIdList.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 */
	private void captureAppsPrintAckRecIds(JobExecution jobExecution) {
		List<Long> currentFileAppsPrintAckRecIdList = UpdateAPPSPrintAckJob.getCurrentFileAppsBatchRunProcessIdList(jobExecution);
		batchRunProcessIdList.addAll(currentFileAppsPrintAckRecIdList);
	}

	/**
	 * Gets the missing patient accounts created for the last processed file and
	 * accrues them in missingPatientAcctList.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 */
	private void captureMissingPatientAccts(JobExecution jobExecution) {
		List<APPSADFileRecord> currentFileMissingPatientAcctList = UpdateAPPSPrintAckJob.getCurrentFileMissingPatientAcctList(jobExecution);
		missingPatientAcctList.addAll(currentFileMissingPatientAcctList);
	}

	/**
	 * Gets the List of PrintAckRec ids that have been created by the job for
	 * the current file being processed from the job execution context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @return List<Long> of PrintAckRec ids created by the job for the current
	 *         file.
	 */
	public static List<Long> getCurrentFileAppsBatchRunProcessIdList(JobExecution jobExecution) {
		@SuppressWarnings("unchecked")
		List<Long> currentFileAppsPrintAckRecIdList = (List<Long>) jobExecution.getExecutionContext()
				.get(BATCH_RUN_PROCESS_ID_LIST_KEY);
		if (currentFileAppsPrintAckRecIdList == null) {
			currentFileAppsPrintAckRecIdList = new ArrayList<Long>();
			UpdatePrintAckJob.putCurrentFilePrintAckRecIdList(jobExecution, currentFileAppsPrintAckRecIdList);
		}

		return currentFileAppsPrintAckRecIdList;
	}

	/**
	 * Puts the PrintAckRec ids that have been created by the job for the
	 * current file being processed in the job execution context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @param currentFilePrintAckRecIdList
	 *            List<Long> of PrintAckRec ids created by the job for the
	 *            current file.
	 */
	public static void putCurrentFileAppsBatchRunProcessIdList(JobExecution jobExecution,
			List<Long> currentFileAppsBatchRunProcessIdList) {
		jobExecution.getExecutionContext().put(BATCH_RUN_PROCESS_ID_LIST_KEY, currentFileAppsBatchRunProcessIdList);
	}

	/**
	 * Gets the List of ADFileRecord objects representing patient accounts for
	 * the current file that are missing statements from the job execution
	 * context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @return List<ADFileRecord> of missing patient accounts for the current
	 *         file.
	 */
	public static List<APPSADFileRecord> getCurrentFileMissingPatientAcctList(JobExecution jobExecution) {
		@SuppressWarnings("unchecked")
		List<APPSADFileRecord> missingPatientAccts = (List<APPSADFileRecord>) jobExecution.getExecutionContext()
				.get(MISSING_PATIENT_ACCT_LIST_KEY);
		if (missingPatientAccts == null) {
			missingPatientAccts = new ArrayList<APPSADFileRecord>();
			UpdateAPPSPrintAckJob.putCurrentFileMissingPatientAcctList(jobExecution, missingPatientAccts);
		}

		return missingPatientAccts;
	}

	/**
	 * Puts the List of ADFileRecord objects representing patient accounts for
	 * the current file that are missing statements in the job execution
	 * context.
	 * 
	 * @param jobExecution
	 *            JobExecution
	 * @param missingPatientAccts
	 *            List<Long> of PrintAckRec ids created by the job for the
	 *            current file.
	 */
	public static void putCurrentFileMissingPatientAcctList(JobExecution jobExecution,
			List<APPSADFileRecord> missingPatientAccts) {
		jobExecution.getExecutionContext().put(MISSING_PATIENT_ACCT_LIST_KEY, missingPatientAccts);
	}

	/**
	 * Update the info message for the batch job with the processing statistics.
	 */
	private void captureStatistics() {
		StringBuffer strBuff = new StringBuffer();

		strBuff.append("\nTotal APPS Received: ").append(totalAPSReceived);
		strBuff.append("\nTotal APPS Updated: ").append(totalAPSUpdated);

		appendInfoMessage(strBuff.toString());
	}

	// Getters Below are used for Setting the Directory Tree for APPSS to modify
	// the directory tree see DirectoryTree.prop file
	@Override
	public String getArchiveSubDirectory() {
		return updatePrintAckAPPSServerArchiveTargetDirectory;
	}

	@Override
	public String getErrorSubDirectory() {
		return updatePrintAckAPPSServerErrorTargetDirectory;
	}

	@Override
	public String getDataDirectory() {
		return updatePrintAckAPPSServerTargetDirectory;
	}

	@Override
	public String getProcessListErrorMessage() {
		// TODO Auto-generated method stub
		return null;
	}

}